// App controller.
(function ($) {
    $.widget('adsk.appcontroller', {
        options: {
            config: {
                productFullName: '',

                ui: {
                    tabs: {
                        order: [
                            'C','I','S','F'
                        ],
                        tocEnabled: true,
                        indexEnabled: true,
                        favoritesEnabled: true,
                        searchEnabled: true
                    },

                    tools: {
                        order: ['home', 'previous', 'parent', 'following', 'separator',
                                'favorites', 'share', 'separator', 'print'],
                        home: true,
                        previous: true,
                        parent: true,
                        following: true,
                        favorites: true,
                        share: true,
                        print: true
                    },

                    responsive: true
                },

                features: {
                    tabsbarParent: 'banner',
                    banner: true,
                    header: true,
                    toolbar: true,
                    breadcrumbs: true,
                    tabsbar: true,
                    navigation: true,
                    splitter: true
                }
            },
            targetId: 'browser',
            viewControllers: {},
            toc: {},
            defaultTopicData: undefined,
            urlParams: undefined
        },

        /* Private members. */

        _view: undefined,
        _defTopicReady: new $.Deferred(),
        _defTocReady: new $.Deferred(),
        _defContextIdsReady: new $.Deferred(),

        _create: function () {
            console.log('appcontroller: _create');

            var self = this;

            this._setupPolyfills();

            $.when(this._defTopicReady, this._defTocReady, this._defContextIdsReady).done(function() {
                self._addHashchangeHandler();

                // Go to default topic page if there was no hash data specified by the user.
                if (location.hash == '') {
                    var href = self.options.defaultTopicData.attr.href;

                    if (self.options.urlParams !== undefined &&
                        self.options.urlParams.length > 0) {

                        var p = self.options.urlParams;

                        // Process contextId query.
                        if (p['contextId'] !== undefined &&
                            p['contextId'] !== '' ) {

                                // Open specified topic.
                                var id = decodeURIComponent(p['contextId']).toUpperCase(),
                                    contextIdHref = self.options.config.contextIds[id];

                                if (contextIdHref !== undefined && contextIdHref != '') {
                                    href = './'.concat(contextIdHref);
                                }
                        }
                    }

                    // Go to topic page.
                    self._handleOpentopic({}, { href: href });
                } else {
                    $(window).trigger('hashchange');
                }
            });

            this._checkUrlParameters();

            this._createView();

            this._createControllers();

            // Hide main preloader.
            this._view.hidePreloader();

            // Load context id data.
            this._loadContextIdData();

            // Load and process TOC data.
            this._loadAndProcessTOC();
        },

        _setupPolyfills: function() {
            $('html').addClass($.fn.details.support ? 'details' : 'no-details');
        },

        _checkUrlParameters: function() {
            console.log('appcontroller: _checkUrlParameters');

            this.options.urlParams = $.getUrlVars();

            if (this.options.urlParams !== undefined && this.options.urlParams.length > 0) {
                var p = this.options.urlParams,
                    features = this.options.config.features,
                    tabs = this.options.config.ui.tabs;

                if (p['panels'] !== undefined && p['panels'] == 'no') {
                    // Turn off all UI features.
                    features.banner = false;
                    features.navigation = false;
                    features.splitter = false;
                    features.breadcrumbs = false;

                    // Turn off all navigation tabs too.
                    tabs.tocEnabled = false;
                    tabs.indexEnabled = false;
                    tabs.favoritesEnabled = false;
                    tabs.searchEnabled = false;
                }
            }
        },

        _createView: function () {
            console.log('appcontroller: _createView');

            var self = this;

            $('#' + this.options.targetId).appview({
                title: this.options.config.productFullName,

                features: this.options.config.features,

                ui: this.options.config.ui,

                labels: this.options.config.labels,

                create: function (event) {
                    self._view = $(this).appview('instance');
                },

                buttonclick: function (event, id) {
                    id = id.substring('main-'.length);
                    var handlerName = '_' + id + 'ButtonclickHandler';
                    self[handlerName](event);
                },

                tabbuttonclick: function (event, id) {
                    self._view.showTab(id.substring('tab-button-'.length));
                }
            });
        },

        _createControllers: function () {
            console.log('appcontroller: _createControllers');

            var config = this.options.config,
                controllers = this.options.viewControllers,
                self = this;

            self._createTocController(self, config, controllers);
            self._createFavoritesDataController(config, controllers);
            self._createTopicController(self, config, controllers);
            self._createIndexController(self, config, controllers);
            self._createFavoritesController(self, config, controllers);
            self._createSearchController(self, config, controllers);
        },

        _createTocController: function(self, config, controllers) {
            // Create TOC view controller.
            if (config.ui.tabs.tocEnabled) {
                controllers.tocController = $.adsk.toccontroller({
                    config: config,

                    opentopic: function (event, data) {
                        self._handleOpentopic(event, data);
                    }
                });
            }
        },

        _createFavoritesDataController: function(config, controllers) {
            // Create favorites controller.
            if (config.ui.tabs.favoritesEnabled) {
                controllers.favoritesController = $.adsk.favoritescontroller();
            }
        },

        _createTopicController: function(self, config, controllers) {
            // Create topic view controller.
            $.adsk.topiccontroller({
                searchOptions: config.searchOptions,

                favoritesController: controllers.favoritesController,

                create: function() {
                    controllers.topicController = $(this).topiccontroller("instance");
                },

                // Called when topic view are ready to render topic.
                ready: function() {
                    self._defTopicReady.resolve();
                },

                initgotopreviousbutton: function (event, data) {
                    self._view.initGoToPreviousButton(data.enabled);
                },

                initgotoparentbutton: function (event, data) {
                    self._view.initGoToParentButton(data.enabled);
                },

                initgotonextbutton: function (event, data) {
                    self._view.initGoToNextButton(data.enabled);
                },

                initfavoritesbutton: function (event, data) {
                    self._view.initFavoritesButton(data.enabled);
                },

                setupbreadcrumbs: function(event, data) {
                    self._view.updateBreadcrumbs(data.breadcrumbs, self.options.defaultTopicData);
                },

                setuptoc: function(event, data) {
                    if (self.options.config.ui.tabs.tocEnabled) {
                        self.options.viewControllers.tocController.toggleSelectedTocTopic(data);
                    }
                },

                opentopic: function (event, data) {
                    self._handleOpentopic(event, data);
                },

                flagfavorites: function(event, data) {
                    self._view.flagFavorites(data.added);

                    if (controllers.favoritestabController) {
                        controllers.favoritestabController.refresh();
                    }
                }
            });
        },

        _createIndexController: function(self, config, controllers) {
            // Create index view controller.
            if (config.ui.tabs.indexEnabled) {
                controllers.indexController = $.adsk.indexcontroller({
                    opentopic: function (event, data) {
                        self._handleOpentopic(event, data);
                    }
                });
            }
        },

        _createFavoritesController: function(self, config, controllers) {
            // Create favorites view controller.
            if (config.ui.tabs.favoritesEnabled) {
                controllers.favoritesTabController = $.adsk.favoritestabcontroller({
                    favoritesController: controllers.favoritesController,

                    labels: this.options.config.labels.favorites,

                    opentopic: function (event, data) {
                        self._handleOpentopic(event, data);
                    }
                });
            }
        },

        _createSearchController: function(self, config, controllers) {
             // Create search view controller.
            if (config.ui.tabs.searchEnabled) {
                $.adsk.searchtabcontroller({
                    config: this.options.config,

                    create: function() {
                        controllers.searchTabController = $(this).searchtabcontroller("instance");
                    },

                    ready: function() {
                        if (self.options.urlParams !== undefined &&
                            self.options.urlParams.length > 0) {

                            var p = self.options.urlParams;

                            if (p['query'] !== undefined &&
                                p['query'] !== '' &&
                                p['panels'] == undefined) {
                                    // Process search query.
                                    controllers.searchTabController.doSearch(decodeURIComponent(p['query']), self);
                                    controllers.searchTabController.closeSearchOptions();
                                    self._view.showTab('search');
                            }

                        }
                    },

                    highlight: function(event, searchOptions) {
                        if (searchOptions && searchOptions.searchQuery) {
                            controllers.topicController.highlightQuery(searchOptions);
                        }
                    },

                    dosearch: function(event) {
                        self._view.showTab('search');
                    }
                });
            }
        },

        _loadAndProcessTOC: function () {
            console.log('appcontroller: _loadAndProcessTOC');

            var controllers = this.options.viewControllers,
                tabs = this.options.config.ui.tabs;

            var tocRootNodesDataProcessor = $.adsk.tocdataprocessor({
                progress: function (event, data) {
                    console.debug('TOC root nodes data processing progress: ', data.percent);
                },

                indexed: $.proxy(function (event, toc) {
                    console.debug('TOC root nodes data indexed: ', toc);

                    if (tabs.tocEnabled) {
                        controllers.tocController.setTocRootNodesData(toc);
                    }
                }, this)
            });

            var tocRootNodesDataLoader = $.adsk.tocdataloader({
                config: this.options.config,

                file: 'treeRootData',

                event: 'register-toc-root-node-data',

                loaded: function (event, data) {
                    if (tabs.tocEnabled) {
                        tocRootNodesDataProcessor.setBookTocData(data);
                    }
                }
            });

            // Load TOC root nodes data.
            tocRootNodesDataLoader.loadTocData();

            var tocDataProcessor = $.adsk.tocdataprocessor({
                progress: function (event, data) {
                    console.log('TOC data processing progress: ', data.percent);
                },

                indexed: $.proxy(function (event, toc) {
                    console.log('TOC data indexed: ', toc);

                    this._setTocData(toc);

                    controllers.topicController.setTocData(toc);

                    if (tabs.searchEnabled) {
                        controllers.searchTabController.setTocData(toc);
                    }

                    if (tabs.tocEnabled) {
                        controllers.tocController.setTocData(toc.tocData);
                        controllers.tocController.renderToc();
                    }

                    if (tabs.favoritesEnabled) {
                        controllers.favoritesController.setTocData(toc);
                        controllers.favoritesTabController.refresh();
                    }
                }, this)
            });

            var tocDataLoader = $.adsk.tocdataloader({
                config: this.options.config,

                file: 'treeData',

                event: 'register-toc-data',

                loaded: function (event, data) {
                    tocDataProcessor.setBookTocData(data);
                }
            });

            // Load TOC data.
            tocDataLoader.loadTocData();
        },

        _loadContextIdData: function() {
            console.log('appcontroller: _loadContextIdData');

            var self = this;

            // Load context id data.
            var contextIdDataLoader = $.adsk.contextiddataloader({
                config: self.options.config,

                loaded: function (event, data) {
                    $.extend(self.options.config.contextIds, data.bookData);

                    if (data._loadedData == data._totalData) {
                        self._defContextIdsReady.resolve();
                    }
                }
            });

            contextIdDataLoader.loadContextIdData();
        },

        _setTocData: function (data) {
            console.log('appcontroller: _setTocData');

            this.options.toc = data;

            // Set default topic data.
            this.options.defaultTopicData = this._getDefaultTopicData();

            // Show breadcrumbs.
            this._view.updateBreadcrumbs([], this.options.defaultTopicData);

            this._defTocReady.resolve();
        },


        _getDefaultTopicData: function () {
            console.log('appcontroller: _getDefaultTopicData, ');

            var rootTopicData,
               config = this.options.config;

            if (config.defaultPage !== '') {
                rootTopicData = {
                    // TODO: default topic title through config.
                    data: "Home",
                    attr: {
                        href: config.defaultPage,
                        // TODO: description through config.
                        desc: ""
                    }
                }
            } else if (this.options.toc && this.options.toc.tocData) {
                // Try to use first topic data from root and component books TOC data as a default.
                $.each(this.options.toc.tocData, function(bookName, bookTocData) {
                    if (bookTocData[0]) {
                        rootTopicData = bookTocData[0];
                        // break
                        return false;
                    }
                });
            }

            return rootTopicData;
        },


        _handleOpentopic: function (event, data) {
            console.log('appcontroller: _handleOpentopic');

            // Change location hash to open provided topic.
            if (data === undefined || data.href === undefined || data.href == '') {
                return;
            }

            this._setHash('#!/url=' + data.href);
        },

        _addHashchangeHandler: function () {
            console.log('appcontroller: _addHashchangeHandler');

            var self = this;

            // Listen to hash change.
            if ('onhashchange' in window) {
                $(window).on('hashchange', function (event) {
                    self._handleHashchange(event);
                });
            } else {
                setInterval($.proxy(self._handleHashchange, this), 400);
            }
        },

        _handleHashchange: function (event, data) {
            console.log('appcontroller: _handleHashchange');

            var newHash = location.hash;

            this._processHash(newHash);
        },

        _processHash: function (hash) {
            console.log('appcontroller: _processHash, ', hash);

            if (!hash || hash == '') {
                // Load default topic.
                this._handleOpentopic({}, {
                    href: this.options.config.defaultPage
                });
            }

            // If hash starts with hashbang, crop it off.
            var url = this._fetchUrlFromHash(hash);

            if (!url || url == '') {
                // Do nothing.
                return;
            }

            this._processUrl(url);
        },

        _fetchUrlFromHash: function (hash) {
            console.log('appcontroller: _fetchUrlFromHash');

            var url = '';

            if (hash.indexOf('#!') == 0) {
                url = hash.substring(2);
            } else if (hash[0] == '#') {
                url = hash.substring(1);
            } else {
                url = hash;
            }

            return url;
        },

        _processUrl: function (url) {
            console.log('appcontroller: _processUrl: ', url);

            if (url.indexOf('/url=') == 0) {
                // Resolve internal path.
                this._resolveUrl(url);
            }
        },

        _resolveUrl: function (url) {
            console.log('appcontroller: _resolveUrl: ', url);

            var href = url.substring('/url='.length);

            if (!href || href == '') {
                return;
            }

            // Load topic content.
            var data = {
                href: href
            };

            this.options.viewControllers.topicController.openTopic(data);
        },

        _setHash: function (hash) {
            console.log('appcontroller: _setHash: ', hash);

            if (!hash || hash == '') {
                return;
            }

            if (hash[0] != '#') {
                return;
            }

            location.hash = hash;
        },

        _homeButtonclickHandler: function (event) {
            console.log('appcontroller: _homeButtonclickHandler');

            // Open default topic.
            this._handleOpentopic({}, {
                href: this.options.config.defaultPage
            });
        },

        _previousButtonclickHandler: function (event) {
            console.log('appcontroller: _previousButtonclickHandler');

            this.options.viewControllers.topicController.openPreviousTopic();
        },

        _parentButtonclickHandler: function (event) {
            console.log('appcontroller: _parentButtonclickHandler');

            this.options.viewControllers.topicController.openParentTopic();
        },

        _followingButtonclickHandler: function (event) {
            console.log('appcontroller: _followingButtonclickHandler');

            this.options.viewControllers.topicController.openFollowingTopic();
        },

        _favoritesButtonclickHandler: function (event) {
            console.log('appcontroller: _favoritesButtonclickHandler');

            var controllers = this.options.viewControllers;

            controllers.topicController.addOrRemoveTopicFromFavorites();
            controllers.favoritesTabController.refresh();
        },

        _shareButtonclickHandler: function (event) {
            console.log('appcontroller: _shareButtonclickHandler');

            var controller = this.options.viewControllers.shareController;

            if (!controller) {
                controller = this.options.viewControllers.shareController = $.adsk.sharecontroller({
                    //opentopic: $.proxy(this._handleOpentopic, this)
                });
            }

            controller.refresh();
            controller.showShareMenu();
        },

        _printButtonclickHandler: function (event) {
            console.log('appcontroller: _printButtonclickHandler');

            window.print();
        }

        /* Public members. */

    });
})(jQuery);
// SIG // Begin signature block
// SIG // MIIXrwYJKoZIhvcNAQcCoIIXoDCCF5wCAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFMPyDGvOox67
// SIG // X/SIu5/CYEMeBesooIIS3zCCA+4wggNXoAMCAQICEH6T
// SIG // 6/t8xk5Z6kuad9QG/DswDQYJKoZIhvcNAQEFBQAwgYsx
// SIG // CzAJBgNVBAYTAlpBMRUwEwYDVQQIEwxXZXN0ZXJuIENh
// SIG // cGUxFDASBgNVBAcTC0R1cmJhbnZpbGxlMQ8wDQYDVQQK
// SIG // EwZUaGF3dGUxHTAbBgNVBAsTFFRoYXd0ZSBDZXJ0aWZp
// SIG // Y2F0aW9uMR8wHQYDVQQDExZUaGF3dGUgVGltZXN0YW1w
// SIG // aW5nIENBMB4XDTEyMTIyMTAwMDAwMFoXDTIwMTIzMDIz
// SIG // NTk1OVowXjELMAkGA1UEBhMCVVMxHTAbBgNVBAoTFFN5
// SIG // bWFudGVjIENvcnBvcmF0aW9uMTAwLgYDVQQDEydTeW1h
// SIG // bnRlYyBUaW1lIFN0YW1waW5nIFNlcnZpY2VzIENBIC0g
// SIG // RzIwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
// SIG // AQCxrLNJVEuXHBIK2CV5kSJXKm/cuCbEQ3Nrwr8uUFr7
// SIG // FMJ2jkMBJUO0oeJF9Oi3e8N0zCLXtJQAAvdN7b+0t0Qk
// SIG // a81fRTvRRM5DEnMXgotptCvLmR6schsmTXEfsTHd+1Fh
// SIG // AlOmqvVJLAV4RaUvic7nmef+jOJXPz3GktxK+Hsz5HkK
// SIG // +/B1iEGc/8UDUZmq12yfk2mHZSmDhcJgFMTIyTsU2sCB
// SIG // 8B8NdN6SIqvK9/t0fCfm90obf6fDni2uiuqm5qonFn1h
// SIG // 95hxEbziUKFL5V365Q6nLJ+qZSDT2JboyHylTkhE/xni
// SIG // RAeSC9dohIBdanhkRc1gRn5UwRN8xXnxycFxAgMBAAGj
// SIG // gfowgfcwHQYDVR0OBBYEFF+a9W5czMx0mtTdfe8/2+xM
// SIG // gC7dMDIGCCsGAQUFBwEBBCYwJDAiBggrBgEFBQcwAYYW
// SIG // aHR0cDovL29jc3AudGhhd3RlLmNvbTASBgNVHRMBAf8E
// SIG // CDAGAQH/AgEAMD8GA1UdHwQ4MDYwNKAyoDCGLmh0dHA6
// SIG // Ly9jcmwudGhhd3RlLmNvbS9UaGF3dGVUaW1lc3RhbXBp
// SIG // bmdDQS5jcmwwEwYDVR0lBAwwCgYIKwYBBQUHAwgwDgYD
// SIG // VR0PAQH/BAQDAgEGMCgGA1UdEQQhMB+kHTAbMRkwFwYD
// SIG // VQQDExBUaW1lU3RhbXAtMjA0OC0xMA0GCSqGSIb3DQEB
// SIG // BQUAA4GBAAMJm495739ZMKrvaLX64wkdu0+CBl03X6ZS
// SIG // nxaN6hySCURu9W3rWHww6PlpjSNzCxJvR6muORH4KrGb
// SIG // sBrDjutZlgCtzgxNstAxpghcKnr84nodV0yoZRjpeUBi
// SIG // JZZux8c3aoMhCI5B6t3ZVz8dd0mHKhYGXqY4aiISo1EZ
// SIG // g362MIIEozCCA4ugAwIBAgIQDs/0OMj+vzVuBNhqmBsa
// SIG // UDANBgkqhkiG9w0BAQUFADBeMQswCQYDVQQGEwJVUzEd
// SIG // MBsGA1UEChMUU3ltYW50ZWMgQ29ycG9yYXRpb24xMDAu
// SIG // BgNVBAMTJ1N5bWFudGVjIFRpbWUgU3RhbXBpbmcgU2Vy
// SIG // dmljZXMgQ0EgLSBHMjAeFw0xMjEwMTgwMDAwMDBaFw0y
// SIG // MDEyMjkyMzU5NTlaMGIxCzAJBgNVBAYTAlVTMR0wGwYD
// SIG // VQQKExRTeW1hbnRlYyBDb3Jwb3JhdGlvbjE0MDIGA1UE
// SIG // AxMrU3ltYW50ZWMgVGltZSBTdGFtcGluZyBTZXJ2aWNl
// SIG // cyBTaWduZXIgLSBHNDCCASIwDQYJKoZIhvcNAQEBBQAD
// SIG // ggEPADCCAQoCggEBAKJjCzlEuLsjp0RJuw7/ofBhClOT
// SIG // sJjbrSwPSsVu/4Y8U1UPFc4EPyv9qZaW2b5heQtbyUyG
// SIG // duXgQ0sile7CK0PBn9hotI5AT+6FOLkRxSPyZFjwFTJv
// SIG // TlehroikAtcqHs1L4d1j1ReJMluwXplaqJ0oUA4X7pbb
// SIG // YTtFUR3PElYLkkf8q672Zj1HrHBy55LnX80QucSDZJQZ
// SIG // vSWA4ejSIqXQugJ6oXeTW2XD7hd0vEGGKtwITIySjJEt
// SIG // nndEH2jWqHR32w5bMotWizO92WPISZ06xcXqMwvS8aMb
// SIG // 9Iu+2bNXizveBKd6IrIkri7HcMW+ToMmCPsLvalPmQjh
// SIG // EChyqs0CAwEAAaOCAVcwggFTMAwGA1UdEwEB/wQCMAAw
// SIG // FgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwDgYDVR0PAQH/
// SIG // BAQDAgeAMHMGCCsGAQUFBwEBBGcwZTAqBggrBgEFBQcw
// SIG // AYYeaHR0cDovL3RzLW9jc3Aud3Muc3ltYW50ZWMuY29t
// SIG // MDcGCCsGAQUFBzAChitodHRwOi8vdHMtYWlhLndzLnN5
// SIG // bWFudGVjLmNvbS90c3MtY2EtZzIuY2VyMDwGA1UdHwQ1
// SIG // MDMwMaAvoC2GK2h0dHA6Ly90cy1jcmwud3Muc3ltYW50
// SIG // ZWMuY29tL3Rzcy1jYS1nMi5jcmwwKAYDVR0RBCEwH6Qd
// SIG // MBsxGTAXBgNVBAMTEFRpbWVTdGFtcC0yMDQ4LTIwHQYD
// SIG // VR0OBBYEFEbGaaMOShQe1UzaUmMXP142vA3mMB8GA1Ud
// SIG // IwQYMBaAFF+a9W5czMx0mtTdfe8/2+xMgC7dMA0GCSqG
// SIG // SIb3DQEBBQUAA4IBAQB4O7SRKgBM8I9iMDd4o4QnB28Y
// SIG // st4l3KDUlAOqhk4ln5pAAxzdzuN5yyFoBtq2MrRtv/Qs
// SIG // JmMz5ElkbQ3mw2cO9wWkNWx8iRbG6bLfsundIMZxD82V
// SIG // dNy2XN69Nx9DeOZ4tc0oBCCjqvFLxIgpkQ6A0RH83Vx2
// SIG // bk9eDkVGQW4NsOo4mrE62glxEPwcebSAe6xp9P2ctgwW
// SIG // K/F/Wwk9m1viFsoTgW0ALjgNqCmPLOGy9FqpAa8VnCwv
// SIG // SRvbIrvD/niUUcOGsYKIXfA9tFGheTMrLnu53CAJE3Hr
// SIG // ahlbz+ilMFcsiUk/uc9/yb8+ImhjU5q9aXSsxR08f5Lg
// SIG // w7wc2AR1MIIE5TCCA82gAwIBAgIQGGKWU9W9DKB4gLDR
// SIG // /Nm99TANBgkqhkiG9w0BAQsFADB/MQswCQYDVQQGEwJV
// SIG // UzEdMBsGA1UEChMUU3ltYW50ZWMgQ29ycG9yYXRpb24x
// SIG // HzAdBgNVBAsTFlN5bWFudGVjIFRydXN0IE5ldHdvcmsx
// SIG // MDAuBgNVBAMTJ1N5bWFudGVjIENsYXNzIDMgU0hBMjU2
// SIG // IENvZGUgU2lnbmluZyBDQTAeFw0xNjAxMjgwMDAwMDBa
// SIG // Fw0xODA0MTAyMzU5NTlaMH0xCzAJBgNVBAYTAkNIMRIw
// SIG // EAYDVQQIEwlOZXVjaGF0ZWwxEjAQBgNVBAcTCU5ldWNo
// SIG // YXRlbDEiMCAGA1UEChQZQXV0b2Rlc2sgRGV2ZWxvcG1l
// SIG // bnQgU2FybDEiMCAGA1UEAxQZQXV0b2Rlc2sgRGV2ZWxv
// SIG // cG1lbnQgU2FybDCCASIwDQYJKoZIhvcNAQEBBQADggEP
// SIG // ADCCAQoCggEBALOBje/8PFtBHV0+YQz+EJZzXU+JAy6j
// SIG // OhMiK0n4pWQrgLdfLmoNZUvUL5AVuWoHkzK4f6WLWEjQ
// SIG // eGR4SEZcvDCJjol0n6vdsYAfRurwhekestab635tKUEy
// SIG // JRNKXhC4zuDSgryc+2zUsZa08HSHZaTsFGZzEuQAdxWf
// SIG // iyvD6G5oTu72nj3kDWWn3vqh+zyGkOIhtsIDKZyHpg71
// SIG // 7hPiqEe96bX5LpZgkSOJdAdwpl40NEUFfpNdelT2sOCh
// SIG // MfG63Dg2PR6EjOK5zHgMZm09mPgZBfyNoBV03C1ySSUi
// SIG // UsAHM2ND4wnGfl/GbYEgIVQXWlspRbtpRHTaNavebLJY
// SIG // tWECAwEAAaOCAV0wggFZMAkGA1UdEwQCMAAwDgYDVR0P
// SIG // AQH/BAQDAgeAMCsGA1UdHwQkMCIwIKAeoByGGmh0dHA6
// SIG // Ly9zdi5zeW1jYi5jb20vc3YuY3JsMGEGA1UdIARaMFgw
// SIG // VgYGZ4EMAQQBMEwwIwYIKwYBBQUHAgEWF2h0dHBzOi8v
// SIG // ZC5zeW1jYi5jb20vY3BzMCUGCCsGAQUFBwICMBkMF2h0
// SIG // dHBzOi8vZC5zeW1jYi5jb20vcnBhMBMGA1UdJQQMMAoG
// SIG // CCsGAQUFBwMDMFcGCCsGAQUFBwEBBEswSTAfBggrBgEF
// SIG // BQcwAYYTaHR0cDovL3N2LnN5bWNkLmNvbTAmBggrBgEF
// SIG // BQcwAoYaaHR0cDovL3N2LnN5bWNiLmNvbS9zdi5jcnQw
// SIG // HwYDVR0jBBgwFoAUljtT8Hkzl699g+8uK8zKt4YecmYw
// SIG // HQYDVR0OBBYEFHqJl6/oWFvaZKJVCptH8DQRl8JIMA0G
// SIG // CSqGSIb3DQEBCwUAA4IBAQAXhA6ZwymxqqV+k4uKThLd
// SIG // PgPsvZBRcCJJZi3eYMl7HjIDbxSdx03ueVc/Sx3n9OUK
// SIG // U+DupPpLFUnKUJa/X8HPWhcvJftXom9NvGWbYPJM7CBc
// SIG // eTKuXgP5ffKwf1r9pJhok0pEhSA8b6N97pmKQwJDE6XI
// SIG // Z4M5FYw9YELKPRCHhinZCfPHLo+0GxaH+W4AbGd1N2BZ
// SIG // dX2egsgNTGBnC6xGxGMC8ckaj5wyIm4LgklE0pKbxuG9
// SIG // 075426gJ6ASAtCevNViBj9hH3TcNy4aU0ruH6By5gZaR
// SIG // 5lc7ZdaTuBWz0+4wGIK1IiviMSJv19v+h0vpEyn1b8sE
// SIG // 6L7D8JD4szHnMIIFWTCCBEGgAwIBAgIQPXjX+XZJYLJh
// SIG // ffTwHsqGKjANBgkqhkiG9w0BAQsFADCByjELMAkGA1UE
// SIG // BhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8w
// SIG // HQYDVQQLExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTow
// SIG // OAYDVQQLEzEoYykgMjAwNiBWZXJpU2lnbiwgSW5jLiAt
// SIG // IEZvciBhdXRob3JpemVkIHVzZSBvbmx5MUUwQwYDVQQD
// SIG // EzxWZXJpU2lnbiBDbGFzcyAzIFB1YmxpYyBQcmltYXJ5
// SIG // IENlcnRpZmljYXRpb24gQXV0aG9yaXR5IC0gRzUwHhcN
// SIG // MTMxMjEwMDAwMDAwWhcNMjMxMjA5MjM1OTU5WjB/MQsw
// SIG // CQYDVQQGEwJVUzEdMBsGA1UEChMUU3ltYW50ZWMgQ29y
// SIG // cG9yYXRpb24xHzAdBgNVBAsTFlN5bWFudGVjIFRydXN0
// SIG // IE5ldHdvcmsxMDAuBgNVBAMTJ1N5bWFudGVjIENsYXNz
// SIG // IDMgU0hBMjU2IENvZGUgU2lnbmluZyBDQTCCASIwDQYJ
// SIG // KoZIhvcNAQEBBQADggEPADCCAQoCggEBAJeDHgAWryyx
// SIG // 0gjE12iTUWAecfbiR7TbWE0jYmq0v1obUfejDRh3aLvY
// SIG // NqsvIVDanvPnXydOC8KXyAlwk6naXA1OpA2RoLTsFM6R
// SIG // clQuzqPbROlSGz9BPMpK5KrA6DmrU8wh0MzPf5vmwsxY
// SIG // aoIV7j02zxzFlwckjvF7vjEtPW7ctZlCn0thlV8ccO4X
// SIG // fduL5WGJeMdoG68ReBqYrsRVR1PZszLWoQ5GQMWXkorR
// SIG // U6eZW4U1V9Pqk2JhIArHMHckEU1ig7a6e2iCMe5lyt/5
// SIG // 1Y2yNdyMK29qclxghJzyDJRewFZSAEjM0/ilfd4v1xPk
// SIG // OKiE1Ua4E4bCG53qWjjdm9sCAwEAAaOCAYMwggF/MC8G
// SIG // CCsGAQUFBwEBBCMwITAfBggrBgEFBQcwAYYTaHR0cDov
// SIG // L3MyLnN5bWNiLmNvbTASBgNVHRMBAf8ECDAGAQH/AgEA
// SIG // MGwGA1UdIARlMGMwYQYLYIZIAYb4RQEHFwMwUjAmBggr
// SIG // BgEFBQcCARYaaHR0cDovL3d3dy5zeW1hdXRoLmNvbS9j
// SIG // cHMwKAYIKwYBBQUHAgIwHBoaaHR0cDovL3d3dy5zeW1h
// SIG // dXRoLmNvbS9ycGEwMAYDVR0fBCkwJzAloCOgIYYfaHR0
// SIG // cDovL3MxLnN5bWNiLmNvbS9wY2EzLWc1LmNybDAdBgNV
// SIG // HSUEFjAUBggrBgEFBQcDAgYIKwYBBQUHAwMwDgYDVR0P
// SIG // AQH/BAQDAgEGMCkGA1UdEQQiMCCkHjAcMRowGAYDVQQD
// SIG // ExFTeW1hbnRlY1BLSS0xLTU2NzAdBgNVHQ4EFgQUljtT
// SIG // 8Hkzl699g+8uK8zKt4YecmYwHwYDVR0jBBgwFoAUf9Nl
// SIG // p8Ld7LvwMAnzQzn6Aq8zMTMwDQYJKoZIhvcNAQELBQAD
// SIG // ggEBABOFGh5pqTf3oL2kr34dYVP+nYxeDKZ1HngXI939
// SIG // 7BoDVTn7cZXHZVqnjjDSRFph23Bv2iEFwi5zuknx0ZP+
// SIG // XcnNXgPgiZ4/dB7X9ziLqdbPuzUvM1ioklbRyE07guZ5
// SIG // hBb8KLCxR/Mdoj7uh9mmf6RWpT+thC4p3ny8qKqjPQQB
// SIG // 6rqTog5QIikXTIfkOhFf1qQliZsFay+0yQFMJ3sLrBkF
// SIG // IqBgFT/ayftNTI/7cmd3/SeUx7o1DohJ/o39KK9KEr0N
// SIG // s5cF3kQMFfo2KwPcwVAB8aERXRTl4r0nS1S+K4ReD6bD
// SIG // dAUK75fDiSKxH3fzvc1D1PFMqT+1i4SvZPLQFCExggQ8
// SIG // MIIEOAIBATCBkzB/MQswCQYDVQQGEwJVUzEdMBsGA1UE
// SIG // ChMUU3ltYW50ZWMgQ29ycG9yYXRpb24xHzAdBgNVBAsT
// SIG // FlN5bWFudGVjIFRydXN0IE5ldHdvcmsxMDAuBgNVBAMT
// SIG // J1N5bWFudGVjIENsYXNzIDMgU0hBMjU2IENvZGUgU2ln
// SIG // bmluZyBDQQIQGGKWU9W9DKB4gLDR/Nm99TAJBgUrDgMC
// SIG // GgUAoHAwEAYKKwYBBAGCNwIBDDECMAAwGQYJKoZIhvcN
// SIG // AQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEO
// SIG // MAwGCisGAQQBgjcCARUwIwYJKoZIhvcNAQkEMRYEFCyA
// SIG // 1tDa8Pse7dRqmmJVTX7wiE4yMA0GCSqGSIb3DQEBAQUA
// SIG // BIIBAJBOSL+NxyhonpWIdUN7THKMHJRBVqr3hXqskAi9
// SIG // ediUyKkCUgVgJ2btL+pv+OQpbzozvlZUMP3Rs8Osxmsm
// SIG // sjStON8ZBdiezQ+ra0xIUZU8+kapxaBuYRNXlNFiHPEE
// SIG // MweCrFMFYQj7R+UEeNsbAbK1tjNUPsdtAGHzLuCfvGzd
// SIG // dB+UXlG7fvUUiJhCbP4nhUjVIK1WseTcsPzzTdN/a1S1
// SIG // dF1LMUQAxPjDIsKS9YJv68d2IwDeA9fDaxyINivmJqrn
// SIG // hNbBA3AuC3218tsIGhkUgcFwzHxErHDgTa8zrYGs8A9n
// SIG // GqPWC12O0kaKBbZd0aaBs7RoUffXVCroG7WymbqhggIL
// SIG // MIICBwYJKoZIhvcNAQkGMYIB+DCCAfQCAQEwcjBeMQsw
// SIG // CQYDVQQGEwJVUzEdMBsGA1UEChMUU3ltYW50ZWMgQ29y
// SIG // cG9yYXRpb24xMDAuBgNVBAMTJ1N5bWFudGVjIFRpbWUg
// SIG // U3RhbXBpbmcgU2VydmljZXMgQ0EgLSBHMgIQDs/0OMj+
// SIG // vzVuBNhqmBsaUDAJBgUrDgMCGgUAoF0wGAYJKoZIhvcN
// SIG // AQkDMQsGCSqGSIb3DQEHATAcBgkqhkiG9w0BCQUxDxcN
// SIG // MTcwMjA5MDg0NzIxWjAjBgkqhkiG9w0BCQQxFgQUWNuY
// SIG // oOzDQnzMLYpQETb2/87W65gwDQYJKoZIhvcNAQEBBQAE
// SIG // ggEAdevdHsvrcROGVe8O3xcFYMP4p/dMKirItGF89fEX
// SIG // ObRX48X1QeJnJErx5dUJ/1cHqppdphcFlIkJabRE9v72
// SIG // VJ6SQi0BFfuGvf4qQ1rAe2JNnPGJuqI7laTyxICxbdqR
// SIG // ozXh8vjWo1J5vcbojo/LMBSuB3p73scO7qqwdQo/3mF0
// SIG // 5wkLCs31CV6nb0SWbw1veIbBWAigw0mXHU6KpdhIy1wo
// SIG // ndmHA5+7GMVdLqXhKAtEQ7ib/FeddRnWDn7nYPDw1wl2
// SIG // E72TaA3oiSIh/XP+IqkJMxElE5pQq54sAGIXjiK7XXcY
// SIG // LGbJpeFurWdoIWaeJ+bIVRd8K9TUtzrIKEU5FQ==
// SIG // End signature block
