"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.assertValidIsolationLevel = assertValidIsolationLevel;
exports.Transaction = exports.isolationLevelByValue = exports.ISOLATION_LEVEL = exports.OPERATION_TYPE = void 0;

var _writableTrackingBuffer = _interopRequireDefault(require("./tracking-buffer/writable-tracking-buffer"));

var _allHeaders = require("./all-headers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
  s2.2.6.8
 */
const OPERATION_TYPE = {
  TM_GET_DTC_ADDRESS: 0x00,
  TM_PROPAGATE_XACT: 0x01,
  TM_BEGIN_XACT: 0x05,
  TM_PROMOTE_XACT: 0x06,
  TM_COMMIT_XACT: 0x07,
  TM_ROLLBACK_XACT: 0x08,
  TM_SAVE_XACT: 0x09
};
exports.OPERATION_TYPE = OPERATION_TYPE;
const ISOLATION_LEVEL = {
  NO_CHANGE: 0x00,
  READ_UNCOMMITTED: 0x01,
  READ_COMMITTED: 0x02,
  REPEATABLE_READ: 0x03,
  SERIALIZABLE: 0x04,
  SNAPSHOT: 0x05
};
exports.ISOLATION_LEVEL = ISOLATION_LEVEL;
const isolationLevelByValue = {};
exports.isolationLevelByValue = isolationLevelByValue;

for (const name in ISOLATION_LEVEL) {
  const value = ISOLATION_LEVEL[name];
  isolationLevelByValue[value] = name;
}

function assertValidIsolationLevel(isolationLevel, name) {
  if (typeof isolationLevel !== 'number') {
    throw new TypeError(`The "${name}" ${name.includes('.') ? 'property' : 'argument'} must be of type number. Received type ${typeof isolationLevel} (${isolationLevel})`);
  }

  if (!Number.isInteger(isolationLevel)) {
    throw new RangeError(`The value of "${name}" is out of range. It must be an integer. Received: ${isolationLevel}`);
  }

  if (!(isolationLevel >= 0 && isolationLevel <= 5)) {
    throw new RangeError(`The value of "${name}" is out of range. It must be >= 0 && <= 5. Received: ${isolationLevel}`);
  }
}

class Transaction {
  constructor(name, isolationLevel = ISOLATION_LEVEL.NO_CHANGE) {
    this.name = void 0;
    this.isolationLevel = void 0;
    this.outstandingRequestCount = void 0;
    this.name = name;
    this.isolationLevel = isolationLevel;
    this.outstandingRequestCount = 1;
  }

  beginPayload(txnDescriptor) {
    const buffer = new _writableTrackingBuffer.default(100, 'ucs2');
    (0, _allHeaders.writeToTrackingBuffer)(buffer, txnDescriptor, this.outstandingRequestCount);
    buffer.writeUShort(OPERATION_TYPE.TM_BEGIN_XACT);
    buffer.writeUInt8(this.isolationLevel);
    buffer.writeUInt8(this.name.length * 2);
    buffer.writeString(this.name, 'ucs2');
    return {
      *[Symbol.iterator]() {
        yield buffer.data;
      },

      toString: () => {
        return 'Begin Transaction: name=' + this.name + ', isolationLevel=' + isolationLevelByValue[this.isolationLevel];
      }
    };
  }

  commitPayload(txnDescriptor) {
    const buffer = new _writableTrackingBuffer.default(100, 'ascii');
    (0, _allHeaders.writeToTrackingBuffer)(buffer, txnDescriptor, this.outstandingRequestCount);
    buffer.writeUShort(OPERATION_TYPE.TM_COMMIT_XACT);
    buffer.writeUInt8(this.name.length * 2);
    buffer.writeString(this.name, 'ucs2'); // No fBeginXact flag, so no new transaction is started.

    buffer.writeUInt8(0);
    return {
      *[Symbol.iterator]() {
        yield buffer.data;
      },

      toString: () => {
        return 'Commit Transaction: name=' + this.name;
      }
    };
  }

  rollbackPayload(txnDescriptor) {
    const buffer = new _writableTrackingBuffer.default(100, 'ascii');
    (0, _allHeaders.writeToTrackingBuffer)(buffer, txnDescriptor, this.outstandingRequestCount);
    buffer.writeUShort(OPERATION_TYPE.TM_ROLLBACK_XACT);
    buffer.writeUInt8(this.name.length * 2);
    buffer.writeString(this.name, 'ucs2'); // No fBeginXact flag, so no new transaction is started.

    buffer.writeUInt8(0);
    return {
      *[Symbol.iterator]() {
        yield buffer.data;
      },

      toString: () => {
        return 'Rollback Transaction: name=' + this.name;
      }
    };
  }

  savePayload(txnDescriptor) {
    const buffer = new _writableTrackingBuffer.default(100, 'ascii');
    (0, _allHeaders.writeToTrackingBuffer)(buffer, txnDescriptor, this.outstandingRequestCount);
    buffer.writeUShort(OPERATION_TYPE.TM_SAVE_XACT);
    buffer.writeUInt8(this.name.length * 2);
    buffer.writeString(this.name, 'ucs2');
    return {
      *[Symbol.iterator]() {
        yield buffer.data;
      },

      toString: () => {
        return 'Save Transaction: name=' + this.name;
      }
    };
  }

  isolationLevelToTSQL() {
    switch (this.isolationLevel) {
      case ISOLATION_LEVEL.READ_UNCOMMITTED:
        return 'READ UNCOMMITTED';

      case ISOLATION_LEVEL.READ_COMMITTED:
        return 'READ COMMITTED';

      case ISOLATION_LEVEL.REPEATABLE_READ:
        return 'REPEATABLE READ';

      case ISOLATION_LEVEL.SERIALIZABLE:
        return 'SERIALIZABLE';

      case ISOLATION_LEVEL.SNAPSHOT:
        return 'SNAPSHOT';
    }

    return '';
  }

}

exports.Transaction = Transaction;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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