"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _writableTrackingBuffer = _interopRequireDefault(require("./tracking-buffer/writable-tracking-buffer"));

var crypto = _interopRequireWildcard(require("crypto"));

var _jsbi = _interopRequireDefault(require("jsbi"));

function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }

function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

class NTLMResponsePayload {
  constructor(loginData) {
    this.data = void 0;
    this.data = this.createResponse(loginData);
  }

  toString(indent = '') {
    return indent + 'NTLM Auth';
  }

  createResponse(challenge) {
    const client_nonce = this.createClientNonce();
    const lmv2len = 24;
    const ntlmv2len = 16;
    const domain = challenge.domain;
    const username = challenge.userName;
    const password = challenge.password;
    const ntlmData = challenge.ntlmpacket;
    const server_data = ntlmData.target;
    const server_nonce = ntlmData.nonce;
    const bufferLength = 64 + domain.length * 2 + username.length * 2 + lmv2len + ntlmv2len + 8 + 8 + 8 + 4 + server_data.length + 4;
    const data = new _writableTrackingBuffer.default(bufferLength);
    data.position = 0;
    data.writeString('NTLMSSP\u0000', 'utf8');
    data.writeUInt32LE(0x03);
    const baseIdx = 64;
    const dnIdx = baseIdx;
    const unIdx = dnIdx + domain.length * 2;
    const l2Idx = unIdx + username.length * 2;
    const ntIdx = l2Idx + lmv2len;
    data.writeUInt16LE(lmv2len);
    data.writeUInt16LE(lmv2len);
    data.writeUInt32LE(l2Idx);
    data.writeUInt16LE(ntlmv2len);
    data.writeUInt16LE(ntlmv2len);
    data.writeUInt32LE(ntIdx);
    data.writeUInt16LE(domain.length * 2);
    data.writeUInt16LE(domain.length * 2);
    data.writeUInt32LE(dnIdx);
    data.writeUInt16LE(username.length * 2);
    data.writeUInt16LE(username.length * 2);
    data.writeUInt32LE(unIdx);
    data.writeUInt16LE(0);
    data.writeUInt16LE(0);
    data.writeUInt32LE(baseIdx);
    data.writeUInt16LE(0);
    data.writeUInt16LE(0);
    data.writeUInt32LE(baseIdx);
    data.writeUInt16LE(0x8201);
    data.writeUInt16LE(0x08);
    data.writeString(domain, 'ucs2');
    data.writeString(username, 'ucs2');
    const lmv2Data = this.lmv2Response(domain, username, password, server_nonce, client_nonce);
    data.copyFrom(lmv2Data);
    const genTime = new Date().getTime();
    const ntlmDataBuffer = this.ntlmv2Response(domain, username, password, server_nonce, server_data, client_nonce, genTime);
    data.copyFrom(ntlmDataBuffer);
    data.writeUInt32LE(0x0101);
    data.writeUInt32LE(0x0000);
    const timestamp = this.createTimestamp(genTime);
    data.copyFrom(timestamp);
    data.copyFrom(client_nonce);
    data.writeUInt32LE(0x0000);
    data.copyFrom(server_data);
    data.writeUInt32LE(0x0000);
    return data.data;
  }

  createClientNonce() {
    const client_nonce = Buffer.alloc(8, 0);
    let nidx = 0;

    while (nidx < 8) {
      client_nonce.writeUInt8(Math.ceil(Math.random() * 255), nidx);
      nidx++;
    }

    return client_nonce;
  }

  ntlmv2Response(domain, user, password, serverNonce, targetInfo, clientNonce, mytime) {
    const timestamp = this.createTimestamp(mytime);
    const hash = this.ntv2Hash(domain, user, password);
    const dataLength = 40 + targetInfo.length;
    const data = Buffer.alloc(dataLength, 0);
    serverNonce.copy(data, 0, 0, 8);
    data.writeUInt32LE(0x101, 8);
    data.writeUInt32LE(0x0, 12);
    timestamp.copy(data, 16, 0, 8);
    clientNonce.copy(data, 24, 0, 8);
    data.writeUInt32LE(0x0, 32);
    targetInfo.copy(data, 36, 0, targetInfo.length);
    data.writeUInt32LE(0x0, 36 + targetInfo.length);
    return this.hmacMD5(data, hash);
  }

  createTimestamp(time) {
    const tenthsOfAMicrosecond = _jsbi.default.multiply(_jsbi.default.add(_jsbi.default.BigInt(time), _jsbi.default.BigInt(11644473600)), _jsbi.default.BigInt(10000000));

    const lo = _jsbi.default.toNumber(_jsbi.default.bitwiseAnd(tenthsOfAMicrosecond, _jsbi.default.BigInt(0xffffffff)));

    const hi = _jsbi.default.toNumber(_jsbi.default.bitwiseAnd(_jsbi.default.signedRightShift(tenthsOfAMicrosecond, _jsbi.default.BigInt(32)), _jsbi.default.BigInt(0xffffffff)));

    const result = Buffer.alloc(8);
    result.writeUInt32LE(lo, 0);
    result.writeUInt32LE(hi, 4);
    return result;
  }

  lmv2Response(domain, user, password, serverNonce, clientNonce) {
    const hash = this.ntv2Hash(domain, user, password);
    const data = Buffer.alloc(serverNonce.length + clientNonce.length, 0);
    serverNonce.copy(data);
    clientNonce.copy(data, serverNonce.length, 0, clientNonce.length);
    const newhash = this.hmacMD5(data, hash);
    const response = Buffer.alloc(newhash.length + clientNonce.length, 0);
    newhash.copy(response);
    clientNonce.copy(response, newhash.length, 0, clientNonce.length);
    return response;
  }

  ntv2Hash(domain, user, password) {
    const hash = this.ntHash(password);
    const identity = Buffer.from(user.toUpperCase() + domain.toUpperCase(), 'ucs2');
    return this.hmacMD5(identity, hash);
  }

  ntHash(text) {
    const unicodeString = Buffer.from(text, 'ucs2');
    return crypto.createHash('md4').update(unicodeString).digest();
  }

  hmacMD5(data, key) {
    return crypto.createHmac('MD5', key).update(data).digest();
  }

}

var _default = NTLMResponsePayload;
exports.default = _default;
module.exports = NTLMResponsePayload;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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