"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _numericn = _interopRequireDefault(require("./numericn"));

var _writableTrackingBuffer = _interopRequireDefault(require("../tracking-buffer/writable-tracking-buffer"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const NULL_LENGTH = Buffer.from([0x00]);
const Numeric = {
  id: 0x3F,
  type: 'NUMERIC',
  name: 'Numeric',
  declaration: function (parameter) {
    return 'numeric(' + this.resolvePrecision(parameter) + ', ' + this.resolveScale(parameter) + ')';
  },
  resolvePrecision: function (parameter) {
    if (parameter.precision != null) {
      return parameter.precision;
    } else if (parameter.value === null) {
      return 1;
    } else {
      return 18;
    }
  },
  resolveScale: function (parameter) {
    if (parameter.scale != null) {
      return parameter.scale;
    } else {
      return 0;
    }
  },

  generateTypeInfo(parameter) {
    let precision;

    if (parameter.precision <= 9) {
      precision = 0x05;
    } else if (parameter.precision <= 19) {
      precision = 0x09;
    } else if (parameter.precision <= 28) {
      precision = 0x0D;
    } else {
      precision = 0x11;
    }

    return Buffer.from([_numericn.default.id, precision, parameter.precision, parameter.scale]);
  },

  generateParameterLength(parameter, options) {
    if (parameter.value == null) {
      return NULL_LENGTH;
    }

    const precision = parameter.precision;

    if (precision <= 9) {
      return Buffer.from([0x05]);
    } else if (precision <= 19) {
      return Buffer.from([0x09]);
    } else if (precision <= 28) {
      return Buffer.from([0x0D]);
    } else {
      return Buffer.from([0x11]);
    }
  },

  *generateParameterData(parameter, options) {
    if (parameter.value == null) {
      return;
    }

    const sign = parameter.value < 0 ? 0 : 1;
    const value = Math.round(Math.abs(parameter.value * Math.pow(10, parameter.scale)));

    if (parameter.precision <= 9) {
      const buffer = Buffer.alloc(5);
      buffer.writeUInt8(sign, 0);
      buffer.writeUInt32LE(value, 1);
      yield buffer;
    } else if (parameter.precision <= 19) {
      const buffer = new _writableTrackingBuffer.default(10);
      buffer.writeUInt8(sign);
      buffer.writeUInt64LE(value);
      yield buffer.data;
    } else if (parameter.precision <= 28) {
      const buffer = new _writableTrackingBuffer.default(14);
      buffer.writeUInt8(sign);
      buffer.writeUInt64LE(value);
      buffer.writeUInt32LE(0x00000000);
      yield buffer.data;
    } else {
      const buffer = new _writableTrackingBuffer.default(18);
      buffer.writeUInt8(sign);
      buffer.writeUInt64LE(value);
      buffer.writeUInt32LE(0x00000000);
      buffer.writeUInt32LE(0x00000000);
      yield buffer.data;
    }
  },

  validate: function (value) {
    if (value == null) {
      return null;
    }

    value = parseFloat(value);

    if (isNaN(value)) {
      throw new TypeError('Invalid number.');
    }

    return value;
  }
};
var _default = Numeric;
exports.default = _default;
module.exports = Numeric;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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