"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getKey = void 0;

var _symmetricKey = _interopRequireDefault(require("./symmetric-key"));

var _lruCache = _interopRequireDefault(require("lru-cache"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

// This code is based on the `mssql-jdbc` library published under the conditions of MIT license.
// Copyright (c) 2019 Microsoft Corporation
const cache = new _lruCache.default(0);

const getKey = async (keyInfo, options) => {
  if (!options.trustedServerNameAE) {
    throw new Error('Server name should not be null in getKey');
  }

  const serverName = options.trustedServerNameAE;
  const keyLookupValue = `${serverName}:${Buffer.from(keyInfo.encryptedKey).toString('base64')}:${keyInfo.keyStoreName}`;

  if (cache.has(keyLookupValue)) {
    return cache.get(keyLookupValue);
  } else {
    const provider = options.encryptionKeyStoreProviders && options.encryptionKeyStoreProviders[keyInfo.keyStoreName];

    if (!provider) {
      throw new Error(`Failed to decrypt a column encryption key. Invalid key store provider name: ${keyInfo.keyStoreName}. A key store provider name must denote either a system key store provider or a registered custom key store provider. Valid (currently registered) custom key store provider names are: ${options.encryptionKeyStoreProviders}. Please verify key store provider information in column master key definitions in the database, and verify all custom key store providers used in your application are registered properly.`);
    }

    const plaintextKey = await provider.decryptColumnEncryptionKey(keyInfo.keyPath, keyInfo.algorithmName, keyInfo.encryptedKey);
    const encryptionKey = new _symmetricKey.default(plaintextKey);

    if (options.columnEncryptionKeyCacheTTL > 0) {
      cache.set(keyLookupValue, encryptionKey, options.columnEncryptionKeyCacheTTL);
    }

    return encryptionKey;
  }
};

exports.getKey = getKey;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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