"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AeadAes256CbcHmac256Algorithm = exports.algorithmName = void 0;

var _types = require("./types");

var _crypto = require("crypto");

var _aeadAes256CbcHmacEncryptionKey = require("./aead-aes-256-cbc-hmac-encryption-key");

// This code is based on the `mssql-jdbc` library published under the conditions of MIT license.
// Copyright (c) 2019 Microsoft Corporation
const algorithmName = 'AEAD_AES_256_CBC_HMAC_SHA256';
exports.algorithmName = algorithmName;
const algorithmVersion = 0x1;
const blockSizeInBytes = 16;

class AeadAes256CbcHmac256Algorithm {
  constructor(columnEncryptionKey, encryptionType) {
    this.columnEncryptionkey = void 0;
    this.isDeterministic = void 0;
    this.keySizeInBytes = void 0;
    this.version = void 0;
    this.versionSize = void 0;
    this.minimumCipherTextLengthInBytesNoAuthenticationTag = void 0;
    this.minimumCipherTextLengthInBytesWithAuthenticationTag = void 0;
    this.keySizeInBytes = _aeadAes256CbcHmacEncryptionKey.keySize / 8;
    this.version = Buffer.from([algorithmVersion]);
    this.versionSize = Buffer.from([1]);
    this.minimumCipherTextLengthInBytesNoAuthenticationTag = 1 + blockSizeInBytes + blockSizeInBytes;
    this.minimumCipherTextLengthInBytesWithAuthenticationTag = this.minimumCipherTextLengthInBytesNoAuthenticationTag + this.keySizeInBytes;
    this.columnEncryptionkey = columnEncryptionKey;
    this.isDeterministic = encryptionType === _types.SQLServerEncryptionType.Deterministic;
  }

  encryptData(plaintText) {
    let iv;

    if (this.isDeterministic === true) {
      const hmacIv = (0, _crypto.createHmac)('sha256', this.columnEncryptionkey.getIvKey());
      hmacIv.update(plaintText);
      iv = hmacIv.digest().slice(0, blockSizeInBytes);
    } else {
      iv = (0, _crypto.randomBytes)(blockSizeInBytes);
    }

    const encryptCipher = (0, _crypto.createCipheriv)('aes-256-cbc', this.columnEncryptionkey.getEncryptionKey(), iv);
    const encryptedBuffer = Buffer.concat([encryptCipher.update(plaintText), encryptCipher.final()]);

    const authenticationTag = this._prepareAuthenticationTag(iv, encryptedBuffer, 0, encryptedBuffer.length);

    return Buffer.concat([Buffer.from([algorithmVersion]), authenticationTag, iv, encryptedBuffer]);
  }

  decryptData(cipherText) {
    const iv = Buffer.alloc(blockSizeInBytes);
    const minimumCiperTextLength = this.minimumCipherTextLengthInBytesWithAuthenticationTag;

    if (cipherText.length < minimumCiperTextLength) {
      throw new Error(`Specified ciphertext has an invalid size of ${cipherText.length} bytes, which is below the minimum ${minimumCiperTextLength} bytes required for decryption.`);
    }

    let startIndex = 0;

    if (cipherText[0] !== algorithmVersion) {
      throw new Error(`The specified ciphertext's encryption algorithm version ${Buffer.from([cipherText[0]]).toString('hex')} does not match the expected encryption algorithm version ${algorithmVersion}.`);
    }

    startIndex += 1;
    let authenticationTagOffset = 0;
    authenticationTagOffset = startIndex;
    startIndex += this.keySizeInBytes;
    cipherText.copy(iv, 0, startIndex, startIndex + iv.length);
    startIndex += iv.length;
    const cipherTextOffset = startIndex;
    const cipherTextCount = cipherText.length - startIndex;

    const authenticationTag = this._prepareAuthenticationTag(iv, cipherText, cipherTextOffset, cipherTextCount);

    if (0 !== authenticationTag.compare(cipherText, authenticationTagOffset, Math.min(authenticationTagOffset + cipherTextCount, authenticationTagOffset + authenticationTag.length), 0, Math.min(cipherTextCount, authenticationTag.length))) {
      throw new Error('Specified ciphertext has an invalid authentication tag.');
    }

    let plainText;
    const decipher = (0, _crypto.createDecipheriv)('aes-256-cbc', this.columnEncryptionkey.getEncryptionKey(), iv);

    try {
      plainText = decipher.update(cipherText.slice(cipherTextOffset, cipherTextOffset + cipherTextCount));
      plainText = Buffer.concat([plainText, decipher.final()]);
    } catch (error) {
      throw new Error(`Internal error while decryption: ${error.message}`);
    }

    return plainText;
  }

  _prepareAuthenticationTag(iv, cipherText, offset, length) {
    const hmac = (0, _crypto.createHmac)('sha256', this.columnEncryptionkey.getMacKey());
    hmac.update(this.version);
    hmac.update(iv);
    hmac.update(cipherText.slice(offset, offset + length));
    hmac.update(this.versionSize);
    return hmac.digest();
  }

}

exports.AeadAes256CbcHmac256Algorithm = AeadAes256CbcHmac256Algorithm;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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