/**************************************************************************
**                                                                        *
**  FILE        :  wcstok.c                                               *
**                                                                        *
**  DESCRIPTION :  Source file for wcstok routine.                        *
**                                                                        *
**  Copyright 1996-2009 Altium BV                                         *
**                                                                        *
**************************************************************************/

#include <wchar.h>

#ifdef WCHAR_SUPPORT_ENABLED

/*
 * A sequence of calls to the wcstok function breaks the wide string
 * pointed to by s1 into a sequence of tokens, each of which is delimited
 * by a wide character from the wide string pointed to by s2. The third
 * argument points to a caller-provided wchar_t pointer into which the
 * wcstok function stores information necessary for it to continue scanning
 * the same wide string.
 *
 * For the first call in the sequence, s1 shall point to a wide string,
 * while in subsequent calls for the same string, s1 shall be a null
 * pointer.
 *
 * Return a pointer to the first wide character of a token, or return a
 * null pointer if there is no token.
 */
wchar_t*
wcstok( wchar_t* restrict s1, const wchar_t* restrict s2,
        wchar_t** restrict ptr)
{
        wchar_t* ptk = NULL;
        wchar_t* st;

        if ( s1 )
        {
                st = s1;
        }
        else if ( ptr && *ptr )
        {
                st = *ptr;
        }
        else
        {
                st = NULL;
        }

        if ( s2 && st )
        {
                /* Locate the first wide character in st not contained in s2.
                 */
                for ( ; *st; st++ )
                {
                        if ( !wcschr( s2, *st ) )
                        {
                                ptk = st;                       /* Begin of token. */
                                break;
                        }
                }

                if ( ptk )
                {
                        *ptr = NULL;

                        /* Locate the first wide character in st contained in s2.
                         */
                        for ( ; *st; st++ )
                        {
                                if ( wcschr( s2, *st ) )
                                {
                                        *st  = (wchar_t) 0;     /* End of token. */
                                        *ptr = st + 1;
                                        break;
                                }
                        }
                }
        }

        return ptk;
}

#endif
