/**************************************************************************
**                                                                        *
**  FILE        :  setvbuf.c                                              *
**                                                                        *
**  DESCRIPTION :  Source file for setvbuf() routine                      *
**                 Adds a buffer to the given stream                      *
**                                                                        *
**  Copyright 1996-2009 Altium BV                                         *
**                                                                        *
**************************************************************************/

#include <stdio.h>
#include <stdlib.h>

/*
 * When setvbuf is referenced, pull in the free function so that weak
 * reference _dofree in fclose is filled in.
 */
#pragma extern  free

int setvbuf(FILE * restrict fp, char * restrict buf, int mode, size_t size)
{
        int             ret;
        _iob_flag_t     flag;

        switch (mode)
        {
        case _IONBF:
                break;
        case _IOLBF:
        case _IOFBF:
                /*
                 * The standard is not clear about the validity of:
                 *
                 *      setvbuf(f, buf, _IOLBF, 0);
                 *
                 * but instead of clearing buf and following the malloc(0)
                 * failure path we bail out early when zero sized buffering
                 * is requested.
                 */
                if (size)
                {
                        break;
                }
                /* fall through */
        default:
                return 1;
        }

        flag = fp->_flag;
        if (flag & _IOMYBUF)
        {
                free(fp->_base);        /* point of no return */
        }
        flag = (flag & ~(_IOMYBUF | _IOBUF)) | (_iob_flag_t)mode;

        /*
         * Allocate the buffer if requested and update flag accordingly.
         */
        ret = 0;
        if (flag & _IONBF)
        {
                buf = NULL;
        }
        else if (buf == NULL)
        {
                buf = malloc(size);
                if (buf)
                {
                        flag |= _IOMYBUF;
                }
                else
                {
                        /*
                         * uh oh, malloc failed: must revert to unbuffered.
                         */
                        flag = (flag & ~_IOBUF) | _IONBF;
                        ret = 1;
                }
        }

        /*
         * Commit and return.
         */
        fp->_base = buf;
        fp->_ptr = buf;
        fp->_bufsiz = buf ? (int)size : EOF;
        fp->_flag = flag;
        /* fp->_cnt is still zero after fopen()/freopen() */
        return ret;
}
