/**************************************************************************
**
**  FILE        :  static_init.c
**
**  DESCRIPTION :  called by _main to handle calling of static constructors
**                 and destructors
**
**  COPYRIGHT   :  Copyright 1998-2009 Altium BV
**
**************************************************************************/

#include "basics.h"
#include "runtime.h"
#pragma hdrstop
#include "main.h"
#include "static_init.h"
#include "dtor_list.h"


void __call_dtors()
/*
Call functions to perform static destruction of objects.

This routine deals with three different means of destroying static
objects.  The primary means is by use of a list of needed destructions
that is built as objects are constructed.  To allow "new" object
files that use the needed destruction list to be linked with
older object files, the "munch" style destruction lists are also
supported.  If a file contains both a list of needed destructions
and a "munch" list, the "munch" list is processed first, followed
by the needed destruction list.  This ensures that any new needed
destructions created while processing the "munch" destructions will
be handled.
*/
{
#ifndef __IA64_ABI
  static a_boolean      dtors_done = FALSE;
  int                   pos = 0;

  if (!dtors_done) {
    dtors_done = TRUE;
    /* Using munch information.  Call the destructors in the _dtors
       array in reverse order. */
    while (_dtors[pos]) pos++;
    while (pos--) (_dtors[pos])();
  }  /* if */
  /* Do the destructions specified by the needed destructions list.
     This is not done unconditionally (i.e., not based on dtors_done),
     because when using EH it is actually possible for exit to be
     called more than once, and there is no ill effect of calling
     __process_needed_destructions multiple times.  */
  __process_needed_destructions();
#else /* defined(__IA64_ABI) */
  ABI_NAMESPACE::__cxa_finalize((a_dso_handle)NULL);
#endif /* defined(__IA64_ABI) */
}  /* __call_dtors */


#if USE_ATEXIT == 0
/* Declare low-level routine or system call that exits a process. */
extern "C" void _exit(int status);


void exit(int status)
/*
This is used on systems that don't support a protocol such as the 
ANSI C "atexit" or the SunOS "on_exit" functions.  This version of exit
should be used instead of the normal OS exit routine.  Which means that
the library containing this routine needs to be linked in ahead of
the standard C library (such as libc.a).  Using this version of exit
means that some processing done by special versions of exit (such as the
profiling version) will not be done.
*/
{
  __call_dtors();
  _exit(status);
}  /* exit */
#endif /* USE_ATEXIT == 0 */


/*
The kind of function pointer accepted by atexit and on_exit.  Note
that the WP has a problem with the definition of atexit (as of 7/96),
because it is unspecified whether atexit has C or C++ linkage.  With
the recent change in which linkage is now part of a routine's type,
that means it is unspecified whether the argument passed to atexit
must have C or C++ linkage.
*/
extern "C" typedef void (*void_c_function_ptr)();

#if USE_ATEXIT == 0

/* Used to register a function to be called by exit to do wrapup
   processing. */
extern "C" void on_exit(void_c_function_ptr, char *);

#else /* USE_ATEXIT != 0 */

/* Used to register a function to be called by exit to do wrapup
   processing. */
extern "C" int atexit(void_c_function_ptr);

#endif /* USE_ATEXIT */

void __register_finalization_routine(void)
/*
Register the function that handles static destruction so that it will be
called when the program exits.
*/
{
#if USE_ATEXIT
  atexit((void_c_function_ptr)__call_dtors);
#endif /* USE_ATEXIT */
}  /* __register_finalization_rotuine */

#ifndef __TASKING_NO_CTORS
void __call_ctors()
/*
Call functions to perform static construction of objects.  This routine
first determines whether the executable as been processed using the
"munch" utility and then uses the appropriate method to call the static
initializer functions.
*/
{
  int   pos = 0;
  /* Establish that the termination routines should be called when exit()
     is called or when main() returns normally.  This needs to be done
     before any of the constructor routines are called so that if
     exit is called during static initialization, any constructed objects
     will be destroyed. */
  __register_finalization_routine();
  /* Using munch information.  Call the constructors in the _ctors
     array. */
  while (_ctors[pos]) (*_ctors[pos++])();
}  /* __call_ctors */
#endif /* __TASKING_NO_CTORS */
