/**************************************************************************
**                                                                        *
**  FILE        :  frexp.c                                                *
**                                                                        *
**  DESCRIPTION :  frexp implementation for IEEE-754 double precision     *
**                 floating point format.                                 *
**                                                                        *
**  Copyright 1996-2005 Altium BV                                         *
**                                                                        *
**************************************************************************/

#include <float.h>
#include <math.h>

/*****************************************************************************
|*
|* FUNCTION:           frexp
|*
|* AVAILABILITY:       EXTERNAL
|*
|* PARAMETERS:
|*
|*     Double precision floating point value (to split into its mantissa and
|*     signed unbiased exponent) and a pointer to the exponent value.
|*
|* RETURN VALUE:
|*
|*     The fraction of the passed variable.
|*
|* DESCRIPTION:
|*
|*     frexp implementation for IEEE-754 single precision/double precision floating point format.
|*     See ANSI-C X3J11 88-001, standard math library definition on frexp (Paragraph 4.5.4.2).
|*     For details see frexpf.c.
|*
*/
#ifndef __SINGLE_FP__
#pragma alias   frexpl  = _frexp_dpf
#pragma alias   frexp   = _frexp_dpf
#endif

double  _frexp_dpf( double d, int *exponent )
{
        double2longs_t  u;
        unsigned long   lo;
        unsigned long   hi;
        int             exp;
        unsigned long   mantissa_hi;

        /* this assumes that the double internally has the *same* endianess as the long */
        u.d     = d;
        hi      = u.s.hi;
        lo      = u.s.lo;

        exp     = GET_DOUBLE_EXPONENT( hi );

        /* test the mantissa == 0 in case denormal values are passed -> denormals are not tested... */
        mantissa_hi     = GET_DOUBLE_HI_MANTISSA( hi );
        if ( exp == 0 && mantissa_hi == 0 && GET_DOUBLE_LO_MANTISSA( lo ) == 0 )
        {
                /* got zero, return that */
                *exponent       = 0;
                return u.d;             
        }

        *exponent       = exp - DOUBLE_BIAS + 1;        /* the unbiased exponent of the returned value */

        /*
         * Now convert the passed variable from 1.0 <= |mantissa| < 2.0 to 0.5 <= |mantissa| < 1.0.
         * The mantissa stays the same but its new exponent is loaded with dp_bias-1. (1.0 * 2^-1 = 0.5)
        */
        exp     = DOUBLE_BIAS - 1;
        hi      = PUT_DOUBLE_EXPONENT( STRIP_DOUBLE_EXPONENT( hi ), exp );
        u.s.hi  = hi;

        return u.d;
}
