/**************************************************************************
**                                                                        *
**  FILE        :  exp.c                                                  *
**                                                                        *
**  DESCRIPTION :  Source file for exp() routine                          *
**                                                                        *
**       The double coefficients are #1067 from Hart and Cheney. (18.08D) *
**       The single coefficients are #1063 from Hart and Cheney. (10.03D) *
**                                                                        *
**  Copyright 1996-2005 Altium BV                                         *
**                                                                        *
**************************************************************************/

/*
        exp returns the exponential function of its
        floating-point argument.

*/

#include <errno.h>
#include <math.h>

#ifdef __SINGLE_FP__
#pragma alias   expl    = _exp_spf
#pragma alias   exp             = _exp_spf
#endif
#pragma alias   expf    = _exp_spf


#define MAXF    89.0
#define P0      0.72152891511493e1f
#define P1      0.576900723731e-1f
#define Q0      0.208189237930062e2f
#define POLY1(fract,xsq)        ( ( P1 * xsq + P0 ) * fract )
#define POLY2(xsq)              ( xsq + Q0 )

#define LOG2E   1.4426950408889634073599247f
#define SQRT2   1.4142135623730950488016887f

static
float
_exp_spf( float arg )
{
        float   fract;
        float   xsq;
        int     ent;

        if( arg == 0.0f )
                return( 1.0f );
        if( arg < -MAXF )
                return( 0.0f );
        if( arg > MAXF )
        {
                errno = ERANGE;
                return( HUGE_VALF );
        }
        arg *= LOG2E;
        ent = floorf( arg );
        fract = ( arg - ent ) - 0.5f;
        xsq = fract * fract;

        fract = POLY1(fract,xsq);
        xsq = POLY2(xsq);
        return( ldexpf( SQRT2 * ( xsq + fract ) / ( xsq - fract ), ent ) );
}


