
/**************************************************************************
**                                                                        *
**  FILE        :  tan.c                                                  *
**                                                                        *
**  DESCRIPTION :  Source file for tan() routine                          *
**                 Computes the 'tangent' for the given angle (in radians)*
**                                                                        *
**      Coefficients double are #4285 from Hart & Cheney. (19.74D)        *
**      Coefficients single are #4282 from Hart & Cheney. ( 7.85D)        *
**                                                                        *
**  Copyright 1996-2009 Altium BV                                         *
**                                                                        *
**************************************************************************/

#include <math.h>
#include <errno.h>
#include <limits.h>

#ifdef __SINGLE_FP__
#pragma alias   tanl    = _tan_spf
#pragma alias   tan             = _tan_spf
#endif
#pragma alias   tanf    = _tan_spf


#define INVPI    ( 1.27323954473516268f)      /*   ( 4 / PI )   */
#define P0       ( 0.21242445758263e3f)
#define P1       (-0.1255329742424e2f)
#define Q0       ( 0.27046722349399e3f)
#define Q1       (-0.7159606050466e2f)
#define POLYNOMIAL(x,arg)       ( ( ( P1 * (arg) + P0 ) * (x) ) / ( ( (arg) + Q1 ) * (arg) + Q0 ) )

static
float
_tan_spf( float arg )
{
        float   x;
        signed char     sign;
        char            flag;
        int             quad;

        flag = 0;
        sign = 1;
        if( arg < 0.0f )
        {
                arg = -arg;
                sign = -1;
        }
        arg = arg * INVPI;   /* overflow? */
        x = modff( arg, &arg ); /* From here we don't need the value of 'arg' anymore */
                                /* So we use 'arg' for other purposes */
        if (arg > (float)INT_MAX)
        {       
                quad = fmodf(arg, 4);
        }
        else
        {
                quad = (int) arg % 4;
        }
        switch( quad )
        {
        case 1:
                x = 1.0f - x;
                flag = 1;
                break;
        case 2:
                sign = -sign;
                flag = 1;
                break;
        case 3:
                x = 1.0f - x;
                sign = -sign;
                break;
        }

        arg = x * x;
        arg = POLYNOMIAL(x,arg);

        if( flag == 1 )
        {
                if( arg == 0.0f )
                {
                        errno = ERANGE;
                        if ( sign > 0 )
                                return( HUGE_VALF );
                        return( -HUGE_VALF );
                }
                arg = 1.0f / arg;
        }

        if( sign == 1 )
                return( arg );
        else
                return( -arg );
}


