
/**************************************************************************
**                                                                        *
**  FILE        :  sinh.c                                                 *
**                                                                        *
**  DESCRIPTION :  Source file for sinh() routine                         *
**                 Computes the 'hyperbolic sinus' for the given angle    *
**                 (in radians)                                           *
**                                                                        *
**  Copyright 1996-2009 Altium BV                                         *
**                                                                        *
**************************************************************************/

#include <math.h>
/*
        sinh(arg) returns the hyperbolic sine of its floating-
        point argument.

        The exponential function is called for arguments
        greater in magnitude than 0.5.

        A series is used for arguments smaller in magnitude than 0.5.
        The double coefficients are #2027 from Hart & Cheney. (16.45D)
        The single coefficients are #2021 from Hart & Cheney. ( 6.98D)

        cosh(arg) is computed from the exponential function for
        all arguments.
*/

#ifdef __SINGLE_FP__
#pragma alias   sinhl   = _sinh_spf
#pragma alias   sinh            = _sinh_spf
#endif
#pragma alias   sinhf   = _sinh_spf


#define P0  (-0.2019614849422e2f)
#define P1  (-0.236617830421e1f)
#define Q0  (-0.2019615061071e2f)
#define POLYNOMIAL(argsq,arg)   ( ( ( P1 * (argsq) + P0 ) * (arg) ) / ( (argsq) + Q0 ) )
#define LOG_HALF (-0.693147180559945309417232121f) /* ln(0.5) */

static
float
_sinh_spf( float arg )
{
        float temp, argsq;
        register signed char sign = 0;

        sign = 1;
        if( arg < 0.0f )
        {
                arg = -arg;
                sign = -1;
        }

        if( arg > 37.0f )
        {
                temp = expf( arg + LOG_HALF );
                        
                if ( sign == 1 )
                        return( temp );
                else
                        return( -temp );
        }

        if( arg > 0.5f )
        {
                return( sign * ( expf( arg + LOG_HALF ) - expf( -arg + LOG_HALF ) ) );
        }

        argsq = arg*arg;
        temp = POLYNOMIAL(argsq,arg);
        return( sign * temp );
}


