
/**************************************************************************
**                                                                        *
**  FILE        :  runtime-malloc.c                                       *
**                                                                        *
**  DESCRIPTION :  Wrappers for malloc/free with run-time checking,       *
**                 selected by the --runtime=m option.                    *
**                                                                        *
**  Copyright 1996-2009 Altium BV                                         *
**                                                                        *
**************************************************************************/

#include <stdlib.h>
#include <stddef.h>
#include <string.h>
#include "malloc-error.h"
#include "runtime-error.h"

#pragma alias   __runtime_malloc = malloc
#pragma alias   _dofree          = free

/*
 *      Next values must be odd. Malloc alignment is always a multiple
 *      of ALIGN, which is at least 2.
 */
#define AMAGIC  ((size_t) 0xdcdccdcd)   /* magic word for allocated blocks */
#define FMAGIC  ((size_t) 0xfcfccfcf)   /* magic word for free blocks */
#define TRAILER ((char) 0x99)
#define MALLPAT ((char) 0xdd)
#define FREEPAT ((char) 0xbb)           /* must be != TRAILER */

typedef struct  /* sizeof(hdr_t) must be a multiple of the malloc alignment! */
{
        size_t  size;
        size_t  magic;
}               hdr_t;

struct  hdr_align
{
        char    c;
        hdr_t   h;
};

#define HALIGN  offsetof(struct hdr_align, h) 

#define FREED   4                       /* must be power of 2 */
#define FLIMIT  4096                    /* limit for freed memory check */

static  void*           freed[FREED];
static  int             freed_count;
static  const char      freepat [] = { FREEPAT, 0 };

/* --------------------------------------------------------------------- */

static  int     check ( void* mem )
{
        hdr_t*  h;
        char*   p;
        char    mempat = (char)(unsigned long int)mem;

        if      ( mempat == MALLPAT )
        {
                __malloc_error( NULL, 0, "calling free with a pointer from uninitialized memory" );
                return 1;
        }
        if      ( mempat == FREEPAT )
        {
                __malloc_error( NULL, 0, "calling free with a pointer from deallocated memory" );
                return 1;
        }
        if      ( (int) (unsigned long) mem & (HALIGN-1) )
        {
                __malloc_error( mem, 0, "calling free with an unaligned pointer" );
                return 1;
        }
        h = (hdr_t*) mem - 1;
        switch  ( h->magic )
        {
        case AMAGIC:
                p = mem;
                if      ( p[h->size] != TRAILER )
                {
                        __malloc_error( mem, h->size, "malloc buffer overflow" );
                        p[h->size] = TRAILER;                   /* repair */
                }
                break;
        case FMAGIC:
                if      ( h->size > 0 && h->size <= FLIMIT )
                {
                        p = mem;
                        p[h->size] = TRAILER;
                        if      ( strspn( mem, freepat ) != h->size )
                        {
                                __malloc_error( mem, h->size, "freed memory was modified" );
                                memset( mem, FREEPAT, h->size ); /* repair */
                        }
                }
                break;
        default:
                __malloc_error( mem, 0, "calling free with an invalid pointer" );
                return 1;
        }
        return 0;
}

/* --------------------------------------------------------------------- */

extern  void*   malloc ( size_t size )
{
        hdr_t*  h;
        int     i;

        SAVE_CALLER;
        for     ( i = 0; i < FREED; i++ )
        {
                if      ( freed[i] )
                {
                        (void) check( freed[i] );
                        freed[i] = NULL;
                }
        }
        CLEAR_CALLER;
        if      ( size == 0 )
        {
                return NULL;
        }
        h = _malloc( sizeof(hdr_t) + size + 1 );/* allocate */
        if      ( h )
        {
                h->size  = size;
                h->magic = AMAGIC;              /* mark block as allocated */
                h++;
                memset(h, MALLPAT, size);       /* make it non-zero */
                *((char*) h + size) = TRAILER;  /* place sentinel */
        }
        return h;
}

extern  void    free ( void* mem )
{
        hdr_t*  h;

        SAVE_CALLER;
        if      ( mem && check( mem ) == 0 )    /* check */
        {
                h = (hdr_t*) mem - 1;
                if      ( h->magic == FMAGIC )
                {
                        __malloc_error( mem, h->size, "memory already freed" );
                }
                else
                {
                        h->magic = FMAGIC;              /* mark block as free */
                        memset( mem, FREEPAT, h->size );/* erase contents */
                        _free( h );                     /* deallocate */
                        freed[freed_count++ & (FREED-1)] = mem;
                }
        }
        CLEAR_CALLER;
}

extern  void*   realloc ( void* mem, size_t newsize )
{
        void*   newmem;
        hdr_t*  h;

        SAVE_CALLER;
        newmem = malloc( newsize );
        if      ( mem && check( mem ) == 0 )
        {
                if      ( newmem )              /* implies (newsize != 0) */
                {
                        h = (hdr_t*) mem - 1;
                        if      ( h->size < newsize )
                        {
                                newsize = h->size;
                        }
                        (void) memcpy( newmem, mem, newsize );
                }
                if      ( newmem || newsize == 0 )
                {
                        free( mem );
                }
        }
        CLEAR_CALLER;
        return newmem;
}
