
/**************************************************************************
**                                                                        *
**  FILE        :  modff.c                                                *
**                                                                        *
**  DESCRIPTION :  modf implementation for IEEE-754 single precision      *
**                 floating point format.                                 *
**                                                                        *
**  Copyright 1996-2009 Altium BV                                         *
**                                                                        *
**************************************************************************/

#include <float.h>
#include <math.h>
#include "fpbits.h"

/*****************************************************************************
|*
|* FUNCTION:           modf
|*
|* AVAILABILITY:       EXTERNAL
|*
|* PARAMETERS:
|*
|*     Single precision floating point value to split
|*     and a pointer to the integer part to determine.
|*
|* RETURN VALUE:
|*
|*     The fraction of the passed variable.
|*
|* DESCRIPTION:
|*
|*     modf implementation for IEEE-754 single precision floating point format.
|*
|*     No need for error handling since it can not occur in this function.
|*     See ANSI-C X3J11 88-001, standard math library definition on modf (Paragraph 4.5.4.6).
|*
|*     Split a single precision floating point value into an integer and a
|*     fraction. The abs(fraction) will be less than one and of the same type
|*     as the passed variable has. The integer value will also be of the
|*     passed variable's type.
|*     Both parts will have the sign of the argument.
|*
|*     The two parts should together still be representing the passed variable's
|*     value, so passed_value = integer_value + fraction.
|*     They both have the same sign as the argument.
|*
|*     If the argument is zero, zero will be returned.
*/
#ifdef __SINGLE_FP__
#pragma alias   modfl   = _modf_spf
#pragma alias   modf    = _modf_spf
#endif
#pragma alias   modff   = _modf_spf

float   _modf_spf( float f, float *intval )
{
        float2long_t    u;
        unsigned long   l;
        short           exp;

        /* this assumes that the float internally has the *same* endianess as the long */
        u.f     = f;
        l       = u.l;

        exp     = GET_FLOAT_EXPONENT( l );
        exp     -= FLOAT_BIAS;
        /* test if too small, to prevent too many shifts below */
        if ( exp < 0 )
        {
                /* integer is zero */
                *intval = copysignf( 0.0f, f );
                return f;
        }

        /*
         * for IEEE-754 single precision, an (unbiased!) exponent >= FLOAT_FRACTION_SIZE tells us
         * that the fraction == 0.0
         * So check if the value EXCEEDS +-1 * 2 ^ 22
        */
        if ( exp >= FLOAT_FRACTION_SIZE )
        {
                /* so the fraction is zero */
                *intval = f;
                return copysignf( 0.0f, f );
        }

        /* here the float is <-1*2^FLOAT_FRACTION_SIZE..+1*2^FLOAT_FRACTION_SIZE> */
        /* now clear the least significant bits, which would drop off at the integer */
        /* that are the last FLOAT_FRACTION_SIZE - exp bits */
        l       &= 0xffffffff << (FLOAT_FRACTION_SIZE - exp);

        u.l     = l;

        *intval = u.f;
        return copysignf( f - *intval, f );
}
