
/**************************************************************************
**                                                                        *
**  FILE        :  malloc.c                                               *
**                                                                        *
**  DESCRIPTION :  Source file for malloc() and free()                    *
**                                                                        *
**  Copyright 1996-2009 Altium BV                                         *
**                                                                        *
**************************************************************************/

#include <stdlib.h>
#include "_malloc.h"

#pragma weak    malloc
#pragma alias   _malloc         = malloc
#pragma weak    free
#pragma alias   _free           = free
#pragma weak    _dofree
#pragma alias   _dofree         = free

/* -------------------- external definitions ------------------ */

extern  void*   _sbrk ( long );

#define NOMEM   ((char*) -1)    /* (char*)_sbrk() return value */

/* -------------------- variables ----------------------------- */

static  mptr_t  head;           /* head of memory block list */
static  mptr_t  start;          /* starting point for block search */

/* -------------------- debug --------------------------------- */

#ifdef  DEBUG

#include <io.h>                 /* _write() */
#include <string.h>             /* strlen() */

static  void    malloc_write ( char* line )
{
        _write( 2, line, strlen( line ) );
}

extern  void    malloc_dump ( void )
{
        mptr_t  p;
        msize_t s;
        char*   descr;
        char    line[80];

        if      ( ! start )
        {
                return;
        }
        malloc_write( "------------ malloc dump ------------\n" );
        for     ( p = head; ; p += (s & ~ 1) )
        {
                s = *(msize_t*)p;
                descr = "";
                if      ( s == 0 )
                {
                        descr = "(tail)";
                }
                else if ( ! (s & 1) )
                {
                        descr = "(free)";
                }
                snprintf( line, sizeof(line), "%p %6lu: %c %p  %s\n",
                        p, (unsigned long) (s & ~ 1), (p == start ? '*' : ' '),
                        p + HEADER, descr );
                malloc_write( line );
                if      ( s == 0 )
                {
                        break;
                }
        }
        p += HEADER;
        if      ( p != (char*)_sbrk( 0 ) )
        {
                snprintf( line, sizeof(line), "%p %6lu: gap\n",
                        p, (unsigned long) ((char*)_sbrk( 0 ) - p) );
                malloc_write( line );
        }
        malloc_write( "-------------------------------------\n" );
}

#endif

/* -------------------- global functions ---------------------- */

extern  void*   malloc ( size_t size )
{
        mptr_t  p;
        mptr_t  next;
        mptr_t  mem;
        msize_t s;
        msize_t s2;
        int     align;

        if      ( size == 0 )
        {
                return NULL;
        }
        size = HEADER + ALIGNED(size);
        if      ( ! start )
        {       /* initialize on first call */
                /* the break may be unaligned; calculate alignment */
                align = (- (int) (long) (char*)_sbrk( 0 )) & (ALIGN-1);
                mem = (char*)_sbrk( align + HEADER );
                if      ( mem == NOMEM )
                {
                        return NULL;
                }
                head = start = mem + align;
                *(msize_t*)head = 0;    /* end marker */
        }
        while   ( (s = *(msize_t*)start) && (s & 1) )
        {
                start += s - 1;
        }
        for     ( p = start
                ; (s = *(msize_t*)p) && ((s & 1) || size > s)
                ; p = next
                )
        {
                next = p + (s & ~ 1);
                if      ( ! (s & 1) )
                {       /* free block */
                        s2 = *(msize_t*)next;
                        if      ( s2 == 0 )
                        {       /* last free block is too small, so remove it */
                                if      (  (char*)_sbrk( 0 ) == next + HEADER
                                        && (char*)_sbrk( - (long) s ) != NOMEM
                                        )
                                {
                                        *(msize_t*)p = 0; /* new end marker */
                                        s = 0;
                                        break;
                                }
                        }
                        else if ( ! (s2 & 1) )
                        {       /* merge two consecutive free blocks */
                                *(msize_t*)p = s + s2;
                                next = p;
                        }
                }
        }
        if      ( s == 0 )
        {       /* nothing found, so allocate more memory */
                mem = (char*)_sbrk( size );
                if      ( mem == NOMEM )
                {
                        return NULL;
                }
                if      ( mem < p + HEADER )
                {       /* someone called (char*)_sbrk() with a negative value! */
                        return NULL;
                }
                if      ( mem > p + HEADER )
                {       /* (char*)_sbrk() was called: encapsulate the memory */
                        /* the break may be unaligned; calculate alignment */
                        align = (- (int) (long) mem) & (ALIGN-1);
                        if      ( (char*)_sbrk( align + HEADER ) == NOMEM )
                        {
                                return NULL;
                        }
                        mem += align;
                        *(msize_t*)p = (mem - p) | 1; /* mark as allocated */
                        p = mem;
                }
                *(msize_t*)(p + size) = 0;      /* new end marker */
        }
        else if ( s >= size + HEADER )
        {       /* the free block is big enough to split */
                next = p + size;
                *(msize_t*)next = s - size;
        }
        else
        {       /* use complete block */
                size = (size_t) s;
        }
        *(msize_t*)p = size | 1;                /* mark block as allocated */
        return (void*) (p + HEADER);
}

extern  void    free ( void* mem )
{
        mptr_t  p;
        mptr_t  p2;
        msize_t s;
        msize_t s2;

        if      ( ! mem )
        {
                return;
        }
        p = (mptr_t)mem - HEADER;
        s = *(msize_t*)p & ~ 1;                 /* mark block as free */
        /*
         * try to merge block with successor(s)
         */
        for     ( ;; )
        {
                *(msize_t*)p = s;
                p2 = p + s;
                s2 = *(msize_t*)p2;
                if      ( ! s2 || (s2 & 1) )
                {
                        break;
                }
                s += s2;                        /* merge free blocks */
        }
        /*
         * try to merge with first free block
         */
        if      ( start + *(msize_t*)start == p )
        {       /* start points to a preceding free block */
                s += *(msize_t*)start;
                *(msize_t*)start = s;           /* merge free blocks */
                p = start;
        }
        /*
         * try to reduce core
         */
        if      (  s2 == 0
                && (char*)_sbrk( 0 ) == p2 + HEADER
                && (char*)_sbrk( - (long) s ) != NOMEM
                )
        {
                *(msize_t*)p = 0;               /* new end marker */
        }
        /*
         * update 'start'
         */
        if      ( p < start )
        {
                start = p;
        }
}
