
/**************************************************************************
**                                                                        *
**  FILE        :  ldexp.c                                                *
**                                                                        *
**  DESCRIPTION :  ldexp implementation for IEEE-754 double precision     *
**                 precision floating point format.                       *
**                 Also includes generic implementation.                  *
**                 Returns x * (2 to the power n)                         *
**                                                                        *
**  Copyright 1996-2009 Altium BV                                         *
**                                                                        *
**************************************************************************/

#include <errno.h>
#include <float.h>
#include <math.h>
#include "fpbits.h"

#if _IEEE_754_FORMAT
#ifndef __SINGLE_FP__
#pragma alias   ldexpl  = _ldexp_dpf
#pragma alias   ldexp   = _ldexp_dpf
#endif

double
_ldexp_dpf( double fr, int pwr )
{
        double2longs_t  u;
        unsigned long   hi;
        signed int      exp;

        /* this assumes that the double internally has the *same* endianess as the long */
        u.d     = fr;
        hi      = u.s.hi;

        exp     = (int) GET_DOUBLE_EXPONENT( hi ) + pwr;
        /* under/overflow test */
        if ( pwr > 2 * DOUBLE_BIAS || exp > MAX_DOUBLE_BIASED_EXP )
        {
                errno   = ERANGE;
                return copysign( HUGE_VAL, fr );
        }
        if ( exp < MIN_DOUBLE_BIASED_EXP )
        {
                /* no ERANGE required, since that's implementation defined */
                return copysign( 0.0, fr );
        }

        hi      = PUT_DOUBLE_EXPONENT( STRIP_DOUBLE_EXPONENT( hi ), exp );
        u.s.hi  = hi;

        return u.d;
}

#else  /* _IEEE_754_FORMAT */
#ifdef  __SINGLE_FP__
#  define       BIAS    FLOAT_BIAS
#else
#  define       BIAS    DOUBLE_BIAS
#endif

double
ldexp( double fr, int pwr )
{
        int             exp;
        register char   neg = 0;

        neg = 0;
        if ( fr < 0.0 )
        {
                fr = -fr;
                neg = 1;
        }
        fr = frexp( fr, &exp );

        if ( pwr > BIAS || pwr + exp > BIAS  )
        {
                errno = ERANGE;

                if ( neg )
                        return( -HUGE_VAL );
                else
                        return( HUGE_VAL );
        }

        exp += pwr;

        if ( exp < -(BIAS-2) )
                return( 0 );

        while ( exp > 30 )
        {
                fr *= 1.073741824e9;    /* fr *= 1L << 30; */
                exp -= 30;
        }
        while ( exp < -30 )
        {
                fr /= 1.073741824e9;    /* fr /= 1L << 30; */
                exp += 30;
        }
        if ( exp > 0 )
                fr *= 1L << exp;
        if ( exp < 0 )
                fr /= 1L << -exp;
        if ( neg )
                fr = -fr;
        return( fr );
}
#endif /* _IEEE_754_FORMAT */
