
/**************************************************************************
**                                                                        *
**  FILE        :  dbg_posix.c                                            *
**                                                                        *
**  DESCRIPTION :  POSIX compliant debugger calls                         *
**                                                                        *
**  Copyright 1996-2009 Altium BV                                         *
**                                                                        *
**************************************************************************/

#include <unistd.h>             /* for most POSIX prototypes */
#include <fcntl.h>              /* for POSIX open() prototype */
#include <stdio.h>              /* for remove() and ISO/IEC rename() */

#include <dbg.h>

/*
 * _exit()
 *
 * This function may be called unconditionally in exit.c. To avoid pulling
 * in POSIX functions for filesystem operations which would cause name space
 * pollution the _exit() function has been moved to a separate file, see
 * dbg__exit.c.
 */

/*
 * clock()
 *
 * The clock() function is special: hardware may provide this
 * without getting inconsistencies with other debugger calls.
 * So, the function is kept in a separate file, see dbg_clock.c
 */

extern int open(const char *pathname, int flags, ... /* mode_t mode */)
{
        return _open(pathname, (_open_flags_t)flags);
}

extern ssize_t read(int fd, void *buffer, size_t nbytes)
{
        return _read((_fd_t)fd, buffer, nbytes);
}

extern ssize_t write(int fd, const void *buffer, size_t nbytes)
{
        return _write((_fd_t)fd, buffer, nbytes);
}

extern off_t lseek(int fd, off_t offset, int whence)
{
        return _lseek((_fd_t)fd, offset, (_whence_t)whence);
}

extern int close(int fd)
{
        return _close((_fd_t)fd);
}

#pragma weak rename             /* because it's also part of ISO/IEC 9899:1999 */
extern int rename(const char *oldpath, const char *newpath)
{
        _dbg_request.nr = _DBG_RENAME;
        _dbg_request.u.rename.oldpath = oldpath;
        _dbg_request.u.rename.newpath = newpath;
        _dbg_trap();
        return _dbg_request.u.rename_result;
}

/*
 * We provide a remove() because it should operate in the same file system
 * name space as all the other functions. Since it is part of ISO/IEC 9899:1999
 * we make it a weak symbol. Notice that almost every remove() implementation
 * refuses to remove a directory: this is slightly incorrect behavior.
 */
#pragma weak remove             /* because it's part of ISO/IEC 9899:1999 */
#pragma alias remove = unlink

extern int unlink(const char *pathname)
{
        _dbg_request.nr = _DBG_UNLINK;
        _dbg_request.u.unlink.pathname = pathname;
        _dbg_trap();
        return _dbg_request.u.unlink_result;
}

extern int access(const char *pathname, int mode)
{
        _dbg_request.nr = _DBG_ACCESS;
        _dbg_request.u.access.pathname = pathname;
        _dbg_request.u.access.mode = mode;
        _dbg_trap();
        return _dbg_request.u.access_result;
}

extern char *getcwd(char *buf, size_t size)
{
        _dbg_request.nr = _DBG_GETCWD;
        _dbg_request.u.getcwd.buf = buf;
        _dbg_request.u.getcwd.size = size;
        _dbg_trap();
        return _dbg_request.u.getcwd_result;
}

extern int chdir(const char *pathname)
{
        _dbg_request.nr = _DBG_CHDIR;
        _dbg_request.u.chdir.pathname = pathname;
        _dbg_trap();
        return _dbg_request.u.chdir_result;
}

extern int stat(const char *pathname, struct stat *buf)
{
        _dbg_request.nr = _DBG_STAT;
        _dbg_request.u.stat.pathname = pathname;
        _dbg_request.u.stat.buf = buf;
        _dbg_trap();
        return _dbg_request.u.stat_result;
}
