
/**************************************************************************
**                                                                        *
**  FILE        :  _wcstod.c                                              *
**                                                                        *
**  DESCRIPTION :  Source file for _wcstod() routine                      *
**                 This function is used by the wcstof()/wcstod() and     *
**                 wcstold() functions.                                   *
**                                                                        *
**  Copyright 1996-2009 Altium BV                                         *
**                                                                        *
**************************************************************************/

#include <float.h>
#include <errno.h>
#include <limits.h>
#include "fpbits.h"

#ifdef WCHAR_SUPPORT_ENABLED

#include <wchar.h>
#include <wctype.h>


/* Compare strings case insensitive
 */
extern int _wcsncasecmp( const wchar_t * s1, const wchar_t * s2, size_t n );
static long double _gethexvalue( const wchar_t* sc, wchar_t** ptr, unsigned char sign,
                                 int min_exponent, int max_exponent, long double huge_value );
static long double _getnan( const wchar_t* sc, wchar_t** ptr, unsigned char sign, int length );
static long double _getinf( const wchar_t* sc, wchar_t** ptr, unsigned char sign, int length );


long double
_wcstod( const wchar_t * restrict p,            /* input string */
         wchar_t ** restrict ptr,               /* next character in input string */
         int min_exponent,                      /* minimal exponent value (decimal) */
         int max_exponent,                      /* maximal exponent value (decimal) */
         long double min_mantissa,              /* minimal mantissa value */
         long double max_mantissa,              /* maximal mantissa value */
         long double huge_value                 /* HUGE_VAL (or equivalent) */
         )
{
        wint_t          wc;
        unsigned char   signm           = 0;    /* sign mantissa */
        unsigned int    numdigits       = 0;    /* number of mantissa digits */
        double          value           = 0.0;
        int             expm            = -1;
        int             expe            = 1;    /* double functionality: exponent and "error" flag */
        const wchar_t*  sc              = p;    /* in case something goes wrong */

        /* Skip whitespace
         */
        do
        {
                wc = *sc++;
        } while ( iswspace( wc ) );

        /* First check for sign
         */
        if ( wc == L'+' || wc == L'-' )
        {
                if ( wc == L'-' )
                {
                        signm = 1;
                }

                wc = *sc++;
        }

        /* Check for NAN, INF, INFINITY. Note that the "sc" pointer points to
         * the next character so we have to adjust the pointer.
         */
        sc--;
        if ( _wcsncasecmp( sc, L"nan", 3 ) == 0 )
        {
                return _getnan( sc, ptr, signm, 3 );
        }
        else if ( _wcsncasecmp( sc, L"infinity", 8 ) == 0 )
        {
                return _getinf( sc, ptr, signm, 8 );
        }
        else if ( _wcsncasecmp( sc, L"inf", 3 ) == 0 )
        {
                return _getinf( sc, ptr, signm, 3 );
        }
        else if ( _wcsncasecmp( sc, L"0x", 2 ) == 0 )
        {
                return _gethexvalue( sc + 2, ptr, signm, min_exponent,
                                     max_exponent, huge_value );
        }

        /* Advance to the next character again.
         */
        sc++;

        /* Remove leading zeros. This means a proper conversion can be made so
         * we set expe to zero.
         */
        if ( wc == L'0' )
        {
                expe = 0;
                while ( ( wc = *sc++ ) == L'0' )
                {
                        ;
                }
        }

        /* Scan the digits before the decimal point.
         */
        while ( iswdigit( wc ) )
        {
                /* avoid overflow */
                if ( numdigits < max_exponent )
                {
                        expm++;
                        value = value * 10.0 + (wc - L'0');
                        numdigits++;
                        expe = 0;
                }
                wc = *sc++;
        }

        /* Scan for the decimal point and any digits following it. Just a '.'
         * is not a valid conversion.
         */
        if ( wc == L'.' )
        {
                wc = *sc++;
                while ( wc == L'0' && expm < 0 )
                {
                        expm--;
                        expe = 0;
                        wc = *sc++;
                }

                while ( iswdigit( wc ) )
                {
                        /* avoid overflow */
                        if ( numdigits < max_exponent )
                        {
                                value = value * 10.0 + (wc - L'0');
                                numdigits++;
                                expe = 0;
                        }
                        wc = *sc++;
                }
        }

        /* Still no valid character found, no conversion can be done
         */
        if ( expe )
        {
                if ( ptr != NULL )
                {
                        *ptr = (wchar_t *)p;
                }

                return 0.0;
        }

        /* Check for exponent
         */
        if ( wc == L'e' || wc == L'E' )
        {
                unsigned char   signe   = 0;

                wc = *sc++;

                /* Check for sign
                 */
                if ( wc == L'-' || wc == L'+' )
                {
                        if ( wc == L'-' )
                        {
                                signe = 1;
                        }

                        wc = *sc++;
                }

                /* Read the exponent value
                 */
                while ( iswdigit( wc ) )
                {
                        if ( expe < ( INT_MAX / 10 ) )
                        {
                                expe = 10 * expe + (wc - L'0');
                        }

                        wc = *sc++;
                }

                if ( signe )
                {
                        expe = -expe;
                }
        }

        /* Set end pointer. Note that the last character is not used.
         */
        if ( ptr != NULL )
        {
                sc--;
                *ptr = (wchar_t *) sc;
        }

        /* Handle input 0.0
         */
        if ( numdigits == 0 )
        {
                if ( signm )
                {
                        value = -value;
                }

                return value;
        }

        /* Combine mantissa and exponent values
         */
        expe += expm;

        if ( expe >= max_exponent || expe <= min_exponent )
        {
                int             i;
                long double     minmaxmant      = max_mantissa;

                if ( expe <= min_exponent )
                {
                        minmaxmant = min_mantissa;
                }

                for ( i = 1; i < numdigits; i++ )
                {
                        minmaxmant *= 10.0;     /* relative intrinsic error: 0 */
                }

                if ( expe > max_exponent
                     ||
                     ( expe == max_exponent && value > minmaxmant ) )
                {
                        errno = ERANGE;
                        value = huge_value;
                        if ( signm )
                        {
                                value = -value;
                        }

                        return value;
                }

                if ( expe < min_exponent
                     ||
                     ( expe == min_exponent && value < minmaxmant ) )
                {
                        errno = ERANGE;
                        return 0.0;
                }
        }

        expe += 1 - numdigits;

        /* The values below are chosen to have a minimal intrinsic error and
         * to minimize the number of multiplications
         */
#ifndef __SINGLE_FP__
        while ( expe >= 64 )
        {
                value *= 1.0e64;        /* relative intrinsic error: 0 (IEEE DP) */
                expe -= 64;
        }
        while ( expe >= 8 )
        {
                value *= 1.0e8;         /* relative intrinsic error: 0 (IEEE DP) */
                expe -= 8;
        }
        while ( expe < -64 )
        {
                value *= 1.0e-64;       /* relative intrinsic error: -3e-17 (IEEE DP) */
                expe += 64;
        }
        while ( expe < -8 )
        {
                value *= 1.0e-8;        /* relative intrinsic error: 0 (IEEE DP) */
                expe += 8;
        }
#else
        while ( expe >= 7 )
        {
                value *= 1.0e7;         /* relative intrinsic error: 0 (IEEE SP) */
                expe -= 7;
        }
        while ( expe < -7 )
        {
                value *= 1.0e-7;        /* relative intrinsic error: +1.2e-8 (IEEE SP) */
                expe += 7;
        }
#endif
        while ( expe < 0 )
        {
                value *= 0.1;           /* relative intrinsic error: +1.5e-8 (IEEE SP) or +1.0e-16 (IEEE DP) */
                expe++;
        }
        while ( expe-- )
        {
                value *= 10.0;          /* relative intrinsic error: 0 */
        }

        if ( signm )                    /* Make negative when necessary */
        {
                value = -value;
        }

        return value;
}


static long double _gethexvalue( const wchar_t* sc,
                                 wchar_t**      ptr,
                                 unsigned char  sign,
                                 int            min_exponent,
                                 int            max_exponent,
                                 long double    huge_value
                                 )
{
        wint_t          wc;
        long double     value;
        int             dotbits         = 0;
        int             numbits         = 0;
        unsigned char   first           = 1;
        unsigned char   dotseen         = 0;
#ifndef __SINGLE_FP__
        int             bitcount        = DOUBLE_FRACTION_SIZE;
        double2longs_t  conv_value;
        unsigned long   lo_fract        = 0;
        unsigned long   hi_fract        = 0;
#else
        int             bitcount        = FLOAT_FRACTION_SIZE;
        float2long_t    conv_value;
        unsigned long   fraction        = 0;
#endif
        int             exponent        = 0;
        unsigned char   signed_exponent = 0;

#ifndef __SINGLE_FP__
        conv_value.d = 0.0;
#else
        conv_value.f = 0.0;
#endif

        /* Scan for the fraction of the floating point number.
         */
        while ( wc = *sc++, ( iswxdigit( wc ) || ( wc == L'.' && !dotseen ) ) )
        {
                if ( first && wc == L'0' )
                {
                        /* Skip all leading '0' characters after the 0x.
                         */
                        continue;
                }

                if ( wc != L'.' )
                {
                        wc = towupper( wc );
                        if ( wc > L'9' )
                        {
                                wc = wc + 10 - L'A';
                        }
                        else
                        {
                                wc = wc - L'0';
                        }

                        if ( first )
                        {
                                wint_t  tmp_wc  = wc;

                                first = 0;
                                while ( tmp_wc != 1 )
                                {
                                        dotbits++;
                                        numbits++;
                                        tmp_wc >>= 1;
                                }

                                /* Mask of the implicit 1
                                 */
                                wc &= (( 1 << dotbits ) - 1);
                        }
                        else
                        {
                                if ( !dotseen )
                                {
                                        dotbits += 4;
                                }
                                numbits  = 4;
                        }
                        /* Only add the digits that can be stored in the
                         * fractional part of the floating point number
                         */
                        if ( numbits && bitcount )
                        {
                                if ( numbits > bitcount )
                                {
                                        wc    >>= ( numbits - bitcount );
                                        numbits = bitcount;
                                }

#ifndef __SINGLE_FP__
                                hi_fract = ((hi_fract << numbits) | (lo_fract >> (32 - numbits)));
                                lo_fract = ((lo_fract << numbits) | (unsigned long) wc);
#else
                                fraction = ((fraction << numbits) | (unsigned long) wc);
#endif
                                bitcount -= numbits;
                        }
                }
                else
                {
                        dotseen = 1;
                }
        }

        /* Check if we need to shift the fractional part
         */
        if ( bitcount > 0 )
        {
#ifndef __SINGLE_FP__
                hi_fract   = ((hi_fract << bitcount) | (lo_fract >> (32 - bitcount)));
                lo_fract <<= bitcount;
#else
                fraction <<= bitcount;
#endif
        }

        /* Scan for possible exponent.
         */
        if ( wc == L'p' || wc == L'P' )
        {
                wc = *sc++;
                if ( wc == L'-' || wc == L'+' )
                {
                        if ( wc == L'-' )
                        {
                                signed_exponent = 1;
                        }

                        wc = *sc++;
                }
                                
                while ( iswdigit( wc ) )
                {
                        exponent = exponent * 10 + (wc - L'0');
                        wc       = *sc++;
                }

                /* Adjust the exponent to its sign and add the bias so
                 * the exponent is between 0 and 255.
                 */
                if ( signed_exponent )
                {
                        exponent = -exponent;
                }

                /* Adjust the exponent for the number of bits before the
                 * decimal point.
                 */
                exponent += dotbits;
        }

        if ( ptr != NULL )
        {
                sc--;
                *ptr = (wchar_t *) sc;
        }

        if ( exponent < min_exponent || exponent > max_exponent )
        {
                errno = ERANGE;
                value = huge_value;

                if ( sign )
                {
                        value = -value;
                }
        }
        else
        {
#ifndef __SINGLE_FP__
                conv_value.s.lo = lo_fract;
                conv_value.s.hi = hi_fract;
                conv_value.s.hi = PUT_DOUBLE_EXPONENT(conv_value.s.hi, exponent + DOUBLE_BIAS);
                conv_value.s.hi = PUT_DOUBLE_SIGN(conv_value.s.hi, sign);
                value           = conv_value.d;
#else
                conv_value.l = fraction;
                conv_value.l = PUT_FLOAT_EXPONENT(conv_value.l, exponent + FLOAT_BIAS);
                conv_value.l = PUT_FLOAT_SIGN(conv_value.l, sign);
                value        = conv_value.f;
#endif
        }

        return value;
}


static long double _getnan( const wchar_t*      sc,
                            wchar_t**           ptr,
                            unsigned char       sign,
                            int                 length
                            )
{
        long double     value;
#ifndef __SINGLE_FP__
        double2longs_t  conv_value;
#else
        float2long_t    conv_value;
#endif

        if ( ptr != NULL )
        {
                *ptr = (wchar_t *) ( sc + length );
        }

        /* For NaN, the fractional part of the floating point value is
         * non-zero, the exponent is maximal. Sign is irrelevant.
         */
#ifndef __SINGLE_FP__
        conv_value.s.lo = 1UL;
        conv_value.s.hi = PUT_DOUBLE_EXPONENT( 0UL, DOUBLE_BIASED_NaN_EXP );
        value           = conv_value.d;
#else
        conv_value.l = PUT_FLOAT_EXPONENT( 1UL, FLOAT_BIASED_NaN_EXP );
        value        = conv_value.f;
#endif

        return value;
}


static long double _getinf( const wchar_t*      sc,
                            wchar_t**           ptr,
                            unsigned char       sign,
                            int                 length
                            )
{
        long double     value;
#ifndef __SINGLE_FP__
        double2longs_t  conv_value;
#else
        float2long_t    conv_value;
#endif

        if ( ptr != NULL )
        {
                *ptr = (wchar_t *) ( sc + length );
        }

        /* For INF, the fractional part of the floating point value is zero,
         * the exponent is maximal. Sign is relevant.
         */
#ifndef __SINGLE_FP__
        conv_value.s.lo = 0UL;
        conv_value.s.hi = PUT_DOUBLE_EXPONENT( PUT_DOUBLE_SIGN(0UL, sign), DOUBLE_BIASED_INF_EXP );
        value           = conv_value.d;
#else
        conv_value.l = PUT_FLOAT_EXPONENT( PUT_FLOAT_SIGN(0UL, sign), FLOAT_BIASED_INF_EXP );
        value        = conv_value.f;
#endif

        return value;
}

#endif
