
/**************************************************************************
**                                                                        *
**  FILE        :  _filbuf.c                                              *
**                                                                        *
**  DESCRIPTION :  Source file for _filbuf() routine                      *
**                 This routine refills the input buffer when necessary.  *
**                 Returns 'EOF' on error.                                *
**                                                                        *
**  Copyright 1996-2009 Altium BV                                         *
**                                                                        *
**************************************************************************/

#include <stdio.h>
#include <io.h>

/*
 * The only way we get here is via the getc() family of functions/macros
 * which increment _cnt unconditionally and call us when the result is
 * no longer negative. So, _cnt is guaranteed to be >= 0 upon entry. The
 * zero case upon entry means that the buffer is ready for new input.
 * Positive values indicate we need to _fflush() buffered output first.
 *
 * Line buffered reads and fully buffered reads are treated the same. The
 * '\n' logic is implemented in fgets() and fscanf() family and will call
 * us automatically when a _read() did not include the '\n'. ISO/IEC 9899:1999
 * says "When a stream is line buffered, characters are intended to be
 * transmitted to or from the host environment as a block when a new-line
 * character is encountered". So, that's what _read() should do, together
 * with input echoing, backspace and the usual stuff in tty drivers.
 *
 * In a hosted environment the buffering and its allocation is usually
 * decided upon the first I/O operation, i.e. a delayed allocation. We don't
 * do that to avoid allocator overhead and use static buffers instead.
 * To line buffer or to do full buffering is typically decided by looking
 * at the fileno(fp) properties using system calls like fstat(). We don't
 * do that either.
 */

int _filbuf(FILE * fp)

#ifdef WCHAR_SUPPORT_ENABLED
{
        fp->_flag |= _IOCHAR;   /* tell fwide() that fp is byte-oriented */
        return _dofil(fp);
}
int     _dofil(FILE * fp)
#endif

{
        int             i;
        char            c, *ptr;
        _iob_flag_t     flag;

        /* Undo unconditional increment done by getc() family */
        fp->_cnt--;

        /* Check if file is readable */
        flag = fp->_flag;
        if (!(flag & _IORD))
        {
                fp->_flag = flag | _IOERR;
                return EOF;
        }

        /*
         * According to ISO/IEC 9899:1999 we should do this only upon _read()
         * and never when the output stream has _IOFBF. And we should do it
         * for all streams. But the sentence "support for these characteristics
         * is implementation-defined" saves us from doing more than:
         */
        if (fp == stdin)
        {
                _fflush(stdout);
        }

        if (flag & _IONBF)
        {
                i = fp->_bufsiz;
                if (i != EOF)           /* ungetc buffer hack */
                {
                        fp->_bufsiz = EOF;
                        return i;
                }
                ptr = &c;
                i = 1;
        }
        else 
        {
                if (fp->_cnt >= 0)      /* flush output, leaving a zero _cnt */
                {
                        _fflush(fp);
                        flag = fp->_flag;
                }
                ptr = fp->_base;
                i = fp->_bufsiz;
        }
        i = _read(fileno(fp), ptr, i);
        if (i < 0)
        {
                flag |= _IOERR;
                i = EOF;
        }
        else if (i == 0)
        {
                flag |= _IOEOF;
                i = EOF;
        }
        else
        {
                fp->_cnt = -i;                  /* i - 1 unread bytes */
                i = (unsigned char)*ptr++;      /* because we use one */
                if (!(flag & _IONBF))
                {
                        fp->_ptr = ptr;
                }
        }
        fp->_flag = flag;
        return i;
}
