
/**************************************************************************
**                                                                        *
**  FILE        :  _doscan.h                                              *
**                                                                        *
**  DESCRIPTION :  Common code for _doscan()/_dowscan() functions         *
**                                                                        *
**                 Define :                                               *
**                 SMALL  - No floats                                     *
**                 LARGE  - Full ANSI scanner                             *
**                                                                        *
**  Copyright 1996-2009 Altium BV                                         *
**                                                                        *
**************************************************************************/

#include <stdarg.h>
#include <errno.h>
#include <stddef.h>
#include <limits.h>
#include <stdio.h>
#include <io.h>
#include <stdint.h>
#if WIDE
#include <wctype.h>
#else /* WIDE */
#include <ctype.h>
#ifdef WCHAR_SUPPORT_ENABLED
#include <wchar.h>
#endif /* WCHAR_SUPPORT_ENABLED */
#endif /* WIDE */

#include "_printflags.h"

#if !defined SMALL && !defined LARGE
#define LARGE           /* Default scanner */
#endif

typedef unsigned short          FORMAT_TYPE;


/* Function prototypes */
#ifndef SMALL
extern  int     GETFLOAT        ( uchar_t, FORMAT_TYPE, unsigned int,
                                  void *, struct _io *, unsigned int * );
#endif
static void     _returnint      ( void *, FORMAT_TYPE, uint_value_t );
static int_t    _getnumber      ( char_t, FORMAT_TYPE, unsigned int,
                                  void *, struct _io *, unsigned int * );
static int_t    _getstring      ( const char_t *, FORMAT_TYPE, unsigned int,
                                  void *, struct _io *, unsigned int * );
static int_t    _readchar       ( struct _io *, unsigned int * );


/*
 * Common code for scanf functions.
 * Returns the number of input items assigned, or EOF.
 */
int DOSCAN( struct _io *fp, const char_t *fmt, va_list ap )
{
        int_t           i;                      /* character from input */
        char_t          ch;                     /* character in format string */
        const char_t *  match           = fmt;
        int             count           = 0;    /* count the read number of items */
        unsigned int    length;                 /* maximum length of input field */
        unsigned int    n_chars         = 0;    /* # characters read */
        FORMAT_TYPE     type;                   /* type of input field (long, short) */
        unsigned char   skipinput       = 0;
        unsigned char   noconvert       = 1;

        for ( ;; )
        {
                ch = *fmt++;
                switch ( ch )
                {
                case '\0':                      /* end of scan reached */
                        return count;
                case ' ':
                case '\t':
                        /* remove all whitespace from the input */
                        do
                        {
                                i = _readchar( fp, &n_chars );
                        } while ( ISSPACE( i ) );
                        UNGETC( i, fp );        /* We did not read the non-whitespace character yet */
                        n_chars--;
                        break;
                case '%':
                        ch = *fmt++;            /* get next character */
                        if ( ch == '%' )
                        {
                                goto literal;
                        }

                        /* Formatter expected */
                        skipinput = (ch == '*'); /* no assignment ? */
                        if ( skipinput )
                        {
                                ch = *fmt++;
                        }

                        length = 0;
                        while ( ISDIGIT( (uchar_t) ch ) )
                        {       /* read length specifier */
                                length = (length * 10) + (unsigned char) ch - '0';
                                ch     = *fmt++;
                        }

                        type = 0;
                        do
                        {
                                switch ( ch )
                                {
                                case 'l':
                                        if ( *fmt == 'l' )
                                        {
                                                fmt++;
                                                type = FLG_LONG_LONG;
                                        }
                                        else
                                        {
                                                type = FLG_LONG;
                                        }
                                        break;
                                case 'h':
                                        if ( *fmt == 'h' )
                                        {
                                                fmt++;
                                                type = FLG_SHORT_SHORT;
                                        }
                                        else
                                        {
                                                type = FLG_SHORT;
                                        }
                                        break;
                                case 'j':
                                        type = FLG_INTMAX_T;
                                        break;
                                case 'z':
                                        type = FLG_SIZE_T;
                                        break;
                                case 't':
                                        type = FLG_PTRDIFF_T;
                                        break;
                                case 'L':
                                        type = FLG_DOUBLE;
                                        break;
                                default:
                                        continue;
                                }
                                ch = *fmt++;
                        } while ( 0 );

                        switch ( ch )
                        {
                        case 'p':
                                if ( sizeof( void * ) == sizeof( long ) )
                                {
                                        type |= FLG_LONG;
                                }
                                /* ... fall through ... */
                        case 'X':
                                ch = 'x';
                                /* ... fall through ... */
                        case 'd':
                        case 'i':
                        case 'o':
                        case 'u':
                        case 'x':
                                /* integers */
                                i = _getnumber( ch, type, length,
                                                skipinput ? NULL : va_arg( ap, char_t * ),
                                                fp, &n_chars );
                                if ( i == E_O_F )
                                {
                                        return noconvert ? EOF : count;
                                }
                                noconvert = 0;
                                count += i;
                                break;
                        case 'c':
                                /* characters */
                                if ( length == 0 )
                                {
                                        length = 1;
                                }
                                if ( !skipinput )
                                {
                                        char*           ptc     = (char *) 0;
#if WIDE || defined WCHAR_SUPPORT_ENABLED
                                        wchar_t*        ptw     = (wchar_t *) 0;
                                        mbstate_t       mbst    = {0};

                                        if ( type & FLG_LONG )
                                        {
                                                ptw = va_arg(ap, wchar_t *);
                                        }
                                        else
                                        {
                                                ptc = va_arg(ap, char *);
                                        }
#else  /* WIDE || WCHAR_SUPPORT_ENABLED */
                                        ptc = va_arg(ap, char *);
#endif /* WIDE || WCHAR_SUPPORT_ENABLED */
                                        i = _readchar( fp, &n_chars );

                                        if ( i != E_O_F )
                                        {
                                                count++;
                                                noconvert = 0;
                                        }

                                        while ( length-- )
                                        {
                                                if ( i == E_O_F )
                                                {
                                                        return noconvert ? EOF : count;
                                                }
#if WIDE
                                                if ( type & FLG_LONG )
                                                {
                                                        *ptw++ = (wchar_t) i;
                                                }
                                                else
                                                {
                                                        wcrtomb( ptc++, (wchar_t) i, &mbst );
                                                }
#else /* WIDE */
#ifdef WCHAR_SUPPORT_ENABLED
                                                if ( type & FLG_LONG )
                                                {
                                                        char c = (char) i;
                                                        if ( mbrtowc(ptw++, & c, 1, &mbst) == (size_t) -1 )
                                                        {
                                                                return noconvert ? EOF : count;
                                                        }
                                                }
                                                else
                                                {
                                                        *ptc++ = (char) i;
                                                }
#else  /* WCHAR_SUPPORT_ENABLED */
                                                *ptc++ = (char) i;
#endif /* WCHAR_SUPPORT_ENABLED */
#endif /* WIDE */
                                                i = _readchar( fp, &n_chars );
                                        }
                                        UNGETC( i, fp );
                                        n_chars--;
                                }
                                else
                                {
                                        while ( length-- != 0 )
                                        {
                                                if ( _readchar( fp, &n_chars ) == E_O_F )
                                                {
                                                        return noconvert ? EOF : count;
                                                }
                                        }
                                }
                                break;
                        case '[':
                                /* matching or non matching string */
                                type |= FLG_LEFT_BRACKET;
                                match = fmt;
                                /* keep pointer to matching string, not including '[' */
                                if ( *fmt == '^' )
                                {
                                        fmt++;  /* no empty list allowed */
                                }
                                if ( *fmt )
                                {
                                        fmt++;  /* initial ']' is not closing bracket */
                                }
                                while ( *fmt && *fmt++ != ']' )
                                {       /* search for the matching ']' */
                                }
                                /* fall through ... */
                        case 's':
                                /* string */
                                i = _getstring( match, type, length,
                                                skipinput ? NULL : va_arg( ap, char_t * ),
                                                fp, &n_chars );
                                if ( i == E_O_F )
                                {
                                        return noconvert ? EOF : count;
                                }
                                else if ( i == 0 && ch == '[' && !skipinput )
                                {
                                        return count;
                                }
                                count += i;
                                noconvert = 0;
                                break;
                        case 'E':
                        case 'G':
                        case 'e':
                        case 'f':
                        case 'g':
                        case 'F':
                        case 'a':
                        case 'A':
                                ch = 'f';

#ifndef SMALL
                                if ( ! GETFLOAT( ch, type, length,
                                                 skipinput ? NULL : va_arg( ap, char_t * ),
                                                 fp, &n_chars ) )
                                {
                                        count++;
                                        noconvert = 0;
                                }
                                break;
#else
                                errno = ERR_NOFLOAT;    /* no floating point support */
                                return count;
#endif
                        case 'n':
                                /* number of characters written so far */
                                if ( ! skipinput )
                                {
                                        _returnint( va_arg( ap, int * ), type, n_chars );
                                }
                                break;
                        default:
                                errno = ERR_FORMAT; /* illegal format string */
                                return count;
                        }
                        break;
                default:        /* must match non-whitespace input character */
                literal:
                        i = _readchar( fp, &n_chars );
                        if ( i != (unsigned char) ch )  /* non matching character found */
                        {
                                UNGETC( i, fp );
                                return count;
                        }
                        break;
                }
        }
}

static void _returnint ( void *p, FORMAT_TYPE type, uint_value_t value )
{
        if ((sizeof(long) < sizeof(sint_value_t))
            && (type & FLG_LONG_LONG) == FLG_LONG_LONG)
        {
                *(sint_value_t *)p = (sint_value_t) value;
        }
        else if ((sizeof(long) < sizeof(intmax_t))
                 && (type & FLG_INTMAX_T))
        {
                *(intmax_t *)p = (intmax_t) value;
        }
        else if ((sizeof(int) != sizeof(long))
            && (type & (FLG_LONG | FLG_INTMAX_T)))
        {
                *(long *)p = (long) value;
        }
        else if ((sizeof(int) != sizeof(size_t))
                 && (type & FLG_SIZE_T))
        {
                *(size_t *)p = (size_t) value;
        }
        else if ((sizeof(int) != sizeof(ptrdiff_t))
                 && (type & FLG_PTRDIFF_T))
        {
                *(ptrdiff_t *)p = (ptrdiff_t) value;
        }
        else if ((sizeof(int) != sizeof(char))
                 && (type & FLG_SHORT_SHORT) == FLG_SHORT_SHORT )
        {
                *(char *)p = (char) value;
        }
        else if ((sizeof(int) != sizeof(short))
                 && (type & FLG_SHORT))
        {
                *(short *)p = (short) value;
        }
        else
        {
                /* Default is "int". */
                *(int *)p = (int) value;
        }
}

/*
 * When ap == NULL, no assignment should be made.
 *
 * Return E_O_F on input failure; otherwise
 * return 1 if conversion is done or
 * return 0 if not.
 */
static int_t _getnumber( char_t ch, FORMAT_TYPE type, unsigned int length,
                         void *ap, struct _io *fp, unsigned int *n_chars )
{
        int_t           i;
        uint_value_t    value           = 0;
        uint_value_t    tst_value;
        unsigned int    hold_char;
        unsigned char   neg             = 0;

        if ( length == 0 )                      /* no length specified ? */
        {
                length = (unsigned int)(-1);
        }

        do
        {
                i = _readchar( fp, n_chars );
        } while ( ISSPACE( i ) );               /* read whitespace */

        /* now 'i' is the first non whitespace character */

        hold_char = *n_chars - 1;               /* Whitespace characters are not 'converted' */

        if ( i == '-' )                         /* negative decimal */
        {
                neg = 1;
                i   = _readchar( fp, n_chars );
                length--;
        }
        else if ( i == '+' )                    /* positive decimal */
        {
                i = _readchar( fp, n_chars );
                length--;
        }

        if ( ((ch == 'i') || (ch == 'x')) && (i == '0') )       /* Check for '0x' */
        {
                if ( ch == 'i' )
                {
                        ch = 'o';               /* Maybe octal, or later we may find it is hexadecimal */
                }
                i = _readchar( fp, n_chars );
                if ( length )
                {
                        length--;
                }
                if ( i == 'x' || i == 'X' )     /* hexadecimal */
                {
                        i = _readchar( fp, n_chars );
                        if ( length )
                        {
                                length--;
                        }
                        ch = 'x';               /* Make sure hexadecimal values will be read */
                }
        }
        else if ( ch == 'i' )
        {
                ch = 'd';                       /* Decimal value to read */
        }

        switch ( ch )
        {
        case 'd':
        case 'u':
                /* decimal */
                while ( length-- && ISDIGIT( i ) )
                {
                        tst_value = value;
                        value = (value * 10) + i - '0';
                        if ( value < tst_value )
                        {
                                value = UVALUE_MAX;
                        }
                        i = _readchar( fp, n_chars );
                }
                break;
        case 'o':
                /* octal */
                while ( length-- && (i >= '0' && i <= '7') )
                {
                        tst_value = value;
                        value = (value * 8) + i - '0';
                        if ( value < tst_value )
                        {
                                value = UVALUE_MAX;
                        }
                        i = _readchar( fp, n_chars );
                }
                break;
        case 'x':
                /* hexadecimal */
                while ( length-- && ISXDIGIT( i ) )
                {
                        i = TOUPPER( i );
                        if ( i > '9' )
                        {
                                i = i + 10 - 'A';
                        }
                        else
                        {
                                i = i - '0';
                        }
                        tst_value = value;
                        value = (value * 16) + i;
                        if ( value < tst_value )
                        {
                                value = UVALUE_MAX;
                        }
                        i = _readchar( fp, n_chars );
                }
                break;
        }

        UNGETC( i, fp );
        (*n_chars)--;

        if ( ch == 'd' || neg ) /* Read a signed value? */
        {
                if ( neg )
                {
                        value = (value > VALUE_MAX) ? VALUE_MIN : 0 - value;
                }
                else
                {
                        value = (value > VALUE_MAX) ? VALUE_MAX : value;
                }

                /* Convert the value to the correct size. The default value
                 * is of type sint_value_t.
                 */
                switch (type & FLG_ALL_MODIFIERS)
                {
                case FLG_LONG_LONG:
                case FLG_INTMAX_T:
                        break;

                case FLG_LONG:
#if VALUE_MAX > LONG_MAX
                        value = ((sint_value_t) value > LONG_MAX ) ? LONG_MAX :
                                ((sint_value_t) value < LONG_MIN ) ? LONG_MIN : value;
#endif
                        break;

                case FLG_SHORT_SHORT:
#if VALUE_MAX > CHAR_MAX
                        value = ((sint_value_t)value > CHAR_MAX ) ? CHAR_MAX :
                                ((sint_value_t)value < CHAR_MIN ) ? CHAR_MIN : value;
#endif
                        break;

                case FLG_SHORT:
#if VALUE_MAX > SHRT_MAX
                        value = ((sint_value_t)value > SHRT_MAX ) ? SHRT_MAX :
                                ((sint_value_t)value < SHRT_MIN ) ? SHRT_MIN : value;
#endif
                        break;

                default:
                        if ((sizeof(int) != sizeof(ptrdiff_t)) && (type & FLG_PTRDIFF_T))
                        {
#if VALUE_MAX > PTRDIFF_MAX
                                value = ((sint_value_t)value > PTRDIFF_MAX ) ? PTRDIFF_MAX :
                                        ((sint_value_t)value < PTRDIFF_MIN ) ? PTRDIFF_MIN : value;
#endif
                        }
                        else
                        {
                                /* This is used for the conversion of both
                                 * normal integers and size_t arguments.
                                 */
#if VALUE_MAX > INT_MAX
                                value = ((sint_value_t)value > INT_MAX ) ? INT_MAX :
                                        ((sint_value_t)value < INT_MIN ) ? INT_MIN : value;
#endif
                        }
                }
        }
        else            /* Read an unsigned value? */
        {
                /* Convert the value to the correct size. The default value
                 * is of type uint_value_t.
                 */
                switch (type & FLG_ALL_MODIFIERS)
                {
                case FLG_LONG_LONG:
                case FLG_INTMAX_T:
                        break;

                case FLG_LONG:
#if UVALUE_MAX > ULONG_MAX
                        if ( value > ULONG_MAX )
                        {
                                value = ULONG_MAX;
                        }
#endif
                        break;

                case FLG_SHORT_SHORT:
#if UVALUE_MAX > UCHAR_MAX                      
                        if ( value > UCHAR_MAX )
                        {
                                value = UCHAR_MAX;
                        }
#endif
                        break;

                case FLG_SHORT:
#if UVALUE_MAX > USHRT_MAX
                        if ( value > USHRT_MAX )
                        {
                                value = USHRT_MAX;
                        }
#endif
                        break;

                default:
                        if ((sizeof(int) != sizeof(size_t)) && (type & FLG_SIZE_T))
                        {
#if UVALUE_MAX > SIZE_MAX
                                if ( value > SIZE_MAX )
                                {
                                        value = SIZE_MAX;
                                }
#endif
                        }
#if UVALUE_MAX > UINT_MAX
                        else
                        {
                                /* This is used for the conversion of both
                                 * normal integers and ptrdiff_t arguments.
                                 */
                                if ( value > UINT_MAX )
                                {
                                        value = UINT_MAX;
                                }
                        }
#endif
                }
        }

        if ( *n_chars != hold_char )    /* conversion is done */
        {
                if ( ap )       /* Do assignment AND a real value is detected */
                {
                        _returnint( ap, type, value );
                        return 1;
                }
        }
        else                    /* No conversion is done, found EOF, return EOF */
        {
                if ( i == E_O_F )
                {
                        return E_O_F;
                }
        }
        return 0;               /* No assignment done -> No conversion */
}

/*
 * match points to matching string, not including '['.
 * When ap == 0, no assignment is done, 0 is returned.
 *
 * Return E_O_F on input failure; otherwise
 * return 1 if conversion is done or
 * return 0 if not.
 */
static int_t _getstring( const char_t *match, FORMAT_TYPE type, unsigned int length,
                         void *ap, struct _io *fp, unsigned int *n_chars )
{
        unsigned char   notsearch;
                /* when '0', search for characters not in matching string */
                /* when '1', search for characters in matching string     */
                /* when '2', no matching string                           */
        unsigned char   converted       = 0;
        const char_t*   search;                 /* just a pointer         */
        int_t           i               = 0;    /* keep read character in */
        unsigned char   stop            = 0;
#if WIDE
        mbstate_t       mbst            = {0};
        size_t          n_bytes         = 0;
#else /* WIDE */
#ifdef WCHAR_SUPPORT_ENABLED
        mbstate_t       mbst            = {0};
        char            c;
#endif /* WCHAR_SUPPORT_ENABLED */
#endif /* WIDE */

        if ( length == 0 )                      /* no length specified? */
        {
                length = 30000;
        }

        if ( type & FLG_LEFT_BRACKET )
        {
                if ( *match == '^' )
                {       /* matching characters NOT in matching string ? */
                        notsearch = 0;
                        ++match;
                }
                else
                {
                        notsearch = 1;
                }
                i = _readchar( fp, n_chars );
        }
        else
        {
                notsearch = 2;
                do
                {
                        i = _readchar( fp, n_chars );
                } while ( ISSPACE( i ) );
        }

        if ( i == E_O_F )
        {
                UNGETC( i, fp );
                (*n_chars)--;
                return E_O_F;   /* EOF reached before conversion is done */
        }

        while ( length != 0 )
        {
                length--;
                if ( i == E_O_F )
                {
                        break;
                }

                if ( notsearch  == 2 )
                {
                        if ( ISSPACE( i ) )
                        {       /* End of string found on whitespace */
                                UNGETC( i, fp );
                                (*n_chars)--;
                                break;
                        }
                        if ( ap )
                        {
#if WIDE
                                if ( type & FLG_LONG )
                                {
                                        *(wchar_t *) ap = (wchar_t) i;
                                }
                                else
                                {
                                        n_bytes = wcrtomb( (char *) ap, (wchar_t) i, &mbst );
                                        if ( n_bytes == (size_t) -1 )
                                        {
                                                return E_O_F;   /* encoding error occured */
                                        }
                                }
#else /* WIDE */
#ifdef WCHAR_SUPPORT_ENABLED
                                if ( type & FLG_LONG )
                                {
                                        c = (char) i;
                                        if ( mbrtowc((wchar_t *)ap, & c, 1, &mbst) == (size_t) -1 )
                                        {
                                                return E_O_F;   /* encoding error occured */
                                        }
                                }
                                else
                                {
                                        *(char *)ap = (char) i;
                                }
#else  /* WCHAR_SUPPORT_ENABLED */
                                *(char *)ap = (char) i;
#endif /* WCHAR_SUPPORT_ENABLED */
#endif /* WIDE */
                        }
                }
                else
                {
                        for ( search = match; *search; )
                        {
                                if ( i == (uchar_t) *search )
                                {
                                        stop = 1;
                                }
                                else if ( *++search != ']' )
                                {
                                        continue;
                                }
                                stop ^= notsearch;
                                if ( stop )
                                {
                                        if ( i != '\n' )
                                        {
                                                UNGETC( i, fp );
                                                (*n_chars)--;
                                        }
                                }
                                else if ( ap )
                                {
#if WIDE
                                        if ( type & FLG_LONG )
                                        {
                                                *(wchar_t *) ap = (wchar_t) i;
                                        }
                                        else
                                        {
                                                n_bytes = wcrtomb( (char *) ap, (wchar_t) i, &mbst );
                                                if (n_bytes == (size_t) -1)
                                                {
                                                        return E_O_F;   /* encoding error occured */
                                                }
                                        }
#else /* WIDE */
#ifdef WCHAR_SUPPORT_ENABLED
                                        if ( type & FLG_LONG )
                                        {
                                                c = (char) i;
                                                if ( mbrtowc((wchar_t *)ap, & c, 1, &mbst) == (size_t) -1 )
                                                {
                                                        return E_O_F;   /* encoding error occured */
                                                }
                                        }
                                        else
                                        {
                                                *(char *)ap = (char) i;
                                        }
#else  /* WCHAR_SUPPORT_ENABLED */
                                        *(char *)ap = (char) i;
#endif /* WCHAR_SUPPORT_ENABLED */
#endif /* WIDE */
                                }
                                break;
                        }
                }

                if ( stop )
                {
                        break;
                }
                else if ( ap )
                {
#if WIDE
                        if ( type & FLG_LONG )
                        {
                                ap = (wchar_t *) ap + 1;
                        }
                        else
                        {
                                ap = (char *) ap + n_bytes;
                        }
#else /* WIDE */
#ifdef WCHAR_SUPPORT_ENABLED
                        if ( type & FLG_LONG )
                        {
                                ap = (wchar_t *) ap + 1;
                        }
                        else
                        {
                                ap = (char *) ap + 1;
                        }
#else  /* WCHAR_SUPPORT_ENABLED */
                        ap = (char *)ap + 1;
#endif /* WCHAR_SUPPORT_ENABLED */
#endif /* WIDE */
                }

                converted = 1;          /* Some conversion had success */
                if ( length != 0 )      /* When not ready converting, read next character */
                {
                        i = _readchar( fp, n_chars );
                }
        }

        if ( ap && converted )
        {
#if WIDE || defined WCHAR_SUPPORT_ENABLED
                if ( type & FLG_LONG )
                {
                        *(wchar_t *) ap = '\0';
                }
                else
                {
                        *(char *) ap = '\0';
                }
#else  /* WIDE || WCHAR_SUPPORT_ENABLED */
                *(char *)ap = '\0';
#endif /* WIDE || WCHAR_SUPPORT_ENABLED */
                return converted;
        }
        return 0;       /* No assignment -> No conversion done. */
}

static int_t _readchar( struct _io *fp, unsigned int *n_chars )
{
        (*n_chars)++;
        return FGETC( fp );
}

