
/**************************************************************************
**                                                                        *
**  FILE        :  _doprint.h                                             *
**                                                                        *
**  DESCRIPTION :  Common code for _doprint()/_dowprin() functions        *
**                                                                        *
**  Copyright 1996-2009 Altium BV                                         *
**                                                                        *
**************************************************************************/

#include <stdarg.h>
#include <errno.h>
#include <string.h>
#include <limits.h>
#include <stdio.h>
#include <io.h>
#include <stdint.h>
#include <stddef.h>
#include <stdlib.h>
#ifdef WCHAR_SUPPORT_ENABLED
#include <wchar.h>
#endif /* WCHAR_SUPPORT_ENABLED */

#if !defined SMALL && !defined MEDIUM && !defined LARGE
#define LARGE           /* Default formatter. */
#endif

#include "_printflags.h"

#define LTOA_DIGITS     ((8*sizeof(sint_value_t)+2)/3)

#ifndef LARGE
static void _fputs( char_t *str, struct _io *fp, unsigned int *n_chars );
#endif

/* convert an unsigned int to a string                  */
static char * _ltoa( char *p, uint_value_t num, unsigned int radix );
#ifndef SMALL
static void _putstring( format_value_type format, int min, int max,
                        char_t *str, struct _io *fp, unsigned int *n_chars );
static void _putnumber( int min, int max,
                        char_t ch, format_value_type format,
                        sint_value_t value, struct _io *fp, unsigned int *n_chars );
#else
static void _putnumber( char_t ch, format_value_type format,
                        sint_value_t value, struct _io *fp, unsigned int *n_chars );
#endif
static void _emitchar( struct _io* fp, char_t ch, unsigned int* n_chars );
#if WIDE
static void convert_chars_to_wchars(wchar_t *ptws, char *ptcs);
#endif /* WIDE */

/* Floating point formatter */
#ifdef  LARGE
extern  void DOFLT( int min, int max, uchar_t type, format_value_type format,
                    float_value_t value, struct _io *fp, unsigned int *n_chars );
#endif


/* does the actual printing     */
int DOPRINT( struct _io *fp, const char_t *fmt, va_list ap )
{
#ifndef SMALL
        int                     min;            /* minimum length of output field */
        int                     max;            /* maximum length of output field */
#endif
        char_t                  ch;             /* character in format string     */
        format_value_type       format;         /* output format specifiers       */
        unsigned int            n_chars = 0;    /* no characters written yet      */

        for ( ;; )
        {
                ch = *fmt;
                if ( ! ch )
                {
                        return n_chars;
                }

                if ( ch != '%' || *++fmt == '%' )
                {
                        fmt++;
                        _emitchar( fp, ch, &n_chars );
                        continue;
                }

                /* Initialize the format specifiers.
                 */
                format  = 0;
#ifndef SMALL
                min     = 0;
                max     = 0;
#endif

#ifndef SMALL
                /* Check for format flags.
                 */
                while ( 1 )
                {
                        ch = *fmt++;
                        if ( ch == '-' )
                        {
                                format |= FLG_LEFT_ALIGNMENT;
                        }
                        else if ( ch == '+' )
                        {
                                /* The ' ' format flag is ignored when the
                                 * '+' flag is used.
                                 */
                                format &= ~FLG_SIGN_OR_SPACE;
                                format |= FLG_ALWAYS_SIGN;
                        }
                        else if ( ch == ' ' )
                        {
                                /* Do not overrule '+'.
                                 */
                                if ( !( format & FLG_ALWAYS_SIGN ) )
                                {
                                        format |= FLG_SIGN_OR_SPACE;
                                }
                        }
                        else if ( ch == '0' )
                        {
                                format |= FLG_ZERO_FILL;
                        }
                        else if ( ch == '#' )
                        {
                                format |= FLG_ALTERNATIVE_FORMAT;
                        }
                        else
                        {
                                break;
                        }
                }

                if ( ch == '*' )
                {
                        /* Next argument is minimum field width specifier. */
                        min = va_arg( ap, int );
                        if ( min < 0 )
                        {
                                format |= FLG_LEFT_ALIGNMENT;
                                min     = -min;
                        }
                        ch = *fmt++;
                }
                else
                {
                        /* Get minimum field width. */
                        while ( ch >= '0' && ch <= '9' )
                        {
                                min = min * 10 + (unsigned char) ch - '0';
                                ch  = *fmt++;
                        }
                }

                if ( ch == '.' )
                {
                        /* Next argument is maximum field width specifier. */
                        format |= FLG_MAX_LENGTH;
                        if ( (ch = *fmt++) == '*' )
                        {
                                max = va_arg( ap, int );
                                if( max < 0 )
                                {
                                        format |= FLG_LEFT_ALIGNMENT;
                                        format &= ~FLG_MAX_LENGTH;
                                }
                                ch = *fmt++;
                        }
                        else
                        {
                                /* Get maximum field width. */
                                while( ch >= '0' && ch <= '9' )
                                {
                                        max = max * 10 + (unsigned char) ch - '0';
                                        ch  = *fmt++;
                                }
                        }
                }
#else
                ch = *fmt++;
#endif

                /* Check for length modifiers.
                 */
                do
                {
                        if ( ch == 'l' )
                        {
                                format |= FLG_LONG;
                        }
                        else if ( ch == 'h' )
                        {
                                format |= FLG_SHORT;
                        }
                        else if ( ch == 'L' )
                        {
                                format |= FLG_DOUBLE;
                        }
                        else if ( ch == 'j' )
                        {
                                format |= FLG_INTMAX_T;
                        }
                        else if ( ch == 'z' )
                        {
                                format |= FLG_SIZE_T;
                        }
                        else if ( ch == 't' )
                        {
                                format |= FLG_PTRDIFF_T;
                        }
                        else
                        {
                                continue;
                        }

                        if ( ( ch == 'l' || ch == 'h' ) && ( ch == *fmt ) )
                        {
                                format |= FLG_DOUBLE_LENGTH;
                                fmt++;
                        }

                        ch = *fmt++;
                } while ( 0 );

                /* Conversion specifiers.
                 */
                if ( ch == 'c' )
                {
                        char_t  tmp_c;

#ifndef SMALL
                        if ( !( format & FLG_LEFT_ALIGNMENT ) )
                        {
                                while ( min > max && min > 1 )
                                {
                                        _emitchar( fp, ' ', &n_chars );
                                        min--;
                                }
                        }
#endif
#if WIDE
                        if ( format & FLG_LONG )
                        {
                                tmp_c = (char_t) va_arg( ap, wint_t );
                        }
                        else
                        {
                                tmp_c = (char_t) btowc( va_arg( ap, int ) );
                        }
#else /* WIDE */
#ifdef WCHAR_SUPPORT_ENABLED
                        if ( format & FLG_LONG )
                        {
                                mbstate_t       mbst = { 0 };

                                if ( wcrtomb( &tmp_c, (wchar_t) va_arg( ap, wint_t ), &mbst ) == (size_t) -1 )
                                {
                                        return n_chars;
                                }
                        }
                        else
                        {
                                tmp_c = (char_t) va_arg( ap, int );
                        }
#else  /* WCHAR_SUPPORT_ENABLED */
                        tmp_c = (char_t) va_arg( ap, int );
#endif /* WCHAR_SUPPORT_ENABLED */
#endif /* WIDE */
                        _emitchar( fp, tmp_c, &n_chars );
#ifndef SMALL
                        if ( format & FLG_LEFT_ALIGNMENT )
                        {
                                while ( min > max && min > 1 )
                                {
                                        _emitchar( fp, ' ', &n_chars );
                                        min--;
                                }
                        }
#endif
                }
                else if ( ch == 'd' || ch == 'i' || ch == 'o' ||
                          ch == 'x' || ch == 'X' || ch == 'u' )
                {
                        /* signed decimal integer       */
                        /* unsigned octal integer       */
                        /* unsigned hexadecimal integer */
                        /* unsigned decimal integer     */

                        sint_value_t    l;
                        unsigned char   size;
                        unsigned char   is_signed       = ( ch == 'd' || ch == 'i' );

                        /* Determine the size of the argument. */
                        if ((sizeof(long) < sizeof(sint_value_t))
                            && (format & FLG_LONG_LONG) == FLG_LONG_LONG)
                        {
                                size = sizeof(sint_value_t);
                        }
                        else if ((sizeof(long) < sizeof(intmax_t))
                                 && (format & FLG_INTMAX_T))
                        {
                                size = sizeof(intmax_t);
                        }
                        else
                        if ((sizeof(int) != sizeof(long))
                                 && (format & (FLG_LONG | FLG_INTMAX_T)))
                        {
                                /* This will also catch "long long". */
                                size = sizeof(long);
                        }
                        else
                        if ((sizeof(int) != sizeof(size_t))
                                 && (format & FLG_SIZE_T))
                        {
                                size = sizeof(size_t);
                        }
                        else if ((sizeof(int) != sizeof(ptrdiff_t))
                                 && (format & FLG_PTRDIFF_T))
                        {
                                size = sizeof(ptrdiff_t);
                        }
                        else
                        {
                                /* Default to "int". */
                                size = sizeof(int);
                        }

                        /* Retrieve the argument and cast it to the correct
                         * value type based on the size.
                         */
                        if ((sizeof(long) < sizeof(sint_value_t))
                            && (size == sizeof(sint_value_t)))
                        {
                                l = (sint_value_t)va_arg( ap, sint_value_t );
                        }
                        else
                        if ((sizeof(int) != sizeof(long))
                            && (size == sizeof(long)))
                        {
                                l = (sint_value_t)va_arg( ap, long );
                                if ( !is_signed )
                                {
                                        l = (sint_value_t)(unsigned long) l;
                                }
                        }
                        else
                        {
                                l = va_arg( ap, int );
                                if ((sizeof(int) != sizeof(short))
                                    && (format & FLG_SHORT_SHORT) == FLG_SHORT_SHORT)
                                {
                                        if (is_signed)
                                        {
                                                l = (signed char) l;
                                        }
                                        else
                                        {
                                                l = (unsigned char) l;
                                        }
                                }
                                else
                                if ((format & (FLG_SHORT | FLG_DOUBLE_LENGTH)) == FLG_SHORT)
                                {
                                        if (is_signed)
                                        {
                                                l = (short) l;
                                        }
                                        else
                                        {
                                                l = (unsigned short) l;
                                        }
                                }
                                else if (!is_signed)
                                {
                                        l = (unsigned int) l;
                                }
                        }

#ifndef SMALL
                        if ( format & FLG_MAX_LENGTH )
                        {
                                /* When precision specified, zero fill must
                                 * be disabled.
                                 */
                                format &= ~FLG_ZERO_FILL;
                        }
                        else
                        {
                                /* Precision defaults to 1.
                                 */
                                max = 1;
                        }

                        _putnumber( min, max, ch, format, l, fp, &n_chars );
#else
                        _putnumber( ch, format, l, fp, &n_chars );
#endif
                }
                else if ( ch == 's' )
                {
                        /* string */

#ifndef SMALL
                        /* precision defaults to "unlimited": */
                        if ( !( format & FLG_MAX_LENGTH ) )
                        {
                                max = INT_MAX;
                        }

                        _putstring( format, min, max, va_arg( ap, char_t * ), fp, &n_chars );
#else
                        _fputs( va_arg( ap, char_t * ), fp, &n_chars );
#endif
                }
#ifdef LARGE            /* Only supported on 'LARGE' formatter */
                else if (  ch == 'f' || ch == 'F'
                        || ch == 'e' || ch == 'E'
                        || ch == 'g' || ch == 'G'
                        || ch == 'a' || ch == 'A'
                        )
                {
                        float_value_t   f;

                        /* floating point */
                        if (sizeof(double) < sizeof(float_value_t)
                            && (format & FLG_DOUBLE_LENGTH))
                        {
                                f = va_arg( ap, float_value_t );
                        }
                        else
                        {
                                f = (float_value_t) va_arg( ap, double );
                        }

                        if ( !( format & FLG_MAX_LENGTH ) )
                        {
                                max = 6;
                        }
                        DOFLT( min, max, ch, format, f, fp, &n_chars );
                }
#else
                else if (  ch == 'f' || ch == 'F'
                        || ch == 'e' || ch == 'E'
                        || ch == 'g' || ch == 'G'
                        || ch == 'a' || ch == 'A'
                        )
                {                        
                        /* Just print we do not support floating point values */
                        _fputs( LITERAL("<no floats>"), fp, &n_chars );
                        /* select next argument */
                        (void) va_arg( ap, double );                    
                }
#endif
                else if ( ch == 'p' )                           /* pointer */
                {
#ifndef SMALL
                        if ( sizeof( void * ) == sizeof( long ) )
                        {
                                _putnumber( 2*sizeof(void*), 2*sizeof(void*), 'x', format | FLG_ZERO_FILL,
                                            (sint_value_t) va_arg( ap, unsigned long ), fp, &n_chars );
                        }
                        else
                        {
                                _putnumber( 2*sizeof(void*), 2*sizeof(void*), 'x', format | FLG_ZERO_FILL,
                                            (sint_value_t) va_arg( ap, unsigned ), fp, &n_chars );
                        }
#else
                        if ( sizeof( void * ) == sizeof( long ) )
                        {
                                _putnumber( 'x', format | FLG_ZERO_FILL,
                                            (sint_value_t) va_arg( ap, unsigned long ), fp, &n_chars );
                        }
                        else
                        {
                                _putnumber( 'x', format | FLG_ZERO_FILL,
                                            (sint_value_t) va_arg( ap, unsigned ), fp, &n_chars );
                        }
#endif
                }
                else if ( ch == 'n' )
                {
                        /* Emitted character count.
                         */
                        if ((sizeof(long) < sizeof(sint_value_t))
                            && (format & FLG_LONG_LONG) == FLG_LONG_LONG)
                        {
                                *(va_arg( ap, sint_value_t * )) = n_chars;
                        }
                        else if ((sizeof(long) < sizeof(intmax_t))
                                 && (format & FLG_INTMAX_T))
                        {
                                *(va_arg( ap, intmax_t * )) = n_chars;
                        }
                        else
                        if ((sizeof(int) != sizeof(long))
                            && (format & (FLG_LONG | FLG_INTMAX_T)))
                        {
                                /* This will also catch "long long" when
                                 * "long" and "long long" have equal sizes.
                                 */
                                *(va_arg( ap, long * )) = n_chars;
                        }
                        else
                        if ((sizeof(int) != sizeof(size_t))
                                 && (format & FLG_SIZE_T))
                        {
                                *(va_arg( ap, size_t * )) = n_chars;
                        }
                        else if ((sizeof(int) != sizeof(ptrdiff_t))
                                 && (format & FLG_PTRDIFF_T))
                        {
                                *(va_arg( ap, ptrdiff_t * )) = n_chars;
                        }
                        else
                        if ((sizeof(int) != sizeof(short))
                                 && (format & (FLG_SHORT | FLG_DOUBLE_LENGTH)) == FLG_SHORT )
                        {
                                *(va_arg( ap, short * )) = (short) n_chars;
                        }
                        else
                        if ((sizeof(int) != sizeof(char))
                                 && (format & FLG_SHORT_SHORT) == FLG_SHORT_SHORT )
                        {
                                *(va_arg( ap, char * )) = (char) n_chars;
                        }
                        else
                        {
                                /* Default is "int".
                                 */
                                *(va_arg( ap, int * )) = n_chars;
                        }
                }
                else
                {
                        errno = ERR_FORMAT;     /* illegal format string */
                        return n_chars;
                }
        }
}

#ifndef SMALL
/* Prints a string to the output. When 'min' > 'max', the result is
 * undefined, when 'left' != 0, left alignment is done, when min=0, and
 * max=0, strings are printed normal. The length of the strings are
 * supposed to be SMALLer than 30000 characters.
 */
static void _putstring( format_value_type format, int min, int max,
                        char_t *str, struct _io *fp, unsigned int *n_chars )
{
        int             length  = 0;
        char_t          c;
#if WIDE
        mbstate_t       mbst    = {0};
        char*           ptc     = (char *) str;

        if ( format & FLG_LONG )
        {
                length = (int)wcslen( str );
        }
        else
        {
                length = (int)strlen( (char *) str );
        }
#else /* WIDE */
#ifdef WCHAR_SUPPORT_ENABLED
        mbstate_t       mbst    = {0};

        if ( format & FLG_LONG )
        {
                length = (int)wcslen( (wchar_t *) str );
        }
        else
        {
                length = (int)strlen( str );
        }
#else
        length = (int)strlen( str );
#endif /* WCHAR_SUPPORT_ENABLED */
#endif /* WIDE */

        /* Check for right alignment and emit fill characters if so.
         */
        if ( !(format & FLG_LEFT_ALIGNMENT) )
        {
                for( ; min > length || min > max; --min )
                {
                        _emitchar( fp, ' ', n_chars );
                }
        }

        /* Print the string.
         */
        while ( max-- )
        {
#if WIDE
                if ( format & FLG_LONG )
                {
                        c = *str++;
                }
                else
                {
                        if (mbrtowc( &c, ptc, MB_CUR_MAX, &mbst ) >= (size_t) -2)
                        {
                                return;
                        }

                        ptc++;
                }
#else /* WIDE */
#ifdef WCHAR_SUPPORT_ENABLED
                if ( format & FLG_LONG )
                {
                        if ( wcrtomb( &c, *(wchar_t *) str, &mbst ) == (size_t) -1 )
                        {
                                return;
                        }

                        str += sizeof (wchar_t);
                }
                else
                {
                        c = *str++;
                }
#else  /* WCHAR_SUPPORT_ENABLED */
                c = *str++;
#endif /* WCHAR_SUPPORT_ENABLED */
#endif /* WIDE */

                if ( c == '\0' )
                {
                        break;
                }

                _emitchar( fp, c, n_chars );
                if ( min )
                {
                        min--;  /* min will contain the minimum number  */
                }               /* of characters to print afterwards    */
        }

        while ( min-- )
        {
                _emitchar( fp, ' ', n_chars );
        }
}
#endif

static void _putnumber(
#ifndef SMALL
        int min, int max,
#endif
        char_t ch, format_value_type format,
        sint_value_t value, struct _io *fp, unsigned int *n_chars )
{
        char_t  buf[LTOA_DIGITS+1];             /* room to print value in       */
        char_t  *buffer;                        /* help variable        */
#ifndef SMALL
        unsigned char length = 0;               /* length needed to print value */
#endif
        unsigned char vsign;                    /* sign of the value    */
#if WIDE
        char    char_buf[LTOA_DIGITS+1];        /* room to print value in       */
#endif /* WIDE */

        vsign = 0;      /* 0 - positive value, otherwise negative value */
        switch ( ch )
        {
        case 'd':
        case 'i':
                if ( value < 0 )
                {
                        vsign = '-';
                        value = - value;
                }
                /* .. fall through ... */
        case 'u':
#if WIDE
                _ltoa( char_buf, value, 10 );   /* convert to ascii */
                convert_chars_to_wchars(buf, char_buf);         
#else /* WIDE */
                _ltoa( buf, value, 10 );        /* convert to ascii */
#endif /* WIDE */
#ifndef SMALL
                if ( max == 0 && value == 0 )   /* If max == 0, it was explicitely */
                {                               /* specified as zero. If the value */
                        return;                 /* is also zero, don't print anything */
                }
                length = (unsigned char)STRLEN( buf );
                if ( vsign == 0 )
                {
                        if ( ch != 'u'
                             && (format & (FLG_ALWAYS_SIGN | FLG_SIGN_OR_SPACE)) == FLG_ALWAYS_SIGN )
                        {
                                /* sign always  */
                                vsign = '+';
                        }
                        else if ( (format & (FLG_ALWAYS_SIGN | FLG_SIGN_OR_SPACE)) == FLG_SIGN_OR_SPACE )
                        {
                                /* sign or space */
                                vsign = ' ';
                        }
                }

                if ( vsign )
                {
                        if ( min )
                        {
                                --min;
                        }

                        if ( format & (FLG_ZERO_FILL | FLG_LEFT_ALIGNMENT) )
                        {
                                _emitchar( fp, vsign, n_chars );
                                vsign = 0;
                        }
                }

                if ( !(format & FLG_LEFT_ALIGNMENT) )   /* right alignment ? */
                {
                        char_t  tmp_c   = (format & FLG_ZERO_FILL) ? '0' : ' ';

                        while ( (min > max) && (min > length) )
                        {
                                _emitchar( fp, tmp_c, n_chars );
                                min--;
                        }
                }
#endif

                if ( vsign )
                {
                        _emitchar( fp, vsign, n_chars );
                }

#ifndef SMALL
                while ( max-- > length )
                {
                        _emitchar( fp, '0', n_chars );
                        if ( min )
                        {
                                min--;
                        }
                }
#endif
                break;
        case 'o':
        case 'x':
        case 'X':
#if WIDE
                _ltoa( char_buf, value, ch == 'o' ? 8 : 16 );
                convert_chars_to_wchars(buf, char_buf);
#else /* WIDE */
                _ltoa( buf, value, ch == 'o' ? 8 : 16 );
#endif /* WIDE */
                if ( ch == 'x' )
                {
                        /* convert string to lowercase */
                        for( buffer = buf; *buffer; ++buffer )
                        {       /* do not use 'tolower' */
                                if ( *buffer >= 'A' && *buffer <= 'Z' )
                                {
                                        *buffer += 'a' - 'A';
                                }
                        }
                }

#ifndef SMALL
                length = (unsigned char)STRLEN( buf );

                if ( value == 0 )
                {
                        if ( max == 0 )
                        {
                                if ( ch != 'o' || ! (format & FLG_ALTERNATIVE_FORMAT) )
                                {
                                        return;
                                }
                                max = 1;
                        }

                        /* Clear alternate when value is '0'.
                         */
                        format &= ~FLG_ALTERNATIVE_FORMAT;
                }

                if ( (format & FLG_ALTERNATIVE_FORMAT) )        /* alternate */
                {
                        if ( format & FLG_ZERO_FILL )
                        {
                                /* Alternate and filling from the left with
                                 * zeros.
                                 */
                                if ( ch != 'o' )
                                {
                                        _emitchar( fp, '0', n_chars );
                                        if ( min )
                                        {
                                                min--;
                                        }
                                        _emitchar( fp, ch, n_chars );
                                        if ( min )
                                        {
                                                min--;
                                        }
                                }
                                format &= ~FLG_ALTERNATIVE_FORMAT;      /* clear alternate */
                        }

                        if ( ch == 'o' )
                        {
                                if ( max <= length )
                                {
                                        max = length+1;
                                }
                        }
                }

                if ( (format & FLG_ALTERNATIVE_FORMAT) && (ch != 'o') )
                {
                        /* '0x' takes up 2 spaces of the field width but
                         * doesn't count for the precision.
                         */
                        length += 2;
                        max += 2;
                }

                if ( !(format & FLG_LEFT_ALIGNMENT) )           /* right alignment      */
                {
                        char_t  tmp_c   = (format & FLG_ZERO_FILL) ? '0' : ' ';

                        while ( (min > max) && (min > length) )
                        {
                                _emitchar( fp, tmp_c, n_chars );
                                min--;
                        }
                }

                if ( (format & FLG_ALTERNATIVE_FORMAT) && (ch != 'o') )
                {
                        _emitchar( fp, '0', n_chars );
                        if ( min )
                        {
                                min--;
                        }
                        _emitchar( fp, ch, n_chars );
                        if ( min )
                        {
                                min--;
                        }
                }

                while ( max-- > length )
                {
                        _emitchar( fp, '0', n_chars );
                        if ( min )
                        {
                                min--;
                        }
                }
#endif
        }

#if WIDE
        /* Set the 'l' specifier, so the characters in buf will be handled as
         * wide characters.
         */
        format &= ~(FLG_LONG_LONG);
        format |= FLG_LONG;
#else /* WIDE */
        /* Clear the 'l' qualifier, otherwise _putstring will always try to
         * use wide characters!
         */
        format &= ~FLG_LONG;
#endif /* WIDE */

#ifndef SMALL
        /* Now print the ascii value, padded with spaces.
         */
        _putstring( format, min, INT_MAX, buf, fp, n_chars );
#else
        /* Now print the ascii value.
         */
        _fputs( buf, fp, n_chars );
#endif
}

static void _emitchar( struct _io* fp, char_t ch, unsigned int* n_chars )
{
        (*n_chars)++;
        FPUTC( (uchar_t) ch, fp );
}

#ifndef LARGE
/* Print only a string to the output. When the 'SMALL' formatter is requested,
 * _fputs() is called instead of _putstring()
 */
static void _fputs( char_t *str, struct _io *fp, unsigned int *n_chars )
{
        while ( *str != '\0' )
        {
                (*n_chars)++;
                FPUTC( (uchar_t) *str, fp );
                str++;
        }
}
#endif

#if WIDE
/* Convert characters to wide characters.
 */
static void convert_chars_to_wchars( wchar_t *ptws, char *ptcs )
{
        for (; *ptcs; ptcs++, ptws++)
        {
                *ptws = (wchar_t) *ptcs;
        }

        *ptws = (wchar_t) 0;
}
#endif /* WIDE */

static char * _ltoa( char *p, uint_value_t num, unsigned int radix )
{
        unsigned int    digit;
        uint_value_t    newnum;
        char *          q;
        char *          hold;

        hold = p;
        /* The address "q" should correspond to buffer size in _putnumber()
         * in _doprint.c.
         */
        q = p + LTOA_DIGITS + 1;

        /* Be sure to terminate the string with a NULL character. */
        *--q = '\0';

        do
        {
                /* avoid modulo operation */
                newnum = num / radix;
                digit = (unsigned int) (num - (newnum * radix));
                if ( digit > 9 )
                        digit += 'A'-'0'-10;    /* Convert to hex A-F */
                *--q = digit + '0';
                num = newnum;
        } while ( newnum != 0 );

        /* Move the string to the left, including the NULL character. */
        while ( (*p++ = *q++) )
                ;

        return hold;
}

