
/**************************************************************************
**                                                                        *
**  FILE        :  _atan.c                                                *
**                                                                        *
**  DESCRIPTION :  Source file for _atan() routine                        *
**                 Is used by 'atan()' and 'atan2()' routines             *
**                                                                        *
**      coefficients double are #5076 from Hart & Cheney. (17.55D)        *
**      coefficients single are #5071 from Hart & Cheney. ( 7.54D)        *
**                                                                        *
**  Copyright 1996-2009 Altium BV                                         *
**                                                                        *
**************************************************************************/

#include <math.h>

#ifndef __SINGLE_FP__
#pragma alias   _atanl  = __atan_dpf
#pragma alias   _atan           = __atan_dpf


static double xatan( double arg );

#define SQ2P1   2.414213562373095048802e0
#define SQ2M1   0.414213562373095048802e0
#define PIO2    1.570796326794896619231e0
#define PIO4    0.785398163397448309615e0
#define P0      0.445413400592906803197511e2
#define P1      0.77477687719204208616481e2
#define P2      0.40969264832102256374186e2
#define P3      0.666057901700926265753e1
#define P4      0.1589740288482307048e0
#define Q0      0.445413400592906804445995e2
#define Q1      0.92324801072300974840693e2
#define Q2      0.62835930511032376833267e2
#define Q3      0.1550397755142198752523e2
#define POLYNOMIAL(argsq,arg)   ( ( ( ( ( ( P4 * (argsq) + P3 ) * (argsq) + P2 ) * (argsq) + P1 ) * (argsq) + P0 ) / \
                        ( ( ( ( (argsq) + Q3 ) * (argsq) + Q2 ) * (argsq) + Q1 ) * (argsq) + Q0 ) ) * (arg) )

/*
        _atan reduces its argument (known to be positive)
        to the range [0,0.414...] and calls xatan.
*/
static
double
__atan_dpf( double arg )
{
        if( arg < SQ2M1 )
                return( xatan( arg ) );
        else if( arg > SQ2P1 )
                return( PIO2 - xatan( 1.0 / arg ) );
        else
                return( PIO4 + xatan( ( arg - 1.0 ) / ( arg + 1.0 ) ) );
}

/*
        xatan evaluates a series valid in the
        range [-0.414...,+0.414...].
*/
static double
xatan( double arg )
{
        double argsq;

        argsq = arg * arg;
        return POLYNOMIAL(argsq,arg);
}

#endif

