-- ----------------------------------------------------------------------------
Library IEEE;
Use     IEEE.Std_Logic_1164.All;
Use     IEEE.Std_Logic_Unsigned.All;
Use     IEEE.Numeric_Std.All;
-- ----------------------------------------------------------------------------

-- ----------------------------------------------------------------------------
Entity ReplaceWith_EntityName Is
   -- ------------------------------------------------------------------------
   Port (
        RD_EMPTY        : Out Std_Logic;                                                   -- IncludeIf_RD_Empty
        RD_ALMOST_EMPTY : Out Std_Logic;                                                   -- IncludeIf_RD_AlmostEmpty
        RD_DATA_COUNT   : Out Std_Logic_Vector(ReplaceWith_FifoAddressWidth - 1 Downto 0); -- IncludeIf_RD_DataCount

        WR_FULL         : Out Std_Logic;                                                   -- IncludeIf_WR_Full
        WR_ALMOST_FULL  : Out Std_Logic;                                                   -- IncludeIf_WR_AlmostFull
        WR_FREE_SLOTS   : Out Std_Logic_Vector(ReplaceWith_FifoAddressWidth - 1 Downto 0); -- IncludeIf_WR_FreeSlots

        RD_CLK          : In  Std_Logic;
        RD_RST          : In  Std_Logic;
        RD_EN           : In  Std_Logic;
        RD_DATA         : Out Std_Logic_Vector(ReplaceWith_FifoWidth - 1 Downto 0);

        WR_CLK          : In  Std_Logic;
        WR_RST          : In  Std_Logic;
        WR_EN           : In  Std_Logic;
        WR_DATA         : In  Std_Logic_Vector(ReplaceWith_FifoWidth - 1 Downto 0)
   );
   -- ------------------------------------------------------------------------
End;
-- ----------------------------------------------------------------------------

-- ----------------------------------------------------------------------------
Architecture RTL Of ReplaceWith_EntityName Is
-- ----------------------------------------------------------------------------
    -- ------------------------------------------------------------------------
    Component ReplaceWith_MemoryName Is
        Port(
            CLKA  : In  Std_Logic;
            CLKB  : In  Std_Logic;
            WEA   : In  Std_Logic;
            WEB   : In  Std_Logic;
            DINA  : In  Std_Logic_Vector(ReplaceWith_FifoWidth - 1 Downto 0);
            DINB  : In  Std_Logic_Vector(ReplaceWith_FifoWidth - 1 Downto 0);
            DOUTA : Out Std_Logic_Vector(ReplaceWith_FifoWidth - 1 Downto 0);
            DOUTB : Out Std_Logic_Vector(ReplaceWith_FifoWidth - 1 Downto 0);
            ADDRA : In  Std_Logic_Vector(ReplaceWith_FifoAddressWidth - 1 Downto 0);
            ADDRB : In  Std_Logic_Vector(ReplaceWith_FifoAddressWidth - 1 Downto 0)
        );
    End Component;
    -- ------------------------------------------------------------------------

    Signal ReadPointer       : Std_Logic_Vector(ReplaceWith_FifoAddressWidth - 1 Downto 0);
    Signal WritePointer      : Std_Logic_Vector(ReplaceWith_FifoAddressWidth - 1 Downto 0);

    Signal FifoFull          : Std_Logic;
    Signal FifoEmpty         : Std_Logic;
    Signal FifoAlmostFull    : Std_Logic;
    Signal FifoAlmostEmpty   : Std_Logic;

    Signal FifoDoWrite       : Std_Logic;

    Signal FifoWritePointer  : Integer Range 0 To ReplaceWith_FifoDepth - 1;

    Signal Signal_GND        : Std_Logic;
    Signal Signal_GND_Bus    : Std_Logic_Vector(ReplaceWith_FifoWidth - 1 Downto 0);

-- ----------------------------------------------------------------------------
Begin
-- ----------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    RD_EMPTY        <= FifoEmpty;         -- IncludeIf_RD_Empty
    RD_ALMOST_EMPTY <= FifoAlmostEmpty;   -- IncludeIf_RD_AlmostEmpty

    WR_FULL         <= FifoFull;          -- IncludeIf_WR_Full
    WR_ALMOST_FULL  <= FifoAlmostFull;    -- IncludeIf_WR_AlmostFull
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    FifoFull               <= '1' When WritePointer + 1 = ReadPointer Else '0';
    FifoEmpty              <= '1' When WritePointer     = ReadPointer Else '0';
    FifoAlmostEmpty        <= '1' When (WritePointer    = ((ReadPointer + ReplaceWith_FifoAlmostEmptyValue))) And FifoFull = '0' Else '0';
    FifoAlmostFull         <= '1' When (((WritePointer + 1 + ReplaceWith_FifoAlmostFullValue)) = ReadPointer) And FifoEmpty='0' Else '0';
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    FifoDoWrite <= '1' When (WR_EN='1' And FifoFull='0') Else '0';
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    process(WritePointer, ReadPointer)                                                                  -- IncludeIf_WR_FreeSlots
    begin                                                                                               -- IncludeIf_WR_FreeSlots
        if (WritePointer >= ReadPointer) then                                                           -- IncludeIf_WR_FreeSlots
            WR_FREE_SLOTS <= (((2**ReplaceWith_FifoAddressWidth) - 1) - (WritePointer - ReadPointer));  -- IncludeIf_WR_FreeSlots
        else                                                                               -- IncludeIf_WR_FreeSlots
            WR_FREE_SLOTS <= (((2**ReplaceWith_FifoAddressWidth) - 1) - ((2**ReplaceWith_FifoAddressWidth) - ReadPointer + WritePointer));  -- IncludeIf_WR_FreeSlots
        end if;                                                                            -- IncludeIf_WR_FreeSlots
    end process;                                                                           -- IncludeIf_WR_FreeSlots

    process(WritePointer, ReadPointer)                                                     -- IncludeIf_RD_DataCount
    begin                                                                                  -- IncludeIf_RD_DataCount
        if (WritePointer >= ReadPointer) then                                              -- IncludeIf_RD_DataCount
            RD_DATA_COUNT <= (WritePointer-ReadPointer);                                   -- IncludeIf_RD_DataCount
        else                                                                               -- IncludeIf_RD_DataCount
            RD_DATA_COUNT <= ((2**ReplaceWith_FifoAddressWidth)-ReadPointer+WritePointer); -- IncludeIf_RD_DataCount
        end if;                                                                            -- IncludeIf_RD_DataCount
    end process;                                                                           -- IncludeIf_RD_DataCount
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    Process(WR_CLK)
    Begin
         If rising_edge(WR_CLK) Then
             If WR_RST='1' Then
                 WritePointer <= (Others=>'0');
             Else
                 If FifoDoWrite = '1' Then
                    WritePointer <= (WritePointer + 1);
                 End If;
             End If;
         End If;
    End Process;
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    Process(RD_CLK)
    Begin
        If rising_edge(RD_CLK) Then
            If RD_RST='1' Then
                ReadPointer <= (Others=>'0');
            Else
                If RD_EN='1' And FifoEmpty='0' Then
                    ReadPointer <= (ReadPointer + 1);
                End If;
            End If;
        End If;
    End Process;
    -- ------------------------------------------------------------------------

-- ------------------------------------------------------------------------
    U_ReplaceWith_MemoryName : ReplaceWith_MemoryName Port Map (
        CLKA   => WR_CLK,
        CLKB   => RD_CLK,
        WEA    => FifoDoWrite,
        WEB    => Signal_GND,
        DINA   => WR_DATA,
        DINB   => Signal_GND_Bus,
        DOUTA  => Open,
        DOUTB  => RD_DATA,
        ADDRA  => WritePointer,
        ADDRB  => ReadPointer
    );
  -- ------------------------------------------------------------------------
    Signal_GND <= '0';
    Signal_GND_Bus <= (Others=>'0');

-- ----------------------------------------------------------------------------
End RTL;
-- ----------------------------------------------------------------------------

