-- ----------------------------------------------------------------------------
Library IEEE;
Use     IEEE.Std_Logic_1164.All;
Use     IEEE.Std_Logic_Unsigned.All;
Use     IEEE.Numeric_Std.All;
-- ----------------------------------------------------------------------------

------------------------------------------------------------------------------
Entity ReplaceWith_EntityName Is
    --------------------------------------------------------------------------
    Port (
    RD_EMPTY        : Out Std_Logic;                                                        -- IncludeIf_RD_Empty
    RD_ALMOST_EMPTY : Out Std_Logic;                                                        -- IncludeIf_RD_AlmostEmpty
    RD_DATA_COUNT   : Out Std_Logic_Vector(ReplaceWith_FifoAddressWidth - 1 Downto 0);      -- IncludeIf_RD_DataCount

    WR_FULL         : Out Std_Logic;                                                        -- IncludeIf_WR_Full
    WR_ALMOST_FULL  : Out Std_Logic;                                                        -- IncludeIf_WR_AlmostFull
    WR_FREE_SLOTS   : Out Std_Logic_Vector(ReplaceWith_FifoAddressWidth - 1 Downto 0);      -- IncludeIf_WR_FreeSlots

    RD_CLK          : In  Std_Logic;
    RD_RST          : In  Std_Logic;
    RD_EN           : In  Std_Logic;
    RD_DATA         : Out Std_Logic_Vector(ReplaceWith_FifoWidth - 1 Downto 0);

    WR_CLK          : In  Std_Logic;
    WR_RST          : In  Std_Logic;
    WR_EN           : In  Std_Logic;
    WR_DATA         : In  Std_Logic_Vector(ReplaceWith_FifoWidth - 1 Downto 0)
);
End;

--------------------------------------------------------------------------------
Architecture RTL Of ReplaceWith_EntityName Is

    ----------------------------------------------------------------------------
    function bin2gray ( bin : in std_logic_vector ) return std_logic_vector is
        variable bin_low : std_logic_vector(ReplaceWith_FifoAddressWidth - 2 downto 0);
    begin
        for n in 0 to ReplaceWith_FifoAddressWidth - 2 loop
            bin_low(n) := bin(n+1) xor bin(n);
        end loop;
        return  bin(ReplaceWith_FifoAddressWidth - 1) & bin_low;
    end function;

    ----------------------------------------------------------------------------
    function gray2bin (gray : in std_logic_vector ) return std_logic_vector is
        variable bin: std_logic_vector(ReplaceWith_FifoAddressWidth - 1 downto 0);
    begin
        bin(ReplaceWith_FifoAddressWidth - 1) := gray(ReplaceWith_FifoAddressWidth - 1);
        for n in ReplaceWith_FifoAddressWidth - 1 downto 1 loop
            bin(n-1) := bin(n) xor gray(n-1);
        end loop;
        return bin;
    end function;

    ----------------------------------------------------------------------------
    signal Readpointer              : std_logic_vector(ReplaceWith_FifoAddressWidth - 1 downto 0);
    signal WritePointer             : std_logic_vector(ReplaceWith_FifoAddressWidth - 1 downto 0);

    signal ReadPointer_Gray         : std_logic_vector(ReplaceWith_FifoAddressWidth - 1 downto 0);
    signal WritePointer_Gray        : std_logic_vector(ReplaceWith_FifoAddressWidth - 1 downto 0);

    signal WR_FULL_Reg              : std_logic;
    signal RD_EMPTY_Reg             : std_logic;

    Signal AlmostFull               : Std_Logic;
    Signal AlmostEmpty              : Std_Logic;

    signal Sync_ReadPointer_Gray    : std_logic_vector(ReplaceWith_FifoAddressWidth - 1 downto 0);
    signal Sync_ReadPointer_Gray0   : std_logic_vector(ReplaceWith_FifoAddressWidth - 1 downto 0);

    signal Sync_WritePointer_Gray   : std_logic_vector(ReplaceWith_FifoAddressWidth - 1 downto 0);
    signal Sync_WritePointer_Gray0  : std_logic_vector(ReplaceWith_FifoAddressWidth - 1 downto 0);

    signal Sync_WR_RST              : std_logic;
    signal Sync_WR_RST0             : std_logic;

    signal Sync_RD_RST              : std_logic;
    signal Sync_RD_RST0             : std_logic;

    signal ReadRst                  : std_logic;
    signal WriteRst                 : std_logic;

    signal Sync_WR_EN               : std_logic;
    signal Sync_WR_EN0              : std_logic;

    signal Sync_RD_EN               : std_logic;
    signal Sync_RD_EN0              : std_logic;

    signal ReadCount                : std_logic_vector(ReplaceWith_FifoAddressWidth - 1 downto 0);
    signal WriteCount               : std_logic_vector(ReplaceWith_FifoAddressWidth - 1 downto 0);

    Signal FifoDoWrite              : Std_Logic;

    type   TFifoMemory is array(((2**ReplaceWith_FifoAddressWidth)-1) downto 0) of std_logic_vector(ReplaceWith_FifoWidth - 1 downto 0);
    signal FifoMemory : TFifoMemory;


begin

    -- -------------------------------------------------------------------------
    -- Syncronize resets
    process(RD_CLK)
    begin
        if rising_edge(RD_CLK) then
            Sync_WR_RST0 <= WR_RST;
            Sync_WR_RST  <= Sync_WR_RST0;
            ReadRst      <= RD_RST or Sync_WR_RST;
        end if;
    end process;

    process(WR_CLK)
    begin
        if rising_edge(WR_CLK) then
            Sync_RD_RST0 <= RD_RST;
            Sync_RD_RST  <= Sync_RD_RST0;
            WriteRst     <= WR_RST or Sync_RD_RST;
        end if;
    end process;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    -- Read and Write pointers
    process(RD_CLK)
    begin
        if rising_edge(RD_CLK) then
            if ReadRst='1' then
                ReadPointer      <= (others=>'0');
                ReadPointer_Gray <= (others=>'0');
            else
                if RD_EN='1' and RD_EMPTY_Reg='0' then
                    ReadPointer_Gray <= bin2gray(ReadPointer+1);
                    ReadPointer      <= ReadPointer + 1;
                end if;
            end if;
        end if;
    end process;

    process(WR_CLK)
    begin
        if rising_edge(WR_CLK) then
            if WriteRst='1' then
                WritePointer      <= (others=>'0');
                WritePointer_Gray <= (others=>'0');
            else
                if WR_EN='1' and WR_FULL_Reg='0' then
                    WritePointer_Gray <= bin2gray(WritePointer+1);
                    WritePointer <= WritePointer + 1;
                end if;
            end if;
        end if;
    end process;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    -- Resyncronize pointers between clock domains (R to W)
    process(WR_CLK)
    begin
        if rising_edge(WR_CLK) then
            if WriteRst='1' then
                Sync_ReadPointer_Gray0 <= (others=>'0');
                Sync_ReadPointer_Gray  <= (others=>'0');
            else
                Sync_ReadPointer_Gray0 <= ReadPointer_Gray;
                Sync_ReadPointer_Gray  <= Sync_ReadPointer_Gray0;
            end if;
        end if;
    end process;
    -- -------------------------------------------------------------------------
    WR_ALMOST_FULL  <= '1' when bin2gray(WritePointer + 1 + ReplaceWith_FifoAlmostFullValue) = Sync_ReadPointer_Gray  else '0'; -- IncludeIf_WR_AlmostFull
    RD_ALMOST_EMPTY <= '1' when bin2gray(ReadPointer  + ReplaceWith_FifoAlmostEmptyValue)    = Sync_WritePointer_Gray else '0'; -- IncludeIf_RD_AlmostEmpty
    -- -------------------------------------------------------------------------
    process(WR_CLK, Sync_WritePointer_Gray, Sync_ReadPointer_Gray)                                                                                                              -- IncludeIf_WR_FreeSlots
    begin                                                                                                                                                                       -- IncludeIf_WR_FreeSlots
        if (gray2bin(Sync_WritePointer_Gray) >= gray2bin(Sync_ReadPointer_Gray)) then                                                                                           -- IncludeIf_WR_FreeSlots
           WR_FREE_SLOTS <= (((2**ReplaceWith_FifoAddressWidth)-1)-(gray2bin(Sync_WritePointer_Gray)-gray2bin(Sync_ReadPointer_Gray)));                                         -- IncludeIf_WR_FreeSlots
        else                                                                                                                                                                    -- IncludeIf_WR_FreeSlots
           WR_FREE_SLOTS <= (((2**ReplaceWith_FifoAddressWidth)-1)-(((2**ReplaceWith_FifoAddressWidth)-gray2bin(Sync_ReadPointer_Gray))+gray2bin(Sync_WritePointer_Gray)));     -- IncludeIf_WR_FreeSlots
        end if;                                                                                                                                                                 -- IncludeIf_WR_FreeSlots
    end process;                                                                                                                                                                -- IncludeIf_WR_FreeSlots

    process(RD_CLK, Sync_WritePointer_Gray, Sync_ReadPointer_Gray)                                                                                                              -- IncludeIf_RD_DataCount
    begin                                                                                                                                                                       -- IncludeIf_RD_DataCount
        if (gray2bin(Sync_WritePointer_Gray) >= gray2bin(Sync_ReadPointer_Gray)) then                                                                                           -- IncludeIf_RD_DataCount
           RD_DATA_COUNT <= (gray2bin(Sync_WritePointer_Gray)-gray2bin(Sync_ReadPointer_Gray));                                                                                 -- IncludeIf_RD_DataCount
        else                                                                                                                                                                    -- IncludeIf_RD_DataCount
           RD_DATA_COUNT <= (((2**ReplaceWith_FifoAddressWidth)-gray2bin(Sync_ReadPointer_Gray))+gray2bin(Sync_WritePointer_Gray));                                             -- IncludeIf_RD_DataCount
        end if;                                                                                                                                                                 -- IncludeIf_RD_DataCount
    end process;                                                                                                                                                                -- IncludeIf_RD_DataCount

    -- -------------------------------------------------------------------------
    process(WR_CLK)
    begin
        if rising_edge(WR_CLK) then
            if WriteRst='1' then
                WR_FULL_Reg <= '0';
            else
                WR_FULL_Reg <= '0';
                if WR_EN='1' then
                    if bin2gray(WritePointer+2) = Sync_ReadPointer_Gray then
                        WR_FULL_Reg <= '1';
                    end if;
                else
                    if bin2gray(WritePointer+1) = Sync_ReadPointer_Gray then
                        WR_FULL_Reg <= '1';
                    end if;
               end if;
            end if;
        end if;
    end process;

    WR_FULL <= WR_FULL_Reg;                                                                 -- IncludeIf_WR_Full

    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    -- Resyncronize pointers between clock domains (W to R)
    process(RD_CLK)
    begin
        if rising_edge(RD_CLK) then
            if ReadRst='1' then
                Sync_WritePointer_Gray0 <= (others=>'0');
                Sync_WritePointer_Gray  <= (others=>'0');
            else
                Sync_WritePointer_Gray0 <= WritePointer_Gray;
                Sync_WritePointer_Gray  <= Sync_WritePointer_Gray0;
            end if;
        end if;
    end process;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    process(RD_CLK)
    begin
        if rising_edge(RD_CLK) then
            if ReadRst='1' then
                RD_EMPTY_Reg <= '1';
            else
                RD_EMPTY_Reg <= '0';
                if RD_EN='1' then
                    if bin2gray(ReadPointer-2) = Sync_WritePointer_Gray then
                        RD_EMPTY_Reg <= '1';
                    end if;
                else
                    if bin2gray(ReadPointer) = Sync_WritePointer_Gray then
                        RD_EMPTY_Reg <= '1';
                   end if;
               end if;
            end if;
        end if;
    end process;

    RD_EMPTY <= RD_EMPTY_Reg;                                                               -- IncludeIf_RD_Empty

    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    process(RD_CLK)
    begin
        if rising_edge(RD_CLK) then
            RD_DATA <= FifoMemory(Conv_Integer(ReadPointer));
        end if;
    end process;

    process(WR_CLK)
    begin
        if rising_edge(WR_CLK) then
            if WR_EN='1' And WR_FULL_Reg='0' then
                    FifoMemory(Conv_Integer(WritePointer)) <= WR_DATA;
            end if;
        end if;
    end process;
    -- -------------------------------------------------------------------------


end rtl;
