/**************************************************************************
**                                                                        *
**  FILE        :  wcsftime.c                                             *
**                                                                        *
**  DESCRIPTION :  Source file for wcsftime routine.                      *
**                                                                        *
**  Copyright 1996-2010 Altium BV                                         *
**                                                                        *
**************************************************************************/

#include <wchar.h>

#ifdef WCHAR_SUPPORT_ENABLED

extern char * _tzget(int);
extern wchar_t * wmemcpy(wchar_t * restrict s1, const wchar_t * restrict s2, size_t n);

static const wchar_t * convert_this(wchar_t format, const struct tm *timeptr);
static wchar_t * strnum(wchar_t *s, int value, int length);


/*
 * Local time values for an English-speaking country.
 */
static const wchar_t *
_ampm_nm[2] =
	{ L"AM", L"PM" 
	};

static const wchar_t *
_swday_nm[7] =
	{ L"Sun", L"Mon", L"Tue", L"Wed", L"Thu", L"Fri", L"Sat" 
	};

static const wchar_t *
_lwday_nm[7] =
	{ L"Sunday",   L"Monday", L"Tuesday", L"Wednesday",
	  L"Thursday", L"Friday", L"Saturday" 
	};

static const wchar_t *
_smon_nm[12] =
	{ L"Jan", L"Feb", L"Mar", L"Apr", L"May", L"Jun",
	  L"Jul", L"Aug", L"Sep", L"Oct", L"Nov", L"Dec" 
	};

static const wchar_t *
_lmon_nm[12] =
	{ L"January",   L"February", L"March",    L"April",
	  L"May",       L"June",     L"July",     L"August",
	  L"September", L"October",  L"November", L"December" 
	};

static const wchar_t *
_fmt_nm[3] =
	{ L"%a %b %d %H:%M:%S %Y", L"%a %b %d, %Y", L"%H:%M:%S" 
	};

/*
 * Place wide characters into the array pointed to by s as controlled by 
 * the wide character string pointed to by format.
 * 
 * Return 0 if maxsize is too short; otherwise
 * return the number of wide characters placed into the wide character 
 * array pointed to by s, not including the terminating null.
 */
size_t wcsftime(wchar_t * restrict s, size_t maxsize, const wchar_t * restrict format,
		const struct tm * restrict timeptr)
{
	const wchar_t	*pts;
	size_t		n_wchars;
	size_t		result		= maxsize;
	wchar_t		buf[32];

	while (*format && maxsize > 1)
	{
		for (pts = format; *format && maxsize > 0; format++, maxsize--)
		{
			/* Search for the L'%' token in the format string.
			 */
			if (*format == L'%')
			{
				break;
			}
		}

		n_wchars = format - pts;

		if (maxsize > 0 && n_wchars > 0)
		{
			/* Add the wide characters before L'%' or before L'\0' to s.
			 */
			wmemcpy(s, pts, n_wchars);
			s += n_wchars;
		}

		if (maxsize > 0 && *format == L'%')
		{
			/* Do the conversion.
			 */
			format++;
			pts = convert_this(*format, timeptr);
			format++;

			if (pts)
			{
				if (pts == _fmt_nm[0] || pts == _fmt_nm[1] || pts == _fmt_nm[2])
				{
					/* Call this function for the format in _fmt_nm[].
					 */
					if (wcsftime(buf, 32, pts, timeptr) == 0) return 0;
					pts = (const wchar_t *) buf;
				}

				n_wchars = wcslen(pts);
				maxsize -= n_wchars;

				if (maxsize < 1)
				{
					/* maxsize was too short, so return zero.
					 */
					result = maxsize = 0;
				}
				else if (n_wchars > 0)
				{
					/* Add the formatted string to s.
					 */
					wmemcpy(s, pts, n_wchars);
					s += n_wchars;
				}
			}
		}
		else if ( maxsize <= 0 )
		{
			result = maxsize = 0;
		}
	}

	/* Add the terminating null character.
	 */
	*s = L'\0';

	return result - maxsize;
}

/*
 * Return a wide character string dependent of the given format specifier.
 */
static const wchar_t *convert_this(wchar_t format, const struct tm *timeptr)
{
	const wchar_t	*pts;
	static wchar_t	buf[8];
	char		*ptc;

	switch (format)
	{
	case L'a':
		/* Abbreviated weekday name.
		 */
		pts = _swday_nm[timeptr->tm_wday];
		break;
	case L'A':
		/* Full weekday name.
		 */
		pts = _lwday_nm[timeptr->tm_wday];
		break;
	case L'b':
		/* Abbreviated month name.
		 */
		pts = _smon_nm[timeptr->tm_mon];
		break;
	case L'B':
		/* Full month name.
		 */
		pts = _lmon_nm[timeptr->tm_mon];
		break;
	case L'c':
		/* Date and time representation.
		 */
		pts = _fmt_nm[0];
		break;
	case L'd':
		/* Day of the month as decimal number (01-31).
		 */
		pts = strnum(buf, timeptr->tm_mday, 2);
		break;
	case L'H':
		/* Hour (24-hour clock) as a decimal number (00-23).
		 */
		pts = strnum(buf, timeptr->tm_hour, 2);
		break;
	case L'I':
		/* Hour (12-hour clock) as a decimal number (01-12).
		 */
		pts = strnum(buf,
			     (timeptr->tm_hour > 12 ? timeptr->tm_hour - 12 : (timeptr->tm_hour == 0 ? 12 : timeptr->tm_hour)),
			     2);
		break;
	case L'j':
		/* Day of the year as a decimal number (001-366).
		 */
		pts = strnum( buf, timeptr->tm_yday + 1, 3);
		break;
	case L'm':
		/* Month as a decimal number (01-12).
		 */
		pts = strnum(buf, timeptr->tm_mon + 1, 2);
		break;
	case L'M':
		/* Minute as a decimal number (00-59).
		 */
		pts = strnum(buf, timeptr->tm_min, 2);
		break;
	case L'p':
		/* AM/PM designations associated with a 12-hour clock.
		 */
		if (timeptr->tm_hour < 12)
		{
			pts = _ampm_nm[0];
		}
		else
		{
			pts = _ampm_nm[1];
		}
		break;
	case L'S':
		/* Second as a decimal number (00-59).
		 */
		pts = strnum(buf, timeptr->tm_sec, 2);
		break;
	case L'U':
		/* Week number of the year as a decimal number (00-53).
		 * The first Sunday as the first day of week 1.
		 */
		pts = strnum(buf, (timeptr->tm_yday - ((timeptr->tm_wday + 7) % 7) + 12) / 7 - 1, 2);
		break;
	case L'w':
		/* Weekday as a decimal number (0-6), where Sunday is 0.
		 */
		pts = strnum(buf, timeptr->tm_wday, 1);
		break;
	case L'W':
		/* Week number of the year as a decimal number (00-53).
		 * The first Monday as the first day of week 1.
		 */
		pts = strnum(buf, (timeptr->tm_yday - ((timeptr->tm_wday + 6) % 7) + 12) / 7 - 1, 2);
		break;
	case L'x':
		/* Date representation.
		 */
		pts = _fmt_nm[1];
		break;
	case L'X':
		/* Time representation.
		 */
		pts = _fmt_nm[2];
		break;
	case L'y':
		/* Year without century as a decimal number (00-99).
		 */
		pts = strnum(buf, timeptr->tm_year % 100, 2);
		break;
	case L'Y':
		/* Year with century as a decimal number.
		 */
		pts = strnum(buf, timeptr->tm_year + 1900, 4);
		break;
	case L'Z':
		/* Time zone name or abbreviation.
		 */
		if (timeptr->tm_isdst > 0)
		{
			ptc = _tzget(1);
		}
		else
		{
			ptc = _tzget(0);
		}
		if (ptc)
		{
			/* Convert to wide character string.
			 */
			for (pts = buf; *ptc != '\0'; ptc++, pts++)
			{
				* (wchar_t *) pts = (wchar_t) *ptc;
			}
			* (wchar_t *) pts = L'\0';
			pts = buf;
		}
		else
		{
			pts = NULL;
		}
		break;
	case L'%':
		/* Just print the wide character L'%'.
		 */
		pts = (wchar_t *) &(L"%");
		break;
	default:
		/* Unknown format, just skip it.
		 */
		pts = NULL;
	}

	return pts;
}

/*
 * Return the wide character string representation of the given integer value.
 */
static wchar_t *strnum(wchar_t *s, int value, int length)
{
	if (value < 0)
	{
		value = 0;
	}

	s += length;
	*s = L'\0';

	while (length--)
	{
		*--s = value % 10 + '0';
		value /= 10;
	}

	return s;
}

#endif
