/**************************************************************************
**                                                                        *
**  FILE        :  sinh.c                                                 *
**                                                                        *
**  DESCRIPTION :  Source file for sinh() routine                         *
**                 Computes the 'hyperbolic sinus' for the given angle    *
**                 (in radians)                                           *
**                                                                        *
**  Copyright 1996-2010 Altium BV                                         *
**                                                                        *
**************************************************************************/

#include <math.h>
/*
	sinh(arg) returns the hyperbolic sine of its floating-
	point argument.

	The exponential function is called for arguments
	greater in magnitude than 0.5.

	A series is used for arguments smaller in magnitude than 0.5.
	The double coefficients are #2027 from Hart & Cheney. (16.45D)
	The single coefficients are #2021 from Hart & Cheney. ( 6.98D)

	cosh(arg) is computed from the exponential function for
	all arguments.
*/

#ifndef __SINGLE_FP__
#pragma alias	sinhl	= _sinh_dpf
#pragma alias	sinh		= _sinh_dpf


#define	P0  ( 0.6282813250833546179346e4)
#define	P1  ( 0.9178748106756360272984e3)
#define	P2  ( 0.3181332190110303793958e2)
#define	P3  ( 0.336083852466226007246e0)
#define	Q0  ( 0.6282813250833546241924e4)
#define	Q1  (-0.1292607311299731203424e3)
#define POLYNOMIAL(argsq,arg)	( ( ( ( ( P3 * (argsq) + P2 ) * (argsq) + P1 ) * (argsq) + P0 ) * (arg) ) / \
	( ( (argsq) + Q1 ) * (argsq) + Q0 ) )
#define LOG_HALF (-0.693147180559945309417232121) /* ln(0.5) */

static
double
_sinh_dpf( double arg )
{
	double temp, argsq;
	register signed char sign = 0;

	sign = 1;
	if( arg < 0.0 )
	{
		arg = -arg;
		sign = -1;
	}

	if( arg > 37.0 )
	{
		temp = exp( arg + LOG_HALF );
			
		if ( sign == 1 )
			return( temp );
		else
			return( -temp );
	}

	if( arg > 0.5 )
	{
		return( sign * ( exp( arg + LOG_HALF ) - exp( -arg + LOG_HALF ) ) );
	}

	argsq = arg*arg;
	temp = POLYNOMIAL(argsq,arg);
	return( sign * temp );
}

#endif

