/**************************************************************************
**                                                                        *
**  FILE        :  _strtol.h                                              *
**                                                                        *
**  DESCRIPTION :  Common code used for the following functions:          *
**                 strtol()                                               *
**                 wcstol()                                               *
**                 strtoll()                                              *
**                 wcstoll()                                              *
**                                                                        *
**  Copyright 1996-2010 Altium BV                                         *
**                                                                        *
**************************************************************************/

#include <errno.h>
#include <limits.h>

#define	NEGATIVE	0x1	/* Negative sign read	*/
#define	READ		0x2	/* Done conversion	*/
#define	ERROR		0x4	/* Out of range		*/

value_t FUNCTION( const char_t * restrict str, char_t ** restrict ptr, int base )
{
	value_t		result	= 0;
	ch_t		ch;
	unsigned char	done	= 0;

	/* Set condition for no conversion
	 */
	if ( ptr )
	{
		*ptr = (char_t *) str;
	}

	/* Skip all whitespace
	 */
	do
	{
		ch = *str++;
	} while ( ((ch >= 9) && (ch <= 13)) || (ch == ' ' ) );

	/* Check for sign
	 */
	if ( ch == '-' )
	{
		done |= NEGATIVE;
	}
	else if ( ch != '+' )
	{
		str--;
	}

	if ( base == 0 || base == 16 )
	{
		if ( *str != '0' )
		{
			/* Set base only when not specified already
			 */
			if ( base == 0 )
			{
				base = 10;
			}
		}
		else
		{
			/* Base should be octal or hexadecimal
			 */
			if ( str[1] == 'x' || str[1] == 'X' )
			{
				/* hexadecimal */
				done |= READ;
				base  = 16;
				str  += 2;
			}
			else
			{
				/* octal */
				if ( base == 0 )
				{
					base = 8;
				}
			}
		}
	}

	/* Read the number
	 */
	for ( ;; )
	{
		ch = *( str++ );

		/* First check numbers, they are most used, keep the routine
		 * fast
		 */
		ch =	((ch >= '0') && (ch < '0'+10) && (ch < '0'+base)) ? ch - '0' :
			((ch >= 'a') && (ch < 'a'-10+base)) ? ch - 'a' + 10 :
			((ch >= 'A') && (ch < 'A'-10+base)) ? ch - 'A' + 10 :
			255;

		/* Unknown character found
		 */
		if ( ch == 255 )
		{
			break;
		}

		if ( result && ((base * ((result - 1) >> 8)) & ~ (MAX_VALUE >> 8)) )
		{
			done |= ERROR;
		}
		result = base * result + ch;

		/* Check for overflow
		 */
		if (  result < 0 
		   && ! ((done & NEGATIVE) && result == MIN_VALUE)
		   )
		{
			done |= ERROR;
		}
		done |= READ;
	}

	/* No conversion done
	 */
	if ( !(done & READ) )
	{
		return (value_t) 0;
	}

	/* Pointer to remaining characters
	 */
	if ( ptr )
	{
		*ptr = (char_t *) (str - 1);
	}

	if ( done & ERROR )
	{
		errno = ERANGE;
		return (value_t) ( done & NEGATIVE ? MIN_VALUE : MAX_VALUE );
	}

	return (value_t) ( done & NEGATIVE ? -result : result );
}
