/**************************************************************************
**                                                                        *
**  FILE        :  _getflt.h                                              *
**                                                                        *
**  DESCRIPTION :  Common code for the _getfloat()/_getwflt() functions   *
**                                                                        *
**  Copyright 1996-2010 Altium BV                                         *
**                                                                        *
**************************************************************************/

#include <stdio.h>
#include <stdint.h>
#include <limits.h>
#include <io.h>
#include <errno.h>
#include <float.h>
#include <math.h>

#include "fpbits.h"
#include "_printflags.h"

#if UINT_MAX == 0xffffffffUL
# define BUFBITS 32
#elif UINT_MAX == 0xffffUL
# define BUFBITS 16
#endif

/*
 * Decimal exponent scanning and processing seems trivial but isn't. For
 * example, using pow(10.0, ...) to calculate the exponent value literally
 * gives an exponentially large rounding error. The binary refactoring
 * getflt() uses is much better but still not exact: the result can be off
 * by 1, 2 and (occasionally) 3 units.
 */
#if LDBL_MAX_10_EXP >= 256 && LDBL_MAX_10_EXP < 512	/* 308 */
static const long double posexp[] = { 1e+1, 1e+2, 1e+4, 1e+8, 1e+16, 1e+32, 1e+64, 1e+128, 1e+256 };
static const long double negexp[] = { 1e-1, 1e-2, 1e-4, 1e-8, 1e-16, 1e-32, 1e-64, 1e-128, 1e-256 };
#elif LDBL_MAX_10_EXP >= 32 && LDBL_MAX_10_EXP < 64	/* 38, -D__SINGLE_FP__ */
static const long double posexp[] = { 1e+1, 1e+2, 1e+4, 1e+8, 1e+16, 1e+32 };
static const long double negexp[] = { 1e-1, 1e-2, 1e-4, 1e-8, 1e-16, 1e-32 };
#endif

struct input
{
	unsigned int	length;		/* length modifier after % */
	unsigned int *	n_chars;	/* for implementing %n in upper layer */
	struct _io *	fp;
};

/*
 * Return the next char if there is any in the input. When the length
 * modifier is exhausted we pretend end of file.
 */
static int_t getinput(struct input *p)
{
	int_t	c;

	c = E_O_F;
	if (p->length)
	{
		c = FGETC(p->fp);
		if (c != E_O_F)
		{
			p->length--;
			(*p->n_chars)++;
		}
	}
	return c;
}

static void ungetinput(struct input *p, int_t c)
{
	if (c != E_O_F)
	{
		p->length++;
		(*p->n_chars)--;
		UNGETC(c, p->fp);
	}
}

static int getinf(struct input *input, int_t sign, long double *val);
static int getnan(struct input *input, int_t sign, long double *val);
static int gethex(struct input *input, int_t sign, long double *val);
static int getflt(struct input *input, int_t sign, long double *val, int_t c);

/* Return zero on success */
int GETFLOAT( uchar_t ch, unsigned short type, unsigned int length,
              void * ap, struct _io * fp, unsigned int *n_chars )
{
	int_t		c, c2, sign;
	int		error;
	long double	value = 0;
	struct input	input;

	while (c = FGETC(fp), ISSPACE(c))
	{
		(*n_chars)++;
	}
	UNGETC(c, fp);
	input.length = length ? length : ~0U;
	input.n_chars = n_chars;
	input.fp = fp;

	c = getinput(&input);
	sign = 0;
	if (c == '-' || c == '+')
	{
		sign = c;
		c = getinput(&input);
	}

	switch (c)
	{
	case 'i':
	case 'I':
		error = getinf(&input, sign, &value);
		break;
	case 'n':
	case 'N':
		error = getnan(&input, sign, &value);
		break;
	case '0':
		c2 = getinput(&input);
		if (c2 == 'x' || c2 == 'X')
		{
			error = gethex(&input, sign, &value);
			break;
		}
		ungetinput(&input, c2);
		/* fall through */
	default:
		error = getflt(&input, sign, &value, c);
		break;
	}

	if (!error && ap)
	{
		if ( ch == 'f' )
		{
			if ( ( type & FLG_LONG_LONG ) == FLG_LONG )
			{
				*(double *)ap = value;
			}
			else if ( type & FLG_DOUBLE )
			{
				*(long double *)ap = value;
			}
			else
			{
				*(float *)ap = value;
			}
		}
	}
	return error;
}

static int getinf(struct input *input, int_t sign, long double *val)
{
	const char_t	*str = LITERAL("nfinite");
	int_t		c, expect;

	while (1)
	{
		c = getinput(input);
		expect = (uchar_t) *str++;
		if (!expect)
		{
			break;
		}
		if (expect != (ISUPPER(c) ? TOLOWER(c) : c))
		{
			if (expect == 'i')
			{
				expect = 0;	/* accept "inf" without trailing "inite" */
			}
			break;
		}
	}
	ungetinput(input, c);
	if (expect)
	{
		return 1;
	}
	{
#ifndef	__SINGLE_FP__
		double2longs_t	value;

		value.s.lo = 0UL;
		value.s.hi = PUT_DOUBLE_EXPONENT(PUT_DOUBLE_SIGN(0UL, sign == '-'), DOUBLE_BIASED_INF_EXP);
		*val = value.d;
#else
		float2long_t	value;

		value.l = PUT_FLOAT_EXPONENT(PUT_FLOAT_SIGN(0UL, sign == '-'), FLOAT_BIASED_INF_EXP);
		*val = value.f;
#endif
	}
	return 0;
}

/*
 * Unsupported formats:
 *
 *	input text		equivalent nanl() call
 *	----------------------------------------------
 *	NAN()			nanl("")
 *	NAN("n-char-sequence")	nanl("n-char-sequence")
 */
static int getnan(struct input *input, int_t sign, long double *val)
{
	const char_t	*str = LITERAL("an");
	int_t		c, expect;

	while (1)
	{
		c = getinput(input);
		expect = (uchar_t) *str++;
		if (!expect)
		{
			break;
		}
		if (expect != (ISUPPER(c) ? TOLOWER(c) : c))
		{
			break;
		}
	}
	ungetinput(input, c);
	if (expect)
	{
		return 1;
	}
#ifndef	__SINGLE_FP__
		double2longs_t	value;

		value.s.lo = 1UL;
		value.s.hi = PUT_DOUBLE_EXPONENT(0UL, DOUBLE_BIASED_NaN_EXP);
		*val = value.d;
#else
		float2long_t	value;

		value.l = PUT_FLOAT_EXPONENT(1UL, FLOAT_BIASED_NaN_EXP);
		*val = value.f;
#endif
	return 0;
}

static int gethex(struct input *input, int_t sign, long double *val)
{
	int_t		c;
	uint_least8_t	n;
	_Bool		error = 1;
	uint_least8_t	scale = 0;
	long double	value = 0.0;
	unsigned int	buf = 0;	/* must contain >= BUFBITS bits */
	uint_least8_t	bits = 0;
	int		exp = 0;

	while (1)
	{
		c = getinput(input);
		if (ISXDIGIT(c))
		{
			error = 0;
			if (ISDIGIT(c))
			{
				n = c - '0';
			}
			else
			{
				n = 10 + (ISUPPER(c) ? TOLOWER(c) : c) - 'a';
			}
			buf = (buf << 4) | n;
			bits += 4;
			if (bits == (BUFBITS & ~3))
			{
				value = ldexpl(value, bits) + buf;
				bits = 0;
				buf = 0;
			}
			exp -= scale;
		}
		else
		{
			if (c != '.' || scale)
			{
				break;
			}
			scale = 4;
		}
	}
	value = ldexpl(value, bits) + buf;
	value = ldexpl(value, exp);
	if (sign == '-')
	{
		value = -value;
	}
	if (!error && (c == 'p' || c == 'P'))
	{
		c = getinput(input);
		sign = 0;
		if (c == '-' || c == '+')
		{
			sign = c;
			c = getinput(input);
		}
		error = 1;		/* exponent number required, let's see.. */
		exp = 0;
		while (ISDIGIT(c))
		{
			error = 0;
			if (exp < (INT_MAX - 9) / 10)
			{
				exp = exp * 10 + (c - '0');
			}
			c = getinput(input);
		}
		if (sign == '-')
		{
			exp = -exp;
		}
		value = ldexpl(value, exp);
	}
	*val = value;
	ungetinput(input, c);
	return error;
}

static int getflt(struct input *input, int_t sign, long double *val, int_t c)
{
	int		error;
	long double	value, scale;
	int		exp, i;

	value = 0.0;
	error = 1;			/* mantissa required, let's see */
	while (ISDIGIT(c))
	{
		error = 0;
		value = value * 10.0 + (c - '0');
		c = getinput(input);
	}
	if (c == '.')
	{
		scale = 0.1;
		while (c = getinput(input), ISDIGIT(c))
		{
			error = 0;
			value += scale * (c - '0');
			scale *= 0.1;
		}
	}
	if (sign == '-')
	{
		value = -value;
	}
	if (!error && (c == 'e' || c == 'E'))
	{
		c = getinput(input);
		sign = 0;
		if (c == '-' || c == '+')
		{
			sign = c;
			c = getinput(input);
		}
		error = 1;		/* exponent number required, let's see.. */
		exp = 0;
		while (ISDIGIT(c))
		{
			error = 0;
			if (exp < (INT_MAX - 9) / 10)	/* saturate here, FP will overflow */
			{
				exp = exp * 10 + (c - '0');
			}
			c = getinput(input);
		}
		for (i = sizeof (posexp) / sizeof (*posexp) - 1; i >= 0; --i)
		{
			if (exp >= (1 << i))
			{
				exp -= 1 << i;
				value *= sign == '-' ? negexp[i] : posexp[i];
			}
		}
	}
	*val = value;
	ungetinput(input, c);
	return error;
}
