/**************************************************************************
**                                                                        *
**  FILE        :  _doflt.h                                               *
**                                                                        *
**  DESCRIPTION :  Common code for the _doflt()/_dowflt() functions       *
**                                                                        *
**  Copyright 1996-2010 Altium BV                                         *
**                                                                        *
**************************************************************************/

#include <float.h>
#include <math.h>
#include <stdio.h>
#include <io.h>

#include "fpbits.h"
#include "_printflags.h"

#ifndef	__SINGLE_FP__
#define	NDIG	(DBL_DIG+3)
#define EXP	0x7FF
#define BIAS	DOUBLE_BIAS
#else
#define	NDIG	(FLT_DIG+3)
#define EXP	0xFF
#define BIAS	FLOAT_BIAS
#endif

/* This macro can be used to check whether or not the given conversion
 * specifier is of the given type. The character 'ch' needs to be lower case
 * for this macro to function properly.
 */
#define	IS_TYPE(type,ch)	((type) == (ch) || (type) == ((ch) - 'a' + 'A'))

/* min		-> minimum output width
 * precision	-> number of characters after the '.'
 * type		-> conversion character ('e', 'E', 'f', 'F', 'g', 'G', 'a', 'A', 'r', 'R')
 * format	-> formatting flags
 * number	-> floating point value to print
 * fout		-> file to print to
 * characters	-> pointer to emitted character count
 */
void DOFLT( int min, int precision, uchar_t type, unsigned short format,
            long double number, struct _io *fout, unsigned int *characters )
{
#ifndef	__SINGLE_FP__
	double2longs_t		conv_value;
	unsigned long		lo_fract	= 0;
	unsigned long		hi_fract	= 0;
#else
	float2long_t		conv_value;
#endif
	unsigned long		fraction	= 0;
	int			exponent = 0;
	char_t			expbuf[5];		/* let compiler overlay this with buf[] */
	unsigned char		explen = 1;
	unsigned char		is_zero = 0;
	unsigned char		is_nan = 0;
	unsigned char		is_inf = 0;
	uchar_t			vsign;			/* sign of the value	*/
	int			decpt		= 0;
	int			i;
	int			j;
	unsigned int		length		= 0;	/* To determine the length of the string to print */
	double			fj;
	double			multiplier	= 1.0;
	char_t*			p;
	char_t* 		q;
	char_t* 		infnan		= NULL;
	unsigned char		first_nonzero	= 0;
	char_t			buf[ NDIG ];		/* Room to print ascii representation into */
	double			arg		= (double) number;

#ifndef	__SINGLE_FP__
	conv_value.d = arg;
	lo_fract     = GET_DOUBLE_LO_MANTISSA(conv_value.s.lo);
	hi_fract     = GET_DOUBLE_HI_MANTISSA(conv_value.s.hi);
#else
	conv_value.f = arg;
	fraction     = GET_FLOAT_MANTISSA(conv_value.l);
#endif

	/* Adjust "precision" as prescribed by the C standards */
	if ( !precision && IS_TYPE(type, 'g') )
	{
		precision = 1;
	}
	else if ( !( format & FLG_MAX_LENGTH ) && IS_TYPE(type, 'a') )
	{
		/* Set precision to the number of hexdigits necessary for a
		 * precise representation of the fractional part. This assumes
		 * nibble aligned mantissa bits which is correct (52)
		 */
#ifndef	__SINGLE_FP__
		precision = (DOUBLE_FRACTION_SIZE + 3) / 4;

		for ( i = 0; i < DOUBLE_FRACTION_SIZE; i += 4 )
		{
			if ( i < (8 * sizeof(unsigned long)) )
			{
				/* low-part of the fraction */
				if ( (( lo_fract >> i ) & 0xF ) != 0 )
				{
					break;
				}
			}
			else
			{
				/* high-part of the fraction */
				if ( (( hi_fract >> (i - (8 * sizeof(unsigned long))) ) & 0xF ) != 0 )
				{
					break;
				}
			}
			precision--;
		}
#else
		precision = (FLOAT_FRACTION_SIZE + 3) / 4;

		for ( i = 0; i < FLOAT_FRACTION_SIZE; i += 4 )
		{
			if ( (( fraction >> i ) & 0xF ) != 0 )
			{
				break;
			}
			precision--;
		}
#endif
	}

	/* 0 - positive value, otherwise negative value
	 */
	vsign = 0;

	/* Check the sign, make sure the value gets positive
	 */
	if ( signbit(arg) )
	{
		vsign = '-';
		arg   = -arg;
	}

	if ( !vsign )
	{
		if ( (format & (FLG_ALWAYS_SIGN | FLG_SIGN_OR_SPACE)) == FLG_ALWAYS_SIGN )
		{
			/* sign always	*/
			vsign = '+';
		}
		else if ( (format & (FLG_ALWAYS_SIGN | FLG_SIGN_OR_SPACE)) == FLG_SIGN_OR_SPACE )
		{
			/* sign or space */
			vsign = ' ';
		}
	}

	/* Do everything necessary for determining the "length" but don't
	 * output anything yet -- need to print sign and handle right alignment
	 * before doing that.
	 */
	if (vsign)
	{
		length++;
	}

#ifndef	__SINGLE_FP__
	exponent = GET_DOUBLE_EXPONENT(conv_value.s.hi);
	is_nan  = ( exponent == DOUBLE_BIASED_NaN_EXP &&  (hi_fract || lo_fract));
	is_inf  = ( exponent == DOUBLE_BIASED_INF_EXP && !(hi_fract || lo_fract));
#else
	exponent = GET_FLOAT_EXPONENT(conv_value.l);
	is_nan =  ( exponent == FLOAT_BIASED_NaN_EXP &&  fraction);
	is_inf =  ( exponent == FLOAT_BIASED_INF_EXP && !fraction);
#endif
	is_zero = !exponent;

	if (is_nan)
	{
		infnan = ISUPPER(type) ? LITERAL("NAN") : LITERAL("nan");
		length += 3;
		format &= ~(FLG_ZERO_FILL|FLG_ALTERNATIVE_FORMAT);
	}
	else if (is_inf)
	{
		infnan = ISUPPER(type) ? LITERAL("INF") : LITERAL("inf");
		length += 3;
		format &= ~(FLG_ZERO_FILL|FLG_ALTERNATIVE_FORMAT);
	}
	else if ( IS_TYPE(type, 'a') )
	{
		if (!is_zero)
		{
			exponent -= BIAS;
		}

		/* The floating point number is emitted as:
		 *
		 *	[s] 0x H . hhh P S ddd
		 *
		 * where s	sign of the number
		 *	 H	either 0 (denomalized) or 1 (normalized)
		 *	 hhh	the fractional part
		 *	 S	sign of the exponent
		 *	 ddd	the exponent
		 */

		length += 3;			/* 0x H */
		if ( (precision != 0) || (format & FLG_ALTERNATIVE_FORMAT) )
		{
			length++;		/* . */
		}
		length += precision;		/* hhh */
		length++;			/* P */
		length++;			/* S-exponent */

#ifndef	__SINGLE_FP__
		if (exponent <= -1000 || exponent >= 1000)
		{
			explen = 4;
		}
		else
#endif
		if (exponent <= -100 || exponent >= 100)
		{
			explen = 3;
		}
		else if (exponent <= -10 || exponent >= 10)
		{
			explen = 2;
		}
		length += explen;
	}
	else
	{
		double fi, tmp_fi;		/* Integral part of 'arg' */

		/* After conversion 'decpt' is :
		 *	< 0	- Only fractional part present
		 *	== 0	- 0.0 or 1.0
		 *	> 0	- Integer part present
		 */

		p = buf;

		if (!is_zero)
		{
			/* Get integer and fractional part. In 'arg', the
			 * fractional part remains
			 */
			arg = modf( arg, &fi );

			/* Do integer part */
			if ( fi != 0.0 )
			{
				q = &buf[ NDIG ];
				tmp_fi = fi;
				while ( fi != 0.0 )
				{
					/*
					 * When buffer is full, shift the complete
					 * buffer one position to the right
					 */
					if ( q == buf )
					{
						p = &buf[ NDIG-1 ];
						q = &buf[ NDIG ];
						while ( p != buf )
						{
							*--q = *--p;
						}
						/*
						 * Now 'q' points to 'buf+1', the left
						 * most position of 'buf' can be used
						 * again.
						 */
					}
					/*
					 * 'fj' gets fractional value (divide by 10 ==
					 * multiply with 1/10
					 */
					fj = modf( fi * 0.1, &fi );
					if ( !first_nonzero )
					{
						multiplier *= 10.0;
						if ( (char)( (fj + 0.03) * 10.0 ) )
						{
							first_nonzero = 1;
							/* In this special case it is more precise
							 * to divide tmp_fi with a multiplier than
							 * to multiply tmp_fi with a divisor. So,
							 * this is less precise.
							 *
							 * divisor *= 0.1;
							 * ..
							 * fj = modf( tmp_fi * divisor, &fi );
							 * ..
							 */
							fj = modf( tmp_fi / multiplier, &fi );
						}
					}
					/* Convert fraction to a character */
					*--q = (char)( (fj + 0.03) * 10.0 ) + '0';
					decpt++;		/* Dot one position to the right */
				}
				while ( q < &buf[ NDIG ] )	/* Move all converted characters to */
				{				/* the left side of the buffer */
					*p++ = *q++;
				}
			}
			else if ( arg > 0.0 )
			{	/* make 'arg' a value between 1 and 10 */
				while ( (fj = arg*10.0) < 1.0 )
				{
					arg = fj;
					decpt--;		/* Dot one position to the left */
				}
			}
		}

		/* Rounding starts one position after the precision we can print */
		i = precision;
		if ( IS_TYPE(type, 'f') )
		{
			i += decpt;
		}

		/* Always one digit in front of the '.', which is not counted */
		if ( IS_TYPE(type, 'e') )
		{
			i += 1;
		}

		q = &buf[ (i < 0) ? 0 : (i >= NDIG-2) ? NDIG-2 : i ];

		/*
		 * Now convert fractional part. 'q' points to the last position
		 * in 'buf' where a character may be written.
		 */
		while ( p <= q && p < &buf[ NDIG ] )
		{
			arg *= 10.0;
			arg = modf( arg, &fj );
			*p++ = (char)fj + '0';
		}

		/* Rounding up at last position */
		p = q;

		/*
		 * Rounding up, real precision starts at position 15
		 * for double precision and at position 6 for single
		 * precision. Rounding up starts at position
		 * DBL_DIG+2, if number of digits > DBL_DIG+1.
		 */
		*q += 5;		/* Add 5 to last position */
		while ( *q > '9' )	/* Carry ? */
		{
			if ( q > buf )	/* Add carry to previous digit */
			{
				p = q;	/* round to '0', and remove 0 digits at the end of the string */
				*--q += 1;
			}
			else
			{
				/* When leftmost digit, it is rounded to '1' */
				*q = '1';
				/* Decimal point one position to the right */
				decpt++;
				/* String termination to the right of this digit */
				p = q + 1;
			}
		}
		*p-- = '\0';	/* Terminate string and point to last character in buffer */

		/* Remove '0' characters at the end of the string */
		for ( ; p >= buf && *p == '0'; p-- )
		{
			*p = 0;
		}

		/*
		 * To do the correct alignment, we need to detect the length of
		 * the string we are going to print
		 */
		if ( IS_TYPE(type, 'e')
		     ||
		     ( !IS_TYPE(type, 'f') && ((decpt > precision) || decpt <= -4)) )
		{
			/* E-style */
			q = buf;
			length += 1;		/* Digit before the '.' */

			if ( IS_TYPE(type, 'e') )
			{
				if ( *q )
				{
					q++;
				}

				if ( *q || (format & FLG_ALTERNATIVE_FORMAT) || precision )
				{
					length++;	/* '.' */
				}

				length += precision;
			}
			else
			{
				/* 'g' or 'G' */
				j = precision;
				if ( *q )
				{
					j--;
					q++;
				}
				if ( *q || (format & FLG_ALTERNATIVE_FORMAT) )
				{
					length++;	/* '.' */
				}
				for ( i=0; i<j; i++ )
				{
					if ( *q || (format & FLG_ALTERNATIVE_FORMAT) )
					{
						q++;
						length++;
					}
				}
			}
			length += 4;	/* two-digit exponent */
#ifndef	__SINGLE_FP__
			if ( decpt <= -99 || decpt >= 101 )
			{
				length++;
			}
#endif
		}
		else
		{
			if ( IS_TYPE(type, 'f') )
			{
				/* F-style */
				if ( decpt <= 0 )
				{
					length++;		/* '0' before '.' */
				}
				else
				{
					length += decpt;	/* nr of digits before '.' */
				}
				if ( precision || (format & FLG_ALTERNATIVE_FORMAT) )
				{
					length++;		/* '.' */
					length += precision;	/* nr of digits after '.' */
				}
			}
			else	/* 'g' or 'G' */
			{
				q = buf;
				i = precision;

				if ( decpt <= 0 )
				{
					length++;		/* '0' before '.' */
				}
				else
				{
					length+= decpt;		/* nr of digits before '.' */
					j = decpt;
					do
					{
						if ( *q )
						{
							q++;
						}
						if ( i )
						{
							i--;
						}
					} while ( --j );
				}
				if ( (*q && i) || (format & FLG_ALTERNATIVE_FORMAT) )
				{
					length++;		/* '.' */
					j = -decpt;
					/* no detect nr of digits after '.' */
					while ( i && j-- > 0 )
					{
						length++;
						i--;
					}
					while ( i && *q )
					{
						length++;
						q++;
						i--;
					}
					if ( format & FLG_ALTERNATIVE_FORMAT )
					{
						length += i;
					}
				}
			}
		}
	}

	if ( vsign && (format & (FLG_ZERO_FILL | FLG_LEFT_ALIGNMENT) ) )
	{
		(*characters)++;
		FPUTC( vsign, fout );
		vsign = 0;
	}

	if ( !(format & FLG_LEFT_ALIGNMENT) )
	{
		/* Right alignment
		 */
		for ( min-=length; min>0; min-- )
		{
			(*characters)++;
			FPUTC( (format & FLG_ZERO_FILL) ? '0' : ' ', fout );
		}
	}

	if ( vsign )
	{
		(*characters)++;
		FPUTC( vsign, fout );
	}

	if (infnan)
	{
		FPUTS(infnan, fout);
		*characters += 3;
	}
	else if ( IS_TYPE(type, 'a') )
	{
		int		shift, bits;
		uchar_t		lsb;
		unsigned long	mask;
		uchar_t		msdigit = '1' - is_zero;

#ifndef __SINGLE_FP__
		bits = DOUBLE_FRACTION_SIZE - precision * 4;
		if (bits > 0)	/* lost bits: 4, 8, 12, ... 52 */
		{
			/*
			 * Round to Nearest Even (RNE). Bits of interest are the (future) lsb,
			 * the msb of the bits to drop (a.k.a. round-bit) and the lesser bits.
			 * When the round-bit is '1' then RNE requires rounding up when the
			 * lsb or any of the lesser bits below the round bit is 1'. By ORing
			 * this "round-up" flag into all the lesser bits we can round by increment.
			 */
			lsb =	bits >= DOUBLE_FRACTION_SIZE
				|| (	bits >= 8 * sizeof (fraction)
					? hi_fract & (1 << (bits - 8 * sizeof (fraction)))
					: lo_fract & (1 << bits)
				   );
			--bits;		/* # bits below the round-bit */
			mask = ~(~0UL << (bits & (8 * sizeof (fraction) - 1)));
			if (bits < 8 * sizeof (fraction))
			{
				if (lsb || (lo_fract & mask))
					lo_fract |= mask;
			}
			else if (lsb || lo_fract || (hi_fract & mask))
			{
				lo_fract = ~0UL;
				hi_fract |= mask;
			}
			if (  ++lo_fract == 0
			   && (++hi_fract & (1UL << (DOUBLE_FRACTION_SIZE - 8 * sizeof (fraction))))
			   )
			{
				hi_fract = 0;
				msdigit = '2';	/* cannot increment exponent at this stage */
			}
		}
#else
		bits = FLOAT_FRACTION_SIZE - precision * 4;
		if (bits > 0)	/* lost bits: 3, 7, 11, ... 23 */
		{
			lsb = bits >= FLOAT_FRACTION_SIZE || (fraction & (1 << bits));
			mask = ~(~0UL << (bits - 1));
			if (lsb || (fraction & mask))
				fraction |= mask;
			if (++fraction & (1UL << (FLOAT_FRACTION_SIZE)))
			{
				fraction = 0;
				msdigit = '2';
			}
		}
#endif
		(*characters) += 3;
		FPUTC( '0', fout );
		FPUTC( type + 'X' - 'A', fout );
		FPUTC( msdigit, fout );

		if ( (precision != 0) || (format & FLG_ALTERNATIVE_FORMAT) )
		{
			(*characters)++;
			FPUTC( '.', fout );
		}

		/* The number of bits of the fractional part might not
		 * be a multiple of 4. Shift the fractional part such
		 * that it becomes a multiple of 4.
		 */
#ifndef	__SINGLE_FP__
		if (DOUBLE_FRACTION_SIZE % 4)
		{
			shift      = 4 - (DOUBLE_FRACTION_SIZE % 4);
			hi_fract <<= shift;
			hi_fract  |= (lo_fract >> ((sizeof(unsigned long) * 8) - shift));
			lo_fract <<= shift;
		}
		shift = ((DOUBLE_FRACTION_SIZE + 3) / 4) * 4 - sizeof(lo_fract) * 8;
		fraction = hi_fract;
#else
		if (FLOAT_FRACTION_SIZE % 4)
		{
			shift      = 4 - (FLOAT_FRACTION_SIZE % 4);
			fraction <<= shift;
		}
		shift = ((FLOAT_FRACTION_SIZE + 3) / 4) * 4;
#endif

		(*characters) += precision;
		while ( precision > 0 )
		{
			char_t val = '0';
#ifndef	__SINGLE_FP__
			if (shift == 0)
			{
				shift = sizeof(lo_fract) * 8;
				fraction = lo_fract;
				lo_fract = 0;
			}
#endif
			if (shift)
			{
				shift -= 4;
				val += (char_t)(fraction >> shift) & 0xF;
				if (val > '9')
				{
					val += type - ('0' + 10);
				}
			}
			FPUTC( (uchar_t) val, fout );
			precision--;
		}

		FPUTC( type + 'P' - 'A', fout );
		if ( exponent < 0 )
		{
			FPUTC( '-', fout );
			exponent = -exponent;
		}
		else
		{
			FPUTC( '+', fout );
		}
		*characters += 2 + explen;
		expbuf[explen] = 0;
		do
			expbuf[--explen] = '0' + exponent % 10;
		while (exponent /= 10);
		FPUTS( expbuf, fout );
	}
	else
	{
		q = buf;

		/* ANSI: Use 'e' notation when:
		 *
		 *	(exponent < -4) || (exponent >= precision)
		 */
		if ( IS_TYPE(type, 'e')
		     ||
		     ( !IS_TYPE(type, 'f') && ((decpt > precision) || decpt <= -4)) )
		{
			/* use E-style */
			if ( !*q )			/* Value 0.0, no digits present */
			{
				decpt++;		/* Add one to the exponent */
			}

			/*
			 * For 'g' or 'G', the digit before the '.' is also
			 * a significant digit.
			 */
			if ( *q && IS_TYPE(type, 'g') )
			{
				precision--;
			}

			(*characters)++;
			FPUTC( *q ? (uchar_t) *q++ : '0', fout );

			if ( *q						/* digits remain */
			     || (format & FLG_ALTERNATIVE_FORMAT)	/* or alternate printing */
			     || (precision && IS_TYPE(type, 'e'))	/* or not the complete precision is printed yet */
			     )
			{
				(*characters)++;
				FPUTC( '.', fout );
			}

			for ( i = 0; i < precision; i++ )
			{
				if ( *q
				     || (format & FLG_ALTERNATIVE_FORMAT)
				     || IS_TYPE(type, 'e')
				     )
				{
					(*characters)++;
					FPUTC( *q ? (uchar_t) *q++ : '0', fout );
				}
			}

			/* Watch this, in ASCII, the codes for
			 * 'e'	-> 0x65		'g'	-> 0x67
			 * 'E'	-> 0x45		'G'	-> 0x47
			 * To convert 'g' to 'e' and 'G' to 'E', we only have to AND the character
			 * with 0xFD (clear bit '1')
			 */
			(*characters)++;
			FPUTC( type & 0xFD, fout );	/* Position the 'exponent' */

			if ( --decpt >= 0 )		/* Now print exponent '+xxx' or '-xxx' */
			{
				(*characters)++;
				FPUTC( '+', fout );	/* First print sign ... */
			}
			else
			{
				decpt = -decpt;
				(*characters)++;
				FPUTC( '-', fout );
			}

			/*
			 * Value of decpt now always >= 0. Next casts are
			 * neccessary to get unsigned division
			 */
#ifndef	__SINGLE_FP__
			i = (unsigned int) decpt / 100U;
			if ( i )			/* Print a three digit exponent */
			{
				decpt = (unsigned int) decpt % 100U;
				(*characters)++;
				FPUTC( i + '0', fout );
			}
#endif
			/*
			 * Now decpt always < 100, we may do character division!
			 * We only print a two digit exponent
			 */
			(*characters)++;
			FPUTC( (unsigned char)decpt / 10U + '0', fout );
			(*characters)++;
			FPUTC( (unsigned char)decpt % 10U + '0', fout );
		}
		else
		{
			i = decpt;
			/*
			 * Make sure at least one digit is printed before the
			 * '.' is placed.
			 */
			if ( decpt <= 0 )
			{
				(*characters)++;
				FPUTC( '0', fout );
			}
			else
			{
				/* Print all digits before the '.' */
				do
				{
					(*characters)++;
					FPUTC( *q ? (uchar_t) *q++ : '0', fout );
					if ( !IS_TYPE(type, 'f') && precision )
					{	/* Count all significant digits */
						precision--;
					}
				} while ( --i > 0 );
			}

			/* Now we are at the '.' position */
			if (  ((IS_TYPE(type, 'f') || *q) && precision)
			   || (format & FLG_ALTERNATIVE_FORMAT)
			   )
			{	/* '*q' != 0 when some digit != '0' is not printed yet */
				(*characters)++;
				FPUTC( '.', fout );
				i = -decpt;
				while ( precision && (i-- > 0) )
				{
					(*characters)++;
					FPUTC( '0', fout );
					precision--;
				}
				while ( precision && *q )	/* Now copy the rest of the digits */
				{
					(*characters)++;
					FPUTC( (uchar_t) *q++, fout );
					precision--;
				}
				if ( IS_TYPE(type, 'f') || (format & FLG_ALTERNATIVE_FORMAT) )
				{
					while ( precision )	/* And trailing zeros */
					{
						(*characters)++;
						FPUTC( '0', fout );
						precision--;
					}
				}
			}
		}
	}

	/* For left alignment, we still have to print some spaces */
	for ( min-=length; min>0; min-- )
	{
		(*characters)++;
		FPUTC( ' ', fout );
	}
}
