/**************************************************************************
**                                                                        *
**  FILE        :  _atan.c                                                *
**                                                                        *
**  DESCRIPTION :  Source file for _atan() routine                        *
**                 Is used by 'atan()' and 'atan2()' routines             *
**                                                                        *
**      coefficients double are #5076 from Hart & Cheney. (17.55D)        *
**      coefficients single are #5071 from Hart & Cheney. ( 7.54D)        *
**                                                                        *
**  Copyright 1996-2010 Altium BV                                         *
**                                                                        *
**************************************************************************/

#include <math.h>

#ifdef __SINGLE_FP__
#pragma alias	_atanl	= __atan_spf
#pragma alias	_atan		= __atan_spf
#endif
#pragma alias	_atanf	= __atan_spf


static float xatan( float arg );

#define	SQ2P1	2.414213562373095048802e0f
#define	SQ2M1	0.414213562373095048802e0f
#define	PIO2	1.570796326794896619231e0f
#define	PIO4	0.785398163397448309615e0f
#define P0	0.1265998609915e2f
#define	P1	0.63691887127e1f
#define	Q0	0.1265998646243e2f
#define Q1	0.105891113168e2f
#define POLYNOMIAL(argsq,arg)	( ( ( P1 * (argsq) + P0 ) / ( ( (argsq) + Q1 ) * (argsq) + Q0 ) ) * (arg) )

/*
	_atan reduces its argument (known to be positive)
	to the range [0,0.414...] and calls xatan.
*/
static
float
__atan_spf( float arg )
{
	if( arg < SQ2M1 )
		return( xatan( arg ) );
	else if( arg > SQ2P1 )
		return( PIO2 - xatan( 1.0f / arg ) );
	else
		return( PIO4 + xatan( ( arg - 1.0f ) / ( arg + 1.0f ) ) );
}

/*
	xatan evaluates a series valid in the
	range [-0.414...,+0.414...].
*/
static float
xatan( float arg )
{
	float argsq;

	argsq = arg * arg;
	return POLYNOMIAL(argsq,arg);
}


