--------------------------------------------------------------
--
--  VERSION:            @(#)pcls_driver.vhdl	1.23 08/08/01
--
--  IN PACKAGE:         pcls-suite
--
--  AUTHOR:             Henk Huiteman           (henkh)
--                      Mark Jongerman          (markj)
--
--  COPYRIGHT:          Copyright 2005-2010 Altium BV
--
--  DESCRIPTION:        Driver for CHC tests
--
--------------------------------------------------------------

library ieee;
use std.textio.all;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity pcls_driver is
	generic(
		int_width : in integer := 32
	);
	port(
		clock     : out std_logic;
		reset     : out std_logic;
		reset_done: in std_logic;
		exit_code : in std_logic_vector( int_width-1 downto 0 );
		start     : out std_logic;
		done      : in std_logic;
		test_init : in std_logic;
		quiet     : out std_logic
		);
end pcls_driver;

architecture mydriver of pcls_driver is

	type STATE is ( STATE_0, STATE_1, STATE_2, STATE_3, STATE_4, STATE_5, STATE_6, STATE_7, STATE_8 );

	signal current_state : STATE := STATE_0;
	signal next_state    : STATE := STATE_0;

	signal clk           : std_logic := '0';
	signal finished      : std_logic := '0';
	signal retval        : std_logic_vector( int_width-1 downto 0 );

begin  -- mydriver

	GEN_CLK : process
		variable nr_clocks : integer := 0;
		variable i         : signed( int_width-1 downto 0 );
		variable l         : line;
	begin
		if finished = '1' then
			i := signed( retval );
			write( l, string'("exit_code = ") );
			write( l, to_integer(i) );
			writeline( output, l );
			write( l, string'("# cycles=") );
			write( l, nr_clocks );
			writeline( output, l );
			wait for 5 ns;
			clk <= '0';
			wait for 5 ns;
			clk <= '1';
			wait for 5 ns;
			clk <= '0';
			wait for 5 ns;
			clk <= '1';
			wait;
		end if;

		clk <= '0';
		wait for 5 ns;
		clk <= '1';
		nr_clocks := nr_clocks + 1;

		wait for 5 ns;

	end process; -- GEN_CLK

	clock <= clk;

	SYNC : process
	begin
		wait until rising_edge( clk );
		current_state <= next_state;
	end process; -- SYNC

	FSM : process( current_state, done, reset_done )
	begin
		next_state <= current_state;
		case current_state is
		when STATE_0 =>
			if ( test_init = '1' ) then
				next_state <= STATE_2;
			else
				next_state <= STATE_1;
			end if;
		when STATE_1 =>
			if ( reset_done = '1' ) then
				next_state <= STATE_2;
			else
				next_state <= STATE_1;
			end if;
		when STATE_2 =>
			next_state <= STATE_3;
		when STATE_3 =>
			if ( done = '1' ) then
				--
				-- The test_init flag is set by the c_root wrapper architecture.
				--
				if ( test_init = '1' ) then
					next_state <= STATE_4;
				else
					next_state <= STATE_8;
				end if;
			else
				next_state <= STATE_3;
			end if;
		when STATE_4 =>
			if ( test_init = '1' ) then
				next_state <= STATE_6;
			else
				next_state <= STATE_5;
			end if;
		when STATE_5 =>
			if ( reset_done = '1' ) then
				next_state <= STATE_6;
			else
				next_state <= STATE_5;
			end if;
		when STATE_6 =>
			next_state <= STATE_7;
		when STATE_7 =>
			if ( done = '1' ) then
				next_state <= STATE_8;
			else
				next_state <= STATE_7;
			end if;
		when STATE_8 =>
			next_state <= STATE_0;
		when others =>
			null;
		end case; -- current_state
	end process; -- FSM

	DP : process
	begin
		wait until rising_edge( clk );
		if current_state = STATE_0 then
			quiet <= test_init;
			reset <= '1';			-- First reset
			start <= '0';
		elsif current_state = STATE_1 then
			reset <= '0';
		elsif current_state = STATE_2 then
			reset <= '0';
			start <= '1';			-- Start first testrun
		elsif current_state = STATE_3 then
			start <= '0';
			retval <= exit_code;
		elsif current_state = STATE_4 then
			quiet <= not test_init;
			reset <= '1';			-- Second reset (if test_init is set)
		elsif current_state = STATE_5 then
			reset <= '0';
		elsif current_state = STATE_6 then
			reset <= '0';
			start <= '1';			-- Start second testrun (if test_init is set)
		elsif current_state = STATE_7 then
			start <= '0';
			retval <= exit_code;
		elsif current_state = STATE_8 then
			finished <= '1';		-- Done
		end if;
	end process; -- DP

end mydriver;

-- __extern void __putchar( _fd_t, char c );
-- used instead of fss

library ieee;
use std.textio.all;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity	putchar is
	port(
		fd       : in std_logic_vector( 7 downto 0 );
		ch       : in std_logic_vector( 7 downto 0 );
		start    : in std_logic;
		done     : out std_logic;
		clock    : in std_logic;
		reset    : in std_logic;
		quiet    : in std_logic
		);
end putchar;

architecture simio of putchar is
begin
	proc_write: process -- ( start, ch )
		variable char : character;
		variable l         : line;
	begin
		wait until rising_edge(clock);
		if start = '1' and quiet = '0' then
			char := character'val(to_integer(unsigned(ch)));
			if to_integer(unsigned(ch)) = 10 then
				writeline( output, l );
			elsif char /= nul then
				write( l, char );
			end if;
			done <= '1';
		else
			done <= '1';
		end if;
	end process; -- proc_write

end simio;

library ieee;
use ieee.std_logic_1164.all;

entity c_root_wrapper is
generic(
	int_width : in integer := 32
);
port(
	clock         : in std_logic;
	reset         : in std_logic;
	reset_done    : out std_logic;
	c_root_retval : out std_logic_vector( int_width-1 downto 0 );
	c_root_start  : in std_logic;
	c_root_done   : out std_logic;
	putchar_fd    : out std_logic_vector( 7 downto 0 );
	putchar_c     : out std_logic_vector( 7 downto 0 );
	putchar_start : out std_logic;
	putchar_done  : in std_logic;
	test_init     : out std_logic
    );
end c_root_wrapper;

--
-- The top level entity of the test environment.
-- It just has two components. One is the test driver
-- and the other is the test (probably compiled from C).
--
-- This entity just connects these components together.
--
library ieee;
use ieee.std_logic_1164.all;

entity test_bench is
generic(
	int_width : in integer := 32
);
end test_bench;

architecture mybench of test_bench is

	component pcls_driver is
	generic(
		int_width : in integer := 32
	);
	port (
		clock      : out std_logic;
		reset      : out std_logic;
		reset_done : in std_logic;
		start      : out std_logic;
		done       : in std_logic;
		exit_code  : in std_logic_vector( int_width-1 downto 0 );
		test_init  : in std_logic;
		quiet      : out std_logic
	);
	end component;

	component c_root_wrapper
	generic(
		int_width : in integer := 32
	);
	port (
		clock         : in std_logic;
		reset         : in std_logic;
		reset_done    : out std_logic;
		c_root_retval : out std_logic_vector( int_width-1 downto 0 );
		c_root_start  : in std_logic;
		c_root_done   : out std_logic;
		putchar_fd    : out std_logic_vector( 7 downto 0 );
		putchar_c     : out std_logic_vector( 7 downto 0 );
		putchar_start : out std_logic;
		putchar_done  : in std_logic;
		test_init     : out std_logic
	);
	end component;

	component putchar
	port(
		fd    : in std_logic_vector( 7 downto 0 );
		ch    : in std_logic_vector( 7 downto 0 );
		start : in std_logic;
		done  : out std_logic;
		clock : in std_logic;
		reset : in std_logic;
		quiet : in std_logic
		);
	end component;

	signal clock       : std_logic;
	signal reset       : std_logic;
	signal reset_done  : std_logic;
	signal exit_code   : std_logic_vector( int_width-1 downto 0 );
	signal start       : std_logic;
	signal done        : std_logic;
	signal test_init   : std_logic;
	signal quiet       : std_logic;

	signal fd          : std_logic_vector( 7 downto 0 );
	signal char        : std_logic_vector( 7 downto 0 );
	signal write_start : std_logic;
	signal write_done  : std_logic;

begin  -- mybench

	test : c_root_wrapper
		generic map (
			int_width  => int_width
		)
		port map (
			clock         => clock,
			reset         => reset,
			reset_done    => reset_done,
			c_root_retval => exit_code,
			c_root_start  => start,
			c_root_done   => done,
			putchar_fd    => fd,
			putchar_c     => char,
			putchar_start => write_start,
			putchar_done  => write_done,
			test_init     => test_init
		);

	test_driver : pcls_driver
		generic map (
			int_width  => int_width
		)
		port map (
			clock      => clock,
			reset      => reset,
			reset_done => reset_done,
			exit_code  => exit_code,
			start      => start,
			done       => done,
			test_init  => test_init,
			quiet      => quiet
		);

	test_write : putchar
		port map (
			clock => clock,
			reset => reset,
			fd    => fd,
			ch    => char,
			start => write_start,
			done  => write_done,
			quiet => quiet
		);

end mybench;


-- 16 bit variant

entity test_bench_16 is
end test_bench_16;

architecture mybench of test_bench_16 is
	component test_bench is
	generic(
		int_width : in integer := 32
	);
	end component;

begin
	test16: component test_bench
	generic map (
		int_width => 16
	);
end mybench;
