//------------------------------------------------------------
//
//  VERSION:		@(#)pcls_driver.v	1.13	08/08/15
//
//  IN PACKAGE:		pcls-suite
//
//  AUTHORS:		Mark Jongerman		(markj)
//
//  COPYRIGHT:		Copyright 2006-2010 Altium BV
//
//  DESCRIPTION:        Driver for CHC tests
//------------------------------------------------------------


`ifdef 	INT_WIDTH
`else
`define	INT_WIDTH	32
`endif

module chc_driver( clock, reset, reset_done, exit_code, start, done, test_init, quiet );

	parameter		int_width = 32;
	parameter		STATE_0 = 4'd0;
	parameter		STATE_1 = 4'd1;
	parameter		STATE_2 = 4'd2;
	parameter		STATE_3 = 4'd3;
	parameter		STATE_4 = 4'd4;
	parameter		STATE_5 = 4'd5;
	parameter		STATE_6 = 4'd6;
	parameter		STATE_7 = 4'd7;
	parameter		STATE_8 = 4'd8;

	output			clock;
	output			reset;
	input			reset_done;
	input	[int_width-1:0]	exit_code	/*verilator public*/;
	output			start;
	input			done		/*verilator public*/;
	input			test_init;
	output			quiet;

	wire			clock;
	wire			reset;
	wire	[int_width-1:0]	exit_code;
	wire			start;
	wire			done;
	reg			quiet;

	reg	[3:0]		current_state = STATE_0;
	reg	[3:0]		next_state = STATE_0;
	reg			finished = 0;

	reg			reg_clock	/*verilator public*/;
	reg			reg_reset	/*verilator public*/;
	reg			reg_start		/*verilator public*/;
	reg			reg_update	/*verilator public*/;

	integer			nr_clocks = 0;
	integer			retval;

	always @( posedge clock )
	begin
		if ( finished )
		begin
			$display( "exit_code = %d", retval );
			$display( "# cycles=%d", nr_clocks );
			$finish;
		end
		nr_clocks = nr_clocks + 1;
	end

	always @( posedge clock )
	begin: SYNC
		current_state <= next_state;
	end // SYNC


	always @( current_state, done, reset_done )
	begin: FSM
		next_state = current_state;
		case ( current_state )
		STATE_0:
			if ( test_init )
			begin
				next_state = STATE_2;
			end else begin
				next_state = STATE_1;
			end

		STATE_1:
			if ( reset_done )
			begin
				next_state = STATE_2;
			end else begin
				next_state = STATE_1;
			end

		STATE_2:
			next_state = STATE_3;

		STATE_3:
			if ( done )
			begin
				//
				// The test_init flag is set by the c_root wrapper architecture.
				//
				if ( test_init )
				begin
					next_state = STATE_4;
				end else begin
					next_state = STATE_8;
				end
			end else begin
				next_state = STATE_3;
			end

		STATE_4:
			if ( test_init )
			begin
				next_state = STATE_6;
			end else begin
				next_state = STATE_5;
			end

		STATE_5:
			if ( reset_done ) 
			begin
				next_state = STATE_6;
			end else begin
				next_state = STATE_5;
			end

		STATE_6:
			next_state = STATE_7;

		STATE_7:
			if ( done )
			begin
				next_state = STATE_8;
			end else begin
				next_state = STATE_7;
			end

		STATE_8:
			next_state = STATE_0;

		default: ;
		endcase // current_state
	end // FSM

	always @( posedge clock )
	begin: DP
		if ( current_state == STATE_0 )
		begin
			quiet = test_init;
			reg_reset = 1;			// First reset
			reg_start = 0;
		end else if ( current_state == STATE_1 )
		begin
			reg_reset = 0;
		end else if ( current_state == STATE_2 )
		begin
			reg_reset = 0;
			reg_start = 1;			// Start first testrun
		end else if ( current_state == STATE_3 )
		begin
			reg_start = 0;
			retval <= exit_code;
		end else if ( current_state == STATE_4 )
		begin
			quiet = !test_init;
			reg_reset = 1;			// Second reset (if test_init is set)
		end else if ( current_state == STATE_5 )
		begin
			reg_reset = 0;
		end else if ( current_state == STATE_6 )
		begin
			reg_reset = 0;
			reg_start = 1;			// Start second testrun (if test_init is set)
		end else if ( current_state == STATE_7 )
		begin
			reg_start = 0;
			retval <= exit_code;
		end else if ( current_state == STATE_8 )
		begin
			finished <= 1;			// Done
		end
	end // DP

`ifdef verilator
	// In Verilator, signals are driven by a .cpp program, but
	// for some reason signal changes are not always propagated
	// correctly. The following construction fixes this.
	// (Don't really know why but this works).
	always @( posedge reg_update )
	begin
		clock	= reg_clock;
		reset	= reg_reset;
		start	= reg_start;
	end
`else
	assign	clock	= reg_clock;
	assign	reset	= reg_reset;
	assign	start	= reg_start;

	initial
	begin
		reg_clock = 0;
		forever #5 reg_clock <= ~reg_clock;
	end
`endif

endmodule

module put_char( fd, ch, start, done, clock, reset, quiet );

	input		[7:0]	fd;
	input		[7:0]	ch;
	input			start;
	output			done;
	input			clock;
	input			reset;
	input			quiet;
	
	reg			done;

	always @( posedge clock )
	begin
		if ( start == 1 && quiet == 0 )
		begin
			if ( ch == 10 ) begin
				$write( "\n" );
			end else if ( ch != 0 )
			begin
				$write( "%0c", ch );
			end

			done = 1;
		end
	end

endmodule

//
// The top level entity of the test environment.
// It just has two components. One is the test driver
// and the other is the test (probably compiled from C).
//
// This entity just connects these components together.
// 
module test_bench;

	parameter		int_width = `INT_WIDTH;

	wire			clock;
	wire			reset;
	wire	[int_width-1:0]	exit_code;
	wire			start;
	wire			done;

	wire		[7:0]	fd;
	wire		[7:0]	char;
	wire			write_start;
	wire			write_done;
	wire			test_init;

	c_root_wrapper #(int_width) test(
		.clock( clock ),
		.reset( reset ),
		.reset_done( reset_done ),
		.c_root_retval( exit_code ),
		.c_root_start( start ),
		.c_root_done( done ),
		.putchar_fd ( fd ),
		.putchar_c ( char ),
		.putchar_start ( write_start ),
		.putchar_done ( write_done ),
		.test_init( test_init )
	);

	chc_driver #(int_width) t_driver(
		.clock( clock ),
		.reset( reset ),
		.reset_done( reset_done ),
		.exit_code( exit_code ),
		.start( start ),
		.done( done ),
		.test_init( test_init ),
		.quiet( quiet )
	);

	put_char test_write(
		.clock( clock ),
		.reset( reset ),
		.fd( fd ),
		.ch( char ),
		.start( write_start ),
		.done( write_done ),
		.quiet( quiet )
	);

endmodule
