/* @(#)uio.c    1.6 */
/*
 * Unified I/O.
 */

#include <osek/osek.h>
#include <osek/uart.h>
#include <osek/host.h>
#include <osek/irq.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>

#include "uio.h"

typedef struct
{
        const char *    prefix;
        driver_t        driver;
} namespace_t;

typedef struct
{
        driver_t        driver;
        device_t        device;
} file_t;

#define NFILE                   32
static  file_t                  files[NFILE];
static  _err_t                  get_fd(void);
static  void                    put_fd(_fd_t fd);
#define bad_fd(fd)              ((fd) >= NFILE || files[fd].driver >= DRIVER_MAX)

/*
 * Events and resources. Currently we use only one event. The resources
 * are all driver specific.
 */
DeclareEvent(io_completed);

/*
 * Enumerate the drivers
 */
enum
{
#define DEV(prefix,major,o,r,w,s,c)     major,
#include "dev.h"
                                        DRIVER_MAX
};

/*
 * Driver operations.
 */
static const uio_t uio[DRIVER_MAX] =
{
#define DEV(prefix,major,o,r,w,s,c)     { o,r,w,s,c },
#include "dev.h"
};

/*
 * Populate the name-space. Whenever a pathname prefix matches, the
 * remaining part is passed to the open method of the corresponding
 * driver. For example, a prefix "/dev/uart" will match pathnames
 * "/dev/uart", /dev/uart0, "/dev/uart1" and will call the drivers
 * open method specifying "", "0" and "1" respectively.
 */
static const namespace_t namespace[DRIVER_MAX + 1] =
{
#define DEV(prefix,major,o,r,w,s,c)     { prefix, major },
#include "dev.h"
                                        { NULL,   0     }
};

/*
 * This should be called before doing any I/O, e.g. from StartupHook()
 */
extern void uio_config(void)
{
        _fd_t   fd;

        memset(files, -1, sizeof (files));
        /*
         * Hack to redirect fd 0, 1 and 2 to the first serial port in order
         * to support the standard I/O streams. We should do an _open() and
         * check various error conditions but this way the linker may drop
         * all open code, get_fd() and even the namespace.
         */
        for (fd = 0; fd < (NFILE < 3 ? NFILE : 3); ++fd)
        {
#if CONFIG_UART
                files[fd].driver = UART_DRIVER;
                files[fd].device = 0;
#elif CONFIG_HOST
                files[fd].driver = HOST_DRIVER;
                files[fd].device = (device_t)fd;
#endif
        }
}

/*
 * Associate a file descriptor with the pathname and call the corresponding driver.
 * Return the file descriptor or a negative errno.
 */
extern _err_t uio_open(const char *pathname, _open_flags_t flags)
{
        int                     len;
        const namespace_t       *ns;
        _err_t                  ret;
        _fd_t                   fd;
        irqflags_t              irqflags;

        ret = get_fd();
        if (ret >= 0)
        {
                fd = ret;
                ret = -ENOENT;
                for (ns = namespace; ns->prefix; ++ns)
                {
                        len = strlen(ns->prefix);
                        if (strncmp(ns->prefix, pathname, len) == 0)
                        {
                                ret = 0;
                                if (uio[ns->driver].open)
                                {
                                        ret = uio[ns->driver].open(pathname + len, flags);
                                }
                                if (ret >= 0)
                                {
                                        irqsuspend(irqflags);
                                        files[fd].driver = ns->driver;
                                        files[fd].device = ret;
                                        irqrestore(irqflags);
                                        ret = fd;
                                }
                                break;
                        }
                }
                if (ret < 0)
                {
                        put_fd(fd);
                }
        }
        return ret;
}

/*
 * Return the number of bytes read.
 */
extern int uio_read(_fd_t fd, void *buffer, size_t nbytes)
{
        ioreq_t request;
        int     ret;

        if (bad_fd(fd))
        {
                ret = -EBADF;
        }
        else
        {
                request.buf = buffer;
                request.len = nbytes;
                GetTaskID(&request.task);
                request.mask = io_completed;
                ClearEvent(io_completed);
                uio[files[fd].driver].read(files[fd].device, &request);
                WaitEvent(io_completed);
                ret = request.err ? request.err : request.pos;
        }
        return ret;
}

/*
 * Return the number of bytes written.
 */
extern int uio_write(_fd_t fd, const void *buffer, size_t nbytes)
{
        ioreq_t request;
        int     ret;

        if (bad_fd(fd))
        {
                ret = -EBADF;
        }
        else
        {
                request.buf = (void *)buffer;           /* discard const, we'll issue a write request */
                request.len = nbytes;
                GetTaskID(&request.task);
                request.mask = io_completed;
                ClearEvent(io_completed);
                uio[files[fd].driver].write(files[fd].device, &request);
                WaitEvent(io_completed);
                ret = request.err ? request.err : request.pos;
        }
        return ret;
}

/*
 * Return the new file position.
 */
extern long uio_lseek(_fd_t fd, long offset, _whence_t whence)
{
        _err_t  status;

        if (bad_fd(fd))
        {
                status = -EBADF;
        }
        else
        {
                status = -ESPIPE;
                if (uio[files[fd].driver].lseek)
                {
                        offset = uio[files[fd].driver].lseek(files[fd].device, offset, whence);
                        status = 0;
                }
        }
        if (status < 0)
        {
                return status;
        }
        return offset;
}

/*
 * Finish I/O on this file descriptor. Notice that the driver
 * may need to do refcounting...
 */
extern _err_t uio_close(_fd_t fd)
{
        _err_t  ret;

        if (bad_fd(fd))
        {
                ret = -EBADF;
        }
        else
        {
                ret = 0;
                if (uio[files[fd].driver].close)
                {
                        ret = uio[files[fd].driver].close(files[fd].device);
                }
                put_fd(fd);
        }
        return ret;
}

/*
 * Get a free file descriptor. I/O on any invalid file descriptor
 * should be handled gracefully in order to be not too far off POSIX
 * so the initial driver for the file descriptor will be set such that
 * all I/O operations yield EBADF.
 */
static _err_t get_fd(void)
{
        _fd_t           fd;
        irqflags_t      irqflags;

        irqsuspend(irqflags);
        for (fd = 0; fd < NFILE; ++fd)
        {
                if (files[fd].driver == (driver_t)-1)
                {
                        files[fd].driver = DRIVER_MAX;
                        break;
                }
        }
        irqrestore(irqflags);
        if (fd < NFILE)
        {
                return fd;
        }
        return -ENFILE;
}

/*
 * Release file descriptor.
 */
static void put_fd(_fd_t fd)
{
        irqflags_t      irqflags;

        irqsuspend(irqflags);
        files[fd].driver = (driver_t)-1;
        irqrestore(irqflags);
}

/*
 * I/O functions used by the C library.
 * These _may_ set a shared errno in which case they are not suited for use
 * from within more than one task at a time. See io.h in the C library for
 * the official prototypes.
 */

#define seterrno(x)             ((x) < 0 ? errno = -(int)(x), -1 : (x))

extern _err_t _open(const char *pathname, _open_flags_t flags)
{
        _err_t  ret;

        ret = uio_open(pathname, flags);
        return seterrno(ret);
}

extern int _read(_fd_t fd, void *buffer, size_t nbytes)
{
        int     ret;

        ret = uio_read(fd, buffer, nbytes);
        return seterrno(ret);
}

extern int _write(_fd_t fd, const void *buffer, size_t nbytes)
{
        int     ret;

        ret = uio_write(fd, buffer, nbytes);
        return seterrno(ret);
}

extern long _lseek(_fd_t fd, long offset, _whence_t whence)
{
        offset = uio_lseek(fd, offset, whence);
        return seterrno(offset);
}

extern _err_t _close(_fd_t fd)
{
        _err_t  ret;

        ret = uio_close(fd);
        return seterrno(ret);
}
