/* @(#)uart.c   1.4 */

/*
 * NOTES:
 * -    Except when noted otherwise, all static functions must be called with
 *      the uart lock held and all extern functions must be called without.
 *
 * RESTRICTIONS:
 * -    receiver/transmitter errors are ignored.
 * -    XON/XOFF characters cannot be defined as '\0'
 *
 * DESIGN NOTES:
 * -    implement cooked/raw/slip as a queued ioctl (tx) and remove transmitter state reset code?
 * -    cooked/raw/slip dependent r/w [abort] post code function pointer for resetting state?
 *
 */

#include <stddef.h>
#include <errno.h>
#include <osek/uio.h>

#include <osek/uart.h>

/*
 * The driver sends an XOFF once when the free buffer space reaches
 * SKIDLEN_LO. An XON will be sent when SKIDLEN_HI is exceeded.
 * Initially, the driver will send an XON after doing an appropriate read
 * request but the driver will start as if an XON has been received.
 * Since all buffers are supplied by the caller, handshaking will break
 * when the pending read request(s) stay <= SKIDLEN_HI.
 */
#define SKIDLEN_LO      10
#define SKIDLEN_HI      (2 * SKIDLEN_LO)
#define CONTROL(c)      ((c) & 31)
#define XON             CONTROL('q')
#define XOFF            CONTROL('s')

/*
 * SLIP constants. These are not really configurable. The receiver logic is
 * optimized to take advantage of this, knowing that ESC_* and ESC,END are
 * all different and nonzero.
 */
#define END             '\300'          /* SLIP frame end */
#define ESC             '\333'          /* escape for END/ESC as payload */
#define ESC_END         '\334'
#define ESC_ESC         '\335'

static  void            uart_update_rxroom(uart_t *uart, int delta);
static  void            uart_rx_putchar_xonxoff(uart_t *uart, char c);
static  bool_t          uart_rx_cooked(uart_t *uart, ioreq_t *request, char c);
static  bool_t          uart_rx_raw(uart_t *uart, ioreq_t *request, char c);
static  bool_t          uart_rx_slip(uart_t *uart, ioreq_t *request, char c);
static  int             uart_tx_getchar_xonxoff(uart_t *uart);
static  int             uart_tx_cooked(uart_t *uart);
static  int             uart_tx_raw(uart_t *uart);
static  int             uart_tx_slip(uart_t *uart);
static  void            uart_start_transmit(uart_t *uart);
static  void            uart_do_echo(uart_t *uart, const char *buf, int len);
static  void            uart_tx_enqueue(uart_t *uart, char c);
static  void            _uart_disable_xonxoff(uart_t *uart);

struct xonxoff
{
        void            (*update_rxroom)(uart_t *, int);
        void            (*rx_putchar)(uart_t *, char);
        int             (*tx_getchar)(uart_t *);
};

/*
 * This has been factored out to help the linker omitting unused code.
 */
static  xonxoff_t       xonxoff_operations =
{
        uart_update_rxroom,
        uart_rx_putchar_xonxoff,
        uart_tx_getchar_xonxoff
};

extern void uart_read(device_t dev, ioreq_t *request)
{
        uart_t  *uart = uart_dev + dev;
        int     len;

        uart_lock(dev);
        ioreq_put(&uart->rx, request);
        len = request->len;
        if (len > 0 && uart->xonxoff)
        {
                uart->xonxoff->update_rxroom(uart, len);
        }
        uart_unlock(dev);
}

extern ioreq_t *uart_read_abort(device_t dev)
{
        uart_t  *uart = uart_dev + dev;
        ioreq_t *request;

        uart_lock(dev);
        if ((request = ioreq_abort(&uart->rx)))         /* then correct uart state */
        {
                if (uart->xonxoff && request->len > 0)  /* tell about lost bufferspace */
                {
                        uart->xonxoff->update_rxroom(uart, request->pos - request->len);
                }
                uart->rx_slipesc = false;
        }
        uart_unlock(dev);
        return request;
}

static void uart_update_rxroom(uart_t *uart, int delta)
{
        char    x;
        int     old, new;

        x = 0;
        old = uart->xonxoff_rxroom;
        new = old + delta;
        uart->xonxoff_rxroom = new;
        if (  new <= SKIDLEN_LO
           && old > SKIDLEN_LO
           )
        {
                x = XOFF;
        }
        else if (  old <= SKIDLEN_HI
                && new > SKIDLEN_HI
                )
        {
                x = XON;
        }
        if (x)
        {
                uart->xonxoff_urgent = x;
                uart_start_transmit(uart);
        }
}

static void uart_rx_putchar_xonxoff(uart_t *uart, char c)
{
        int     n;
        ioreq_t *request;

        if (c == XOFF)
        {
                uart->xoff_received = true;
        }
        else if (c == XON)
        {
                uart->xoff_received = false;
                uart_start_transmit(uart);
        }
        else if ((request = ioreq_get(&uart->rx)))
        {
                n = request->pos;
                if (uart->rx_data(uart, request, c))
                {
                        ioreq_end(request);
                        n -= request->len - request->pos;       /* unused bufferspace is lost */
                }
                uart_update_rxroom(uart, n - request->pos);
        }
}

/*
 * Must be called with uart lock held.
 */
extern void uart_rx_putchar(device_t dev, char c)
{
        uart_t  *uart = uart_dev + dev;
        ioreq_t *request;

        if (uart->xonxoff)
        {
                uart->xonxoff->rx_putchar(uart, c);
        }
        else if (  (request = ioreq_get(&uart->rx))
                && uart->rx_data(uart, request, c)
                )
        {
                ioreq_end(request);
        }
}

static bool_t uart_rx_cooked(uart_t *uart, ioreq_t *request, char c)
{
        bool_t  done;

        done = false;
        switch (c)
        {
        case CONTROL('c'):
                request->err = -EINTR;
                done = true;
                break;
        case '\177':
        case '\b':
                if (request->pos > 0)
                {
                        --request->pos;
                        uart_do_echo(uart, "\b \b", 3);
                }
                break;
        case '\r':
                c = '\n';
                /* fall through */
        case '\n':
                done = true;
                /* fall through */
        default:
                request->buf[request->pos++] = c;
                if (!done)
                {
                        if (request->pos == request->len)
                        {
                                --request->pos;
                                break;
                        }
                        if ((unsigned char)c < ' ')     /* We're assuming ASCII or a printable 8 bit superset here */
                        {
                                c = '?';
                        }
                        if ((unsigned char)uart->tx_echo > ECHO_ON)
                        {
                                c = (char)uart->tx_echo;
                        }
                }
                uart_do_echo(uart, &c, 1);
                break;
        }
        return done;
}

static bool_t uart_rx_raw(uart_t *uart, ioreq_t *request, char c)
{
        request->buf[request->pos++] = c;
        return request->pos == request->len;
}

static bool_t uart_rx_slip(uart_t *uart, ioreq_t *request, char c)
{
        bool_t  done;

        done = false;
        if (c == ESC)
        {
                uart->rx_slipesc = true;
        }
        else
        {
                if (c == END)
                {
                        if (request->pos)               /* skip empty frames */
                        {
                                done = true;
                        }
                        uart->rx_slipesc = false;       /* reset state, just in case */
                }
                else
                {
                        if (uart->rx_slipesc)
                        {
                                c = c == ESC_ESC ? ESC : END;
                                uart->rx_slipesc = false;
                        }
                        if (request->pos < request->len)
                        {
                                request->buf[request->pos++] = c;
                        }
                }
        }
        return done;
}

extern void uart_write(device_t dev, ioreq_t *request)
{
        uart_t  *uart = uart_dev + dev;

        uart_lock(dev);
        if (ioreq_put(&uart->tx, request))
        {
                uart_start_transmit(uart);
        }
        uart_unlock(dev);
}

extern ioreq_t *uart_write_abort(device_t dev)
{
        uart_t  *uart = uart_dev + dev;
        ioreq_t *request;

        uart_lock(dev);
        if ((request = ioreq_abort(&uart->tx)))
        {
                uart->tx_slipstate = 0;
        }
        uart_unlock(dev);
        return request;
}

static int uart_tx_getchar_xonxoff(uart_t *uart)
{
        int     ic;

        if (uart->xonxoff_urgent)
        {
                ic = (unsigned char)uart->xonxoff_urgent;
                uart->xonxoff_urgent = 0;
        }
        else if (uart->xoff_received)
        {
                ic = -1;
        }
        else
        {
                ic = uart->tx_data(uart);
        }
        return ic;
}

/*
 * Must be called with uart lock held.
 */
extern int uart_tx_getchar(device_t dev)
{
        uart_t  *uart = uart_dev + dev;
        int     ic;

        ic = uart->xonxoff ? uart->xonxoff->tx_getchar(uart) : uart->tx_data(uart);
        if (ic < 0)
        {
                uart->tx_running = false;
        }
        return ic;
}

static int uart_tx_cooked(uart_t *uart)
{
        int     ic;
        ioreq_t *request;

        ic = -1;
        if (uart->tx_bufin != uart->tx_bufout)
        {
                ic = (unsigned char)uart->tx_buf[uart->tx_bufout];
                uart->tx_bufout = TXBUF_NORM(uart->tx_bufout + 1);
        }
        else if ((request = ioreq_get(&uart->tx)))
        {
                ic = (unsigned char)request->buf[request->pos++];
                if (request->pos == request->len)
                {
                        ioreq_end(request);
                }
        }
        if (ic == '\n')
        {
                uart_tx_enqueue(uart, '\r');
        }
        return ic;
}

static int uart_tx_raw(uart_t *uart)
{
        char    c;
        ioreq_t *request;

        if ((request = ioreq_get(&uart->tx)))
        {
                c = request->buf[request->pos++];
                if (request->pos == request->len)
                {
                        ioreq_end(request);
                }
        }
        else
        {
                return -1;
        }
        return (unsigned char)c;
}

static int uart_tx_slip(uart_t *uart)
{
        char    c;
        ioreq_t *request;
        int     state;

        if ((request = ioreq_get(&uart->tx)))
        {
                state = uart->tx_slipstate;
                switch (state)
                {
                case 0:                         /* frame start: send END to terminate bogus frame */
                        c = END;
                        state = 2;
                        break;
                case 2:                         /* frame data */
                        c = request->buf[request->pos++];
                        state += request->pos == request->len;
                        if (c == ESC)
                        {
                                state += 2;
                        }
                        else if (c == END)
                        {
                                c = ESC;
                                state += 4;
                        }
                        break;
                case 3:                         /* frame end */
                        c = END;
                        ioreq_end(request);
                        state = 0;
                        break;
                case 4:                         /* translated ESC octet */
                case 5:                         /* translated ESC octet, no more data */
                        c = ESC_ESC;
                        state -= 2;
                        break;
                case 6:                         /* translated END octet */
                case 7:                         /* translated END octet, no more data */
                default:
                        c = ESC_END;
                        state -= 4;
                        break;
                }
                uart->tx_slipstate = state;
        }
        else
        {
                return -1;
        }
        return (unsigned char)c;
}

static void uart_start_transmit(uart_t *uart)
{
        if (!uart->tx_running)
        {
                uart->tx_running = true;
                uart_transmit((device_t)(uart - uart_dev));
        }
}

static void uart_do_echo(uart_t *uart, const char *buf, int len)
{
        if (uart->tx_echo)
        {
                while (len--)
                {
                        uart_tx_enqueue(uart, *buf++);
                }
        }
}

static void uart_tx_enqueue(uart_t *uart, char c)
{
        if (TXBUF_NORM(uart->tx_bufin + 1) != uart->tx_bufout)
        {
                uart->tx_buf[uart->tx_bufin] = c;
                uart->tx_bufin = TXBUF_NORM(uart->tx_bufin + 1);
                uart_start_transmit(uart);
        }
}

/*
 * Device driver initialization and initial I/O configuration -- see uart.h
 */
extern void uart_init(device_t dev, ResourceType resource)
{
        uart_dev[dev].resource = resource;
}

extern void uart_config_cooked(device_t dev)
{
        uart_t  *uart = uart_dev + dev;

        uart->rx_data = uart_rx_cooked;
        uart->tx_data = uart_tx_cooked;
        uart->tx_echo = ECHO_ON;
}

extern void uart_config_raw(device_t dev)
{
        uart_t  *uart = uart_dev + dev;

        uart->rx_data = uart_rx_raw;
        uart->tx_data = uart_tx_raw;
}

extern void uart_config_slip(device_t dev)
{
        uart_t  *uart = uart_dev + dev;

        uart->rx_data = uart_rx_slip;
        uart->tx_data = uart_tx_slip;
}

extern void uart_config_xonxoff(device_t dev)
{
        uart_dev[dev].xonxoff = &xonxoff_operations;
}

/*
 * I/O controls -- see additional comment in uart.h
 */
extern void uart_set_cooked(device_t dev)
{
        uart_lock(dev);
        uart_config_cooked(dev);
        uart_unlock(dev);
}

extern void uart_set_raw(device_t dev)
{
        uart_lock(dev);
        uart_config_raw(dev);
        uart_unlock(dev);
}

extern void uart_set_slip(device_t dev)
{
        uart_t  *uart = uart_dev + dev;

        uart_lock(dev);
        uart->rx_slipesc = false;
        uart->tx_slipstate = 0;
        _uart_disable_xonxoff(uart);
        uart_config_slip(dev);
        uart_unlock(dev);
}

extern void uart_enable_xonxoff(device_t dev)
{
        uart_t  *uart = uart_dev + dev;
        ioreq_t *request;
        int     n;

        uart_lock(dev);
        uart->xoff_received = false;
        uart->xonxoff_urgent = 0;
        n = 0;
        for (request = uart->rx.head; request; request = request->next)
        {
                if (request->len > 0)
                {
                        n += request->len - request->pos;
                }
        }
        uart->xonxoff_rxroom = n;
        uart_config_xonxoff(dev);
        uart_start_transmit(uart);
        uart_unlock(dev);
}

extern void uart_disable_xonxoff(device_t dev)
{
        uart_lock(dev);
        _uart_disable_xonxoff(uart_dev + dev);
        uart_unlock(dev);
}

static void _uart_disable_xonxoff(uart_t *uart)
{
        uart->xonxoff = NULL;
        uart_start_transmit(uart);
}

extern void uart_echo(device_t dev, echo_t mode)
{
        uart_lock(dev);
        uart_dev[dev].tx_echo = mode;
        uart_unlock(dev);
}
