/******************************************************************************
 * FILE : @(#)t_target.h        1.40 04/11/15       
 * DESCRIPTION:                                                 
 *      Internal 8051 interface. Divided in two parts:         
 *      1. Internal stuff
 *      2. Interface with c_target.h 
 *****************************************************************************/
#ifndef _T_TARGET_H
#define _T_TARGET_H

#if (_os_OS_0_ORTI>0)
#include <time.h>
#define _os_CLOCKS_PER_SEC      CLOCKS_PER_SEC
#endif

#include <osek/osek.h>
#include "g_conf.h"

/****************************************
 * From c_target.h                      *
 ***************************************/
#define _os_INLINE            inline


/****************************************
 * internal 8051 target interface       
 ***************************************/

#define   __INTMYNO(nr)                ((8*nr) +3)
#define   _os_SaveIntExtendData      _os_SaveExtendData

#if (_os_OS_0_CORE == _os_OS_TSK51A)
#define _os_ENALLBIT  EA
#else 
#define _os_ENALLBIT  EAL
#endif

#define _os_ISR_TARGETSTATUS()     (_os_ENALLBIT==1)?1:0

/* in advance prototyping -see 'c_target.h'- */
extern  _os_TASK* volatile __idata _os_current_task;

#if (_os_OS_0_USERTOSTIMER ==1)
extern void InitRTOSTimer(void);
extern void EnableRTOSTimer(void);
extern void DisableRTOSTimer(void);
extern void ReloadRTOSTimer(void);      
#endif

extern char __idata _lc_bs[];
extern char __xdata * __data _SP;
extern char __xdata _os_VISRSTACK[];

#if (_os_OS_0_USERTOSTIMER ==1)
extern void _os_DummyExtractTimer(void);
#endif

/************************************************
 * In '_os_GoToTask': We restore the CPU        *
 * context by means of popping values from      *
 * the system stack.                            *
 * Pre: The system stack has been restored      *
 * for the next task to execute. SP is pointing *
 * at the top of the context values             *
 ***********************************************/
_os_INLINE
void _os_RestoreContextFromStack(void)
{
        _os_CORE_EXTRAPOPISR                    
        __asm("pop      __SP+1  ");     
        __asm("pop      __SP    ");
        __asm("pop      PSW     ");
        __asm("pop      AR7     ");
        __asm("pop      AR6     ");
        __asm("pop      AR5     ");
        __asm("pop      AR4     ");
        __asm("pop      AR3     ");
        __asm("pop      AR2     ");
        __asm("pop      AR1     ");
        __asm("pop      AR0     ");
        __asm("pop      DPH     ");
        __asm("pop      DPL     ");
        __asm("pop      B       ");
        __asm("pop      ACC     ");

}


/************************************************
 * In '_os_SaveTaskContext': We save the CPU    *
 * context by means of pushing values to        *
 * the system stack.                            *
 ***********************************************/
_os_INLINE
void _os_SaveContextToStack(void)
{
        __asm("push     ACC ");
        __asm("push     B   ");
        __asm("push     DPL ");
        __asm("push     DPH ");
        __asm("push     AR0 ");
        __asm("push     AR1 ");
        __asm("push     AR2 ");
        __asm("push     AR3 ");
        __asm("push     AR4 ");
        __asm("push     AR5 ");
        __asm("push     AR6 ");
        __asm("push     AR7 ");
        __asm("push     PSW ");         
        __asm("push    __SP   ");   
        __asm("push    __SP+1 ");

        _os_CORE_EXTRAPUSHISR   

}

/********************************************************       
 * It copies the saved context and system stack of the  *
 * next task to execute into the system stack.          *
 * -It is an online function since we cannot do RET     *
 * (system stack has been overwritten).                 *
 * - It cannot make usage of the extended area.         *
 *******************************************************/
_os_INLINE
void _os_RestoreSystemStack(void)
{
    uint_fast8_t i;
    char __idata * dst = &_lc_bs[0];
    uint8_t __xdata * src = (uint8_t __xdata*)_os_current_task->bstack;

    for (i=0;i<_os_current_task->sp;i++)
    {
        *dst++=*src++;
    }
    
}

/*****************************************************
 * Resets the _VSP to point at the interrupt virtual *
 * buffer (only at first nesting level               *
 ****************************************************/
_os_INLINE
void _os_SetVSPtoIS(void)
{
#if (_os_OS_0_VISRSTACK<2)
        _SP =  _os_VISRSTACK + 1;
#else
        _SP =  _os_VISRSTACK + _os_OS_0_VISRSTACK - 1;
#endif
}


extern void _os_SaveExtendData(void);

/********************************************
 * Definition requests from 'c_target.h'    *
 *******************************************/
#define _os_SYSTEM_STACK_NO     _os_NO_TASK

/*****************************************************************************
 * Task Initialization
 *****************************************************************************/
#if (_os_OS_0_ORTI>0)
_os_INLINE void _os_TargetORTITaskContext(TaskType tid)
{
}
#endif

/***********************************************
 * Interrupt handling:        
 * _os_TargetDisableInterrupts
 * _os_TargetEnableInterrupts       
 * _os_TargetDisableRTOSTimer       
 * _os_TargetEnableRTOSTimer        
 * _os_TargetSuspendAllISR2         
 * _os_TargetSesumeAllISR2          
 * _os_TargetSuspendOS              
 * _os_TargetResumeOS               
 ***********************************************/
_os_INLINE void _os_TargetDisableInterrupts(void)
{
        _os_ENALLBIT = 0;
}


_os_INLINE void _os_TargetEnableInterrupts(void)
{
        _os_ENALLBIT = 1;
}

#if ( (_os_NO_ALARM>0) && (_os_OS_0_USERTOSTIMER>0))
_os_INLINE void _os_TargetDisableRTOSTimer(void)
{
        /* needed: B */
        DisableRTOSTimer();
}

_os_INLINE void _os_TargetEnableRTOSTimer(void)
{
        /* needed: B */
        EnableRTOSTimer();
}
#endif

_os_INLINE void    _os_TargetSuspendAllISR2(void)
{
        /* not needed: B */
}

_os_INLINE void    _os_TargetResumeAllISR2(void)
{
        /* not needed: B */
}

extern void    _os_TargetSuspendOS(void);
extern void    _os_TargetResumeOS(void);


 /*****************************************************************************
 * Context switch:    
 * _os_TargetGoToTask
 * _os_TargetSaveTaskContext
 * _os_TargetSaveIntTaskContext
 * _os_TargetEnterScheduler
 * _os_TargetChangeStack
 *****************************************************************************/
extern void _os_TargetGoToTask(_os_RTOSMEM _os_TASK* tcb);
extern void _os_TargetSaveTaskContext(void);
extern void _os_TargetSaveIntTaskContext(void);

_os_INLINE void  _os_TargetEnterScheduler(void) {}

_os_INLINE void _os_TargetChangeStack(void) {}

/******************************************************************************
 * Miscellaneous:                       
 *      _os_ReturnToStartOS             
 *      _os_SaveStartOS                 
 *      _os_TargetClock               
 *****************************************************************************/
extern void _os_TargetReturnToStartOS(void);

_os_INLINE void _os_TargetSaveStartOS(void)
{
        /***********************************************
         * The context (as previous to this call is saved
         * in an special buffer). ShutdownOS() will
         * restore this context and the execution flow
         * gets back to the instruction next to
         * StartOS().
         ***********************************************/
        __asm(".extern  __os_MainTask");
        __asm(".extern  __os_current_task");
        
        __asm("push     AR0");
        __asm("mov      R0,#__os_current_task");
        __asm("mov      @R0,#@msb(__os_MainTask)");
        __asm("inc      R0");
        __asm("mov      @R0,#@lsb(__os_MainTask)");
        __asm("pop      AR0");
        

        /* R0- ..-R7 values and extended-data-area before 'StartOS'
         * must have prevailed */
        _os_TargetSaveTaskContext();
}

#if (_os_OS_0_ORTI>0)
_os_INLINE clock_t  _os_TargetClock(void)
{
        return clock();
}
#endif


#endif

