/******************************************************************************
 *  FILE:           @(#)t_stackmonitor.c        1.2 04/11/05
 *  DESCRIPTION:        Utility functions for the 8051 architecture
 *****************************************************************************/
#include "g_conf.h"
#include "c_target.h"
#include "c_common.h"

#if (_os_OS_0_STACKMONITOR >0)
extern char __idata _lc_ue_stack[];

/******************************************************************************
 * FUNCTION:    _os_TargetInitStackMonitor
 * DESCRIPTION: 
 *      It initializes the '_os_stack_monitor' array. The 8051 uses, a very 
 *      particular case, two stacks per task:
 *      - task's user stacks:    virtual stack: grows downwards 
 *      - task's system stacks:  system stack grows upwards.
 *      plus a couple of shared stacks:
 *      - global system stack in 'idata'
 *      - rtos virtual interrupt stack 
 * 
 *      The '_os_stack_monitor' deals only with stacks residing in
 *      external memory, i.e. the two stacks per task.
 *
 *      The 'virtual interrupt stack' (OIL configured) and 
 *      'global system stack' (memory qualifier) are considered to be 
 *      target stacks.
 *
 *****************************************************************************/
void _os_TargetInitStackMonitor(void)
{
        TaskType id;
        uint8_t __idata * iptr;

        /*********************************************
         * initialize the '_os_stack_monitor' array
         *********************************************/
        
        /* task virtual stacks: user stacks */
        for (id=0;id<_os_NO_TASK;id++)
        {
                _os_stack_monitor[id].address = _os_stack_address[id];
                _os_stack_monitor[id].growth  = _os_STACK_DOWNWARDS;
        }

        /* task system stacks */
        for (id=0;id<_os_NO_TASK;id++)
        {
                _os_stack_monitor[id+_os_NO_TASK].address = (_os_STACKMEM uint8_t *)
                        (uintptr_t)_os_stack_address[id] + _os_task_table[id].VSTACK 
                        + sizeof(_os_CTX) + _os_task_table[id].SSTACK - 1  ;
                _os_stack_monitor[id+_os_NO_TASK].growth  = _os_STACK_UPWARDS;
        }
        
        /***************************************************************** 
         * target stacks
         ****************************************************************/
        /* virtual interrupt stack */
        _os_VISRSTACK[0] = _os_STCK_PATTERN_1;
        _os_VISRSTACK[1] = _os_STCK_PATTERN_2;

         /* last two bytes of the global system stack with patterns */
        iptr            = (uint8_t __idata *)(uint8_t)_lc_ue_stack;  
        *(--iptr)       = _os_STCK_PATTERN_1;
        *(--iptr)       = _os_STCK_PATTERN_2;

        return;
}

uint_fast8_t _os_TargetStackMonitor(void)
{
        uint8_t __idata * iptr;  

        /********************************************************************** 
         * check that virtual interrupt stack lower limit has not been overwritten 
         *********************************************************************/
        if ( ((uint8_t)_os_VISRSTACK[1] != _os_STCK_PATTERN_2)
                        || ((uint8_t)_os_VISRSTACK[1] != _os_STCK_PATTERN_2) )
        {
                return 0;
        }
        
        /********************************************************************** 
         * check that global system stack upper limit has not been overwritten 
         *********************************************************************/
        iptr  = (uint8_t __idata *)(uint8_t)_lc_ue_stack;  
        if ( *(--iptr) != _os_STCK_PATTERN_1 || *(--iptr) != _os_STCK_PATTERN_2 )
        {
                return 0;
        }
        
        return 1;
}

uint_fast8_t _os_TargetIsStackInRange(void)
{
        /**************************************************************
         * check rtos virtual interrupt is still in range
         *************************************************************/
        if ( _os_IsInterruptLevel() &&  _SP < (_os_STACKMEM char *)&_os_VISRSTACK[0] )
        {
                return 0;
        }
        
        /************************************************************** 
         * check system stack is still in range.
         *************************************************************/
        if ((uint8_t)SP >=  (uint8_t) _lc_ue_stack - sizeof(_os_CTX) )
        {
                return 0;
        }
        
        return 1;
}

_os_STACKMEM uint8_t * _os_GetStackPointer(TaskType id)
{
        _os_STACKMEM uint8_t * ret = (_os_STACKMEM uint8_t *)0;
        
        /* virtual task's stack */
        if (id < _os_NO_TASK)
        {
                if ( _os_IsTaskLevel() && id == _os_current_task->user_id)
                {
                        /* let us test */
                        ret = (_os_STACKMEM uint8_t *) _SP; 
                }
                else
                {
                        /* test always true: tested with VISRSTACK in
                         * _os_TargetIsStackInRange() */
                        ret = (_os_STACKMEM uint8_t *)_os_stack_monitor[id].address + 1;
                }
        }
        /* system task's stack */
        else if ( id<2*_os_NO_TASK)
        {
                /* test always true: tested with SP / lc_ue_stack in
                 * _os_TargetIsStackInRange() */
                ret = (_os_STACKMEM uint8_t *)_os_stack_monitor[id].address - 1;
        }
        else
        {
                ShutdownOS(E_OS_SYS_ERROR);
        }
        /* compiler happy */
        return ret;
}

#endif

