/******************************************************************************
 *  FILE:       @(#)t_isr.c     1.25 04/10/06
 *  DESCRIPTION:
 *      ISR initialization and ISR ceiling protocol for
 *      the C51 architecture.
 *****************************************************************************/
#include "g_conf.h"
#include "c_target.h"
#include "c_common.h"

#if (_os_NO_ISR>0)
#if (MAX_RESOURCE_ISR>0)
uint_fast8_t _os_FASTALLOC _os_in_ceiling = 0;
#endif
#endif

#if ((_os_NO_ISR>0)||(_os_OS_0_USERTOSTIMER>0))
void _os_InitISR(void)
{
#if (MAX_RESOURCE_ISR>0)
        _os_ISR* isrptr;
        uint_fast8_t i;
        uint_fast8_t j;

        /* search all ISR objects */
        for (i=0,isrptr=&_os_isr_table[0];i<_os_NO_ISR;i++,isrptr++)
        {
                /* look up in all owned resources */
                for (j=0;j<MAX_RESOURCE_ISR && 
                        isrptr->RESOURCE[j] != _os_END_SEARCH; j++)
                {
                        /* all owned resources will have a non empty
                         * ceiling field */ 
                        _os_resource_table[isrptr->RESOURCE[j]].isrceiling= 1;
                }
        }
#       endif
        return;
}
#endif


#if (MAX_RESOURCE_ISR>0)
/*************************************************
 * Should happen:
 * - 'res' is owned by at least an ISR:
 *     - scheduler is locked.
 *     - all ISR2s which priority is less or
 *     equal than the maximum priority of
 *     all ISRs accessing the resource
 *     should be disabled.
 * - 'res' is not owned by any ISR:
 *     - returns
 *  
 * Instead:
 * - 'res' is owned by at least an ISR:
 *     - scheduler is locked.
 *     - all ISR2s are disabled.
 * - 'res' is not owned by any ISR:
 *     - returns
 *
 * Due to:
 * - Lack of hardware priority registers
 *   support.
 * - Many different derivatives.
 * - Optional
 * - Avoid complexity and overhead 
 *
 ***********************************************/    
void _os_GetISRCeilingProtocol(ResourceType res)
{
        _os_RESOURCE*   resptr = &_os_resource_table[res];
        
        if (resptr->isrceiling)
        {
                /* lock scheduler across system services.
                 * No other task can preempt this one
                 * until resource is released.
                 */
                _os_lock++;

                /* no ISR2 either */
                _os_SuspendOS();
                
                /* indicates that it is locked */
                resptr->preprio = 1;    
        }
        return;
}


/* reverse actions */
void _os_ReleaseISRCeilingProtocol(ResourceType res)
{
        _os_RESOURCE*   resptr = &_os_resource_table[res];
        
        if (resptr->isrceiling)
        {

                /* indicates that it is unlocked */
                resptr->preprio = 0;    

                /* allow ISR2 */
                _os_ResumeOS();
                
                /* unlock scheduler */
                _os_lock--;
        }
        return;
}
#endif

/****************************************************
 * MACRO:       _os_TargetSuspendOS()
 * VALUE:       
 * It disables the isr2 interrupts
 * adding a necessary nesting handling
 * if we run a program with Extended
 * Ceiling Priority Protocol to interrupt
 * levels.
 * Why: ISR2s are suspended when we get
 * resources which are owned by an ISR.
 * ISR2s resume in 'ReleaseResource'.
 * Special care is requested to handle
 * nested calls to '_os_suspend/resume_isr2'
 * in between.
 ****************************************************/
void    _os_TargetSuspendOS(void)
{
        uint_fast8_t local = _os_ISR_TARGETSTATUS();
        
        _os_DisableInterrupts();

#if (MAX_RESOURCE_ISR>0)
        if(_os_in_ceiling++==0)
        {
#endif
                _os_SuspendISR2();
                
#if (MAX_RESOURCE_ISR>0)
        }
#endif
        if (local)
        {
                _os_EnableInterrupts();
        }
}

void    _os_TargetResumeOS(void)
{
        uint_fast8_t local = _os_ISR_TARGETSTATUS();
        
        _os_DisableInterrupts();

#if (MAX_RESOURCE_ISR>0)
        if(--_os_in_ceiling==0)
        {
#endif
             _os_ResumeISR2();

#if (MAX_RESOURCE_ISR>0)
        }
#endif
        if(local)
        {
                _os_EnableInterrupts();
        }
}

