/****************************************************************
**  FILE:       @(#)t_cs.c      1.79 04/11/05 
**  DESCRIPTION:                                                *        
**  Implementation of context switching routines for 8051       *
****************************************************************/

#include "c_common.h"
#include "c_target.h"
#include "c_rtos.h"


/* statics */
static void     _os_RestoreExtendData(void);
static void     _os_SaveSystemStack(void);
static void     _os_SaveIntSystemStack(void);


/*******************************************************************************
 *  FUNCTION:              os_TargetGoToTask                                    *
 *  DESCRIPTION:                                                                *
 *  It restores the context of the current task and restarts its execution as   *
 *  it has never been interrupted. OS Interrupts must be kept disabled during   *
 *  the process. The scheduler should be the only user of this routine since    *
 *  it is the only one with task dispatching knowledge.                         *
 *  It does never return, the control is passed to the given task who           *
 *  starts executing from the address indicated in its context.                 *
 *******************************************************************************/
void _os_TargetGoToTask(_os_TASK* tcb)
{
    __asm(".extern  __SP");
    (void)tcb;

#if (_os_OS_0_EXTDATASIZE>0)
    /****************************************************
     * This function restores the extended              *
     * data area of the task.                           *
     ***************************************************/
    _os_RestoreExtendData();
#endif

    /************************************************
     * This 'inline' routine restores the context   *
     * and the system stack. It is inline since we  *
     * modify the whole systam stack, thus          *
     * there can be no 'ret' instruccion            *
     * ATT: The compiler cannot generate code that  *
     * use extended area for this routine (it would *
     * overwrite what has been previously restored) *
     * Otherwise we do it in assembly               * 
     ***********************************************/ 
    _os_RestoreSystemStack();

     /* Update the SP to our convenience */
     SP = _os_current_task->sp + (int_fast8_t)&_lc_bs -1;      
    
     /* disable interrupts globally */
    _os_DisableInterrupts();
    
    /* let us resume Category 2 interrupts */
    _os_ResumeOS();   
        
     /*****************************
     * Restore the task context  *
     ****************************/
    _os_RestoreContextFromStack();

    /* we allow NOW rescheduling */
    _os_EnableInterrupts();
        
    __asm("reti         "); 

    /* for the compiler */
        return;         

}


/********************************************************************************
 *  FUNCTION:              _os_TargetSaveTaskContext                            *
 *  DESCRIPTION:                                                                *
 *  This function saves the current working environment when a task is entering *
 *  the waiting state. The task shall resume execution at the next instruction  *
 *  of the caller of the callers code (i.e. next to '_os_wait()')               *
 *  Interrupts are disabled.                                                    *
 *******************************************************************************/
void _os_TargetSaveTaskContext(void)
{
        __asm(".extern  __lc_bs"); 
        
    /************************************************
     * Push context at the end of system stack  *
     ************************************************/
    _os_SaveContextToStack();

#if (_os_OS_0_EXTDATASIZE>0)
    /********************************
     * Save current extended area.  *
     *******************************/
    _os_SaveExtendData();
#endif

    /********************************
     * Save current system stack    *
     *******************************/
    _os_SaveSystemStack();

    /********************************************** 
     * update the SP in the backup area. 
     * Be aware we copy all stack but two bytes:
     * the return address of this call!
     *********************************************/
    _os_current_task->sp = SP-1-(uint8_t)&_lc_bs;

    /********************************
     * Let us return correctly  *
     *******************************/
    SP = SP - _os_NOBYTESCONTEXTPOPPUSH;
    
    return;      

}


/********************************************************************************
 *  FUNCTION:              _os_TargetSaveIntTaskContext                         *
 *  DESCRIPTION:                                                                *
 *  Saves the context and the system stack of the interrupted task.             *
 *  Used in ISR2 of upon of expiration of an alarm.                             *
 *  Whenever the scheduler dispatches the task again, the existing environment  *
 *  of the task before the interrupt shall  fully recover.                      *
 *                                                                              *
 *  REMARKS:                                                                    *
 *  1- The interrupt handler must have beed defined as:                         *
 *   _interrupt(1) _frame(PSW, A, B, DPL, DPH, R0, R1, R2, R3, R4, R5, R6, R7)  * 
 *      void IntHandler( void )                                                 *
 *  specifying which registers and SPRs are pushed and popped already by the    *   
 *  interrupt function itself ( compiler ).                                     *
 *  2-  interrupts must be currently disabled.                                  *
 *  3- The data_extend area has been saved already                              *
 *      (first to do in the ISR code).                                          *
 *******************************************************************************/  
void _os_TargetSaveIntTaskContext(void)
{
    /********************************
     * Save the system stack from   *
     * interrupt level              *
     *******************************/
    _os_SaveIntSystemStack();
    
    /********************************************** 
     * update the SP in the backup area. 
     * Be aware we copy all stack but four bytes:
     *  the return addressies inside the isr.
     *********************************************/
    _os_current_task->sp = SP-1-(uint8_t)&_lc_bs;
    return;
}


#if (_os_OS_0_EXTDATASIZE>0)
/************************************************************************
 * FUNCTION:    _os_SaveExtendData                                      *
 * DESCRIPTION:                                                         *
 *  It saves the current extended data area.                            *
 ***********************************************************************/
void _os_SaveExtendData(void)
{
    uint8_t __idata *src = (uint8_t __idata *)(uintptr_t)&_lc_ub_data_extend;
    uint8_t __xdata *dst = (uint8_t __xdata*)_os_current_task->beda;
    uint_fast8_t i;
    
    for (i=0;i<(uint8_t)&_lc_ue_data_extend-(uint8_t)&_lc_ub_data_extend;i++)
    {
        *dst++ = *src++;
    }
    
    return; 
}

/************************************************************************
 * FUNCTION:    _os_RestoreExtendData                                   *
 * DESCRIPTION:                                                         *
 *  It restores in the right place the extended data area.              *
 ***********************************************************************/ 
static void _os_RestoreExtendData(void)
{
    uint8_t __idata *dst = (uint8_t __idata *)(uintptr_t)&_lc_ub_data_extend;
    uint8_t __xdata *src = (uint8_t __xdata*)_os_current_task->beda;
    uint_fast8_t i;
    
    for (i=0;i<(uint8_t)&_lc_ue_data_extend-(uint8_t)&_lc_ub_data_extend;i++)
    {
        *dst++ = *src++;
    }
    
    return; 
}
#endif

/************************************************************************
 * FUNCTION:    _os_SaveSystemStack                                     *
 * DESCRIPTION:                                                         *
 *  It saves thecurrent system stack.                                   *
 ***********************************************************************/
static void _os_SaveSystemStack(void)
{
    uint_fast8_t i;
    uint8_t __idata* src = (uint8_t __idata*)(uintptr_t)&_lc_bs;
    uint8_t __xdata* dst = (uint8_t __xdata*)_os_current_task->bstack;

    /************************************************************
     * We copy the system stack to the backup                   *
     * area of the uitgoing task.                               *
     * - We skip copying the two bytes that should              *
     *   correspond to the return from _os_SaveTaskContext      *
     *   (task resumes after 'os_wait').                        *
     * - The '2' in the upper limit of the copy loop is due     *
     *   to the usage of this subroutine.                       *
     ***********************************************************/
    for (i=0;i<=SP-(uint8_t)&_lc_bs -2;i++,src++)
    {
        if (
            i < SP-_os_NOBYTESCONTEXTPOPPUSH-(uint8_t)&_lc_bs-3 ||
            i > SP-_os_NOBYTESCONTEXTPOPPUSH-(uint8_t)&_lc_bs-2
           )
        {
            *dst++=*src;
            
        }
    }
#if (_os_OS_0_ORTI>0)
    /* user's return address after WaitSwitch() or TaskSwitch() */
    _os_current_task->pc = (uintptr_t)*(src-sizeof(_os_CTX)-4);
    _os_current_task->pc |= (uintptr_t)(*(src-sizeof(_os_CTX)-3))<<8;
#endif
}

/************************************************************************
 * FUNCTION:    _os_SaveIntSystemStack                                  *
 * DESCRIPTION:                                                         *
 *  It saves the current system stack (from interrupt level)            *
 ***********************************************************************/
static void _os_SaveIntSystemStack(void)
{
    uint_fast8_t i;
    uint8_t __idata* src = (uint8_t __idata*)(uintptr_t)&_lc_bs;
    uint8_t __xdata* dst = (uint8_t __xdata*)_os_current_task->bstack;

    /********************************************************************
     * We copy the system stack to the backup                           *
     * area of the uitgoing task.                                       *
     * - We do not copy the last 2 bytes since they                     *
     *   store the return address of SaveIntTaskContext/..              *
     *   interrupt level (nothing to do with the interrupted task.      *
     *******************************************************************/
    for (i=0;i<SP-(uint8_t)&_lc_bs-3;i++)
    {
        *dst++=*src++;  
    }
#if (_os_OS_0_ORTI>0)
    /* user's return address after WaitSwitch() or TaskSwitch() */
    _os_current_task->pc = (uintptr_t)*(src-sizeof(_os_CTX));
    _os_current_task->pc |= (uintptr_t)(*(src-sizeof(_os_CTX)+1))<<8;
#endif
    
}


/****************************************************************
 * FUNCTION:            _os_TargetRunTimeTaskContext            *
 * DESCRIPTION:                                                 *
 *      This routine ends the initialization of the task        *
 *      control block and the task context with everything      *
 *      that could not be initialized by the 'init' of          *
 *      the compiler. Only target parts.                        *
 *      Most of the information can be provided at compile      *
 *      time ( see _os_TASK_TARGETRTOSINIT macro in t_rtos.h).  *
 * PRE:                                                         *
 *      'init' of the compiler has been executed.               *
 * POST:                                                        *
 *      The control block and context of the given task         *
 *      is ready for the first activation.                      *
 **************************************************************/
void _os_TargetRunTimeTaskContext(TaskType task_id)
{
    _os_RTOSMEM  _os_TASK* task = (_os_RTOSMEM _os_TASK*)&_os_task_table[task_id];
    _os_STACKMEM _os_CTX * ctx = (_os_STACKMEM _os_CTX*) ((uintptr_t) _os_stack_address[task_id] + 
                                                          task->VSTACK );

    ctx->pch  = (uint8_t)((uint16_t)task->ADDRESS  >> 8); 
    ctx->pcl  = (uint8_t)((uint16_t)task->ADDRESS); 
    ctx->vsph = (uint8_t)((uint16_t)ctx  >> 8) ;
    ctx->vspl = (uint8_t)((uint16_t)ctx) ;

     /* Set the begin of sstack and eda for the task */
     task->bstack = (uint16_t)ctx;
#if (_os_OS_0_EXTDATASIZE>0)
     task->beda   = (uint16_t)ctx + task->SSTACK + sizeof(_os_CTX);
#endif

     return;
}

/********************************************************
 * FUNCTION:            _os_TargetResetTaskContext      *
 * DESCRIPTION:                                         *
 *      This routine is called from the RTOS reset      *
 *      component in order to reset the task control    *
 *      and context parts which value could have        * 
 *      been changed at run time (only target parts).   *
 *      It is used in:                                  *
 *      - TerminateTask                                 *
 *      - ChainTask                                     *
 *      - ShutdownOS (if MULTISTART, only for           *
 *              non-suspended tasks).                   *
 *                                                      *
 *      Among typical action, the reset of:             *
 *      - task's start address.                         *
 *      - some context registers.                       *
 *      - stack pointers                                *
 *      - ..                                            *
 *                                                      *
 * PARAMETERS: id of the task                           *
 *                                                      *
 * POST:                                                *
 *      The control block and context of the given task *
 *      is ready for a new activation.                  *       
 *******************************************************/
void _os_TargetResetTaskContext(TaskType task_id)
{
    _os_RTOSMEM  _os_TASK* task = (_os_RTOSMEM _os_TASK*)&_os_task_table[task_id];
    _os_STACKMEM _os_CTX * ctx = (_os_STACKMEM _os_CTX*) ( (uintptr_t) _os_stack_address[task_id] + 
                                                          task->VSTACK );
    
    /* Reset context */     
    task->sp    = sizeof(_os_CTX) ;
    ctx->pcl  = (uint8_t)((uint16_t)task->ADDRESS);
    ctx->pch  = (uint8_t)((uint16_t)task->ADDRESS  >> 8); 
    ctx->vsph = (uint8_t)(task->bstack  >> 8) ;
    ctx->vspl = (uint8_t)(task->bstack);
    ctx->psw  = 0;

    /* not necessary */
    ctx->ro = 0;
    ctx->r1 = 1;
    ctx->r2 = 2;
    ctx->r3 = 3;
    ctx->r4 = 4;
    ctx->r5 = 5;
    ctx->r6 = 6;
    ctx->r7 = 7;
    
    return;
}


/******************************************************************
 * The routines below are necessary when the system service
 * 'ShutdownOS' is called. The system stops all rtos activities
 * and gets back to where it was before 'StartOS' was called.
 * Therefore mechanisms must exist to save/restore the context
 * as it was before 'StartOS'
 *****************************************************************/

/**********************
 * Main stack buffer
 *********************/
typedef struct _os_buffer_main _os_MAINBUFFER;
struct _os_buffer_main
{
        /* no need to buffer virtual stack bytes,
         * they keep being always in the allocated 
         * space for the virtual stack in the
         * linker/locator options.
         *
         * uint8_t  vstack[_os_OS_0_VMAINSTACK];
         */
        _os_CTX        ctx;
        uint8_t  sstack[_os_OS_0_SMAINSTACK];
#if (_os_OS_0_EXTDATASIZE>0)
        uint8_t  eda[_os_OS_0_EXTDATASIZE];
#endif
};

/******************************************* 
 * no need to initialize since we do never 
 * do '_os_GoToTask' with this as current
 * task.
 *******************************************/
_os_MAINBUFFER _os_MAIN_stackbase;

/********************************************************
 * We build this 'fake' _os_TASK object to make use     *
 * of the existing routines to save/restore the main    *
 * context.                                             *
 * The ONLY fields we need to initialize are:           *
 * - bstack (where to copy sstack + cpu ctxt)           *
 * - beda   (where to copy extended area)               *
 * - VSTACK,SSTACK                                      *
 *******************************************************/
_os_TASK _os_MainTask = 
{
         (uint16_t)&_os_MAIN_stackbase.ctx.pcl, /* bstack */
#if (_os_OS_0_EXTDATASIZE>0)                            
         (uint16_t)&_os_MAIN_stackbase.eda[0],  /* beda */
#endif          
         0,                                             /* sp */
         SUSPENDED,
#if (_os_OS_0_ONE_TASK_PER_PRIO !=1)
         (_os_TASK*)0,                                  /* state */
#endif
         0,                     /* rc */
         0,                     /* staticprio */
#if (_os_NO_EVENT>0)
         0,                     /* events */
         0,                     /* set */
         0,                     /* wait */
#endif
         0,                     /* user_id */
         0,                     /* repeat */
#if (_os_OS_0_ORTI>0)
         0,                     /* noruns */
         0,                     /* pc */
         0,                     /* servicetrace */
         0,                     /* errid */
         0,                     /* cycles */
#endif
#if ((_os_OS_0_STATUS == _os_OS_EXTENDED)||(_os_OS_0_ORTI>0))
        _os_INVALID_RES,        /* res */
#endif
         _os_INVALID_RES,       /* internal */
         0,                      /* PRIORITY    */
         _os_TA_FULL,            /* SCHEDULE    */
         0,                      /* ACTIVATION  */
         0,                      /* AUTOSTART   */
         { -1 },                 /* APPMODE[]   */
         { -1 },                 /* RESOURCE[]  */
         { -1 },                 /* MESSAGE[]   */
         (uintptr_t)0,           /* ADDRESS     */
         0,                      /* EVENTS      */
         0,                      /* VSTACK      */
         _os_OS_0_SMAINSTACK,    /* SSTACK      */
};




void _os_TargetReturnToStartOS(void)
{
        _os_current_task = (_os_TASK*)&_os_MainTask;
        
#if (_os_OS_0_EXTDATASIZE>0)
    _os_RestoreExtendData();
#endif
    _os_RestoreSystemStack();

    SP = _os_current_task->sp + (uint8_t)&_lc_bs -1;      
     
    _os_RestoreContextFromStack();

    __asm("reti         "); 

    /* never here */
    return;
}




