/* @(#)ioreq.c  1.2 */

/*
 * The caller must protect against concurrent updates by, for example,
 * disabling interrupts. It is a bit pointless to do it here because
 * the caller needs to do that anyway: most of the time other information
 * must be kept in sync.
 */

#include <stddef.h>
#include <errno.h>
#include <osek/ioreq.h>

/*
 * Put a new request in the queue. The return value is true when the queue
 * was empty and becomes nonempty (i.e. upon transition to nonempty state
 * only).
 */
extern bool_t ioreq_put(ioqueue_t *queue, ioreq_t *request)
{
        bool_t  start;

        request->pos = 0;
        request->err = 0;
        request->queue = queue;
        request->next = NULL;
        start = false;
        if (queue->head)
        {
                queue->tail->next = request;
        }
        else
        {
                queue->head = request;
                start = true;
        }
        queue->tail = request;
        return start;
}

/*
 * Get the currently active request, to be called from an ISR servicing the
 * queue. This does not dequeue the request, i.e. multiple calls will yield
 * the same request until it has explicitly been completed by ioreq_end().
 */
extern ioreq_t *ioreq_get(ioqueue_t *queue)
{
        ioreq_t         *request;

        while ((request = queue->head))
        {
                if (request->len > 0)
                {
                        break;
                }
                ioreq_end(request);
        }
        return request;
}

/*
 * Abort the currently active request and notify as usual.
 */
extern ioreq_t *ioreq_abort(ioqueue_t *queue)
{
        ioreq_t *request;

        if ((request = queue->head))
        {
                request->err = -ECANCELED;
                ioreq_end(request);
        }
        return request;
}

/*
 * Notify and activate next request, if there is any.
 */
extern void ioreq_end(ioreq_t *request)
{
        if (request->task != INVALID_TASK)
        {
                SetEvent(request->task, request->mask);
        }
        request->queue->head = request->next;
}
