/* @(#)host.c   1.2 */

/*
 * RESOURCES:   RES_SCHEDULER
 */

#include <stddef.h>
#include <errno.h>
#include <osek/irq.h>

#include <osek/host.h>

static  void            host_end_request(ioreq_t *request, int n);

/*
 * NOTES
 * -----
 *
 * Host file-system:
 * This is implemented by a debug trap mechanism where the debugger will
 * set a breakpoint on either _fss_break() or _dbg_trap(). The latter is
 * a more generic mechanism which will supersede the old method, eventually.
 * After the trap the debugger will execute the I/O request and write back
 * the result into target memory. Finally it will resume target execution.
 *
 * errno:
 * The host file-system access routines _do_ set errno. But the upper layers
 * are agnostic to this so we need to return it as usual. We deliberately do
 * not save-restore the global errno variable. The errno here should _never_
 * be aliased to a task specific one but simply be that ONE global variable.
 *
 * Locking:
 * Stopping target execution is effectively the same as disabling _all_ interrupts.
 * So, we might as well do that ourselves and we need a lock around the _host_*()
 * calls anyway because the command struct accessed inside is a global one.
 *
 * Event notification:
 * This is pointless as all debugger traps are synchronous and target execution
 * is stopped. However, we need to conform to the generic interface. For the
 * same reason it is not possible to abort an I/O request: the request is either
 * not done yet or already completed.
 */

extern int_least8_t host_open(const char *pathname, uint_least16_t flags)
{
        irqflags_t      irqflags;
        _err_t          ret;

        irqsuspend(irqflags);
        ret = _host_open(pathname, flags);
        irqrestore(irqflags);
        return ret < 0 ? -errno : ret;
}
        
extern void host_read(device_t dev, ioreq_t *request)
{
        irqflags_t      irqflags;
        int             n;

        n = 0;
        irqenter(irqflags);
        if (request->len > 0)
        {
                n = _host_read((_fd_t)dev, request->buf, request->len);
        }
        host_end_request(request, n);
        irqleave(irqflags);
}

extern ioreq_t *host_read_abort(device_t dev)
{
        return NULL;
}

extern void host_write(device_t dev, ioreq_t *request)
{
        irqflags_t      irqflags;
        int             n;

        n = 0;
        irqenter(irqflags);
        if (request->len > 0)
        {
                n = _host_write((_fd_t)dev, request->buf, request->len);
        }
        host_end_request(request, n);
        irqleave(irqflags);
}

extern ioreq_t *host_write_abort(device_t dev)
{
        return NULL;
}

extern long host_lseek(device_t dev, long offset, uint_least8_t whence)
{
        irqflags_t      irqflags;

        irqsuspend(irqflags);
        offset = _host_lseek((_fd_t)dev, offset, whence);
        irqrestore(irqflags);
        return offset < 0 ? -errno : offset;
}

extern int_least8_t host_close(device_t dev)
{
        irqflags_t      irqflags;
        _err_t          ret;

        irqsuspend(irqflags);
        ret = _host_close((_fd_t)dev);
        irqrestore(irqflags);
        return ret < 0 ? -errno : ret;
}

static void host_end_request(ioreq_t *request, int n)
{
        if (n < 0)
        {
                request->pos = 0;
                request->err = -errno;
        }
        else
        {
                request->pos = n;
                request->err = 0;
        }
        if (request->task != INVALID_TASK)
        {
                SetEvent(request->task, request->mask);
        }
}
