/******************************************************************************
 * FILE:        @(#)c_target.h  1.60 04/11/10
 * DESCRIPTION:
 *      Common Target Interface for an RTOS implementation.
 *      The description of this interface must be generic and
 *      target independent (it isolates the platform details).
 *
 *      Porting the RTOS to a new target implies the definition
 *      of this interface in the new target files ('t_*')
 ****************************************************************************/
#ifndef _C_TARGET_H
#define _C_TARGET_H

#include <osek/osek.h>

#include "c_common.h"
#include "t_target.h"

/*****************************************************************************
 * FUNCTION:    _os_IsInterruptLevel, _os_IsTaskLevel
 * DESCRIPTION:
 *      These inline functions return 1 when called from 
 *      interrupt/task level.
 *      IDEA: We are at interrupt level only if _os_isr2_counter >0.
 *
 * REMARK: Strictly speaking, this would not be generally true 
 *         since the services
 *              - Disable/EnableAllInterrupts 
 *              - Suspend/ResumeAllInterrupts
 *              - Suspend/ResumeOSInterrupts
 *         can be also reached from an ISR1.  
 *         It is however true in practice since this code
 *         is not used in any of the above routines. 
 ****************************************************************************/
_os_INLINE
uint_fast8_t       _os_IsInterruptLevel(void)
{ 
#if (_os_NO_ISR>0)
        return _os_isr2_counter;
#else
        return 0;
#endif
}

_os_INLINE      
uint_fast8_t       _os_IsTaskLevel(void) 
{       
#if (_os_NO_ISR>0)
        return _os_isr2_counter==0;
#else
        return 0;
#endif
}

/******************************************************************************
 * _os_SIA : Software Interrupt Available 
 * determines whether the current target provides
 * support (and it is used) of software interrupts
 * for the implementation of OSEK system services (see types.h)
 * 1: Software interrupts/traps used
 * 0: A system service is a normal function call
 ****************************************************************************/
#define _os_SIA                 _os_USETRAPS

/* source code uses legacy definitions.
 * Parameters in 'LogError' */
#define _os_TO_SEM_TAKE                 _os_N_GetResource       
#define _os_TO_SEM_GIVE                 _os_N_ReleaseResource    
#define _os_TO_START_OS                 _os_N_StartOS
#define _os_TO_APP_MODE                 _os_N_GetActiveApplicationMode    
#define _os_TO_SHUTDOWN                 _os_N_ShutdownOS
#define _os_TO_START_COM                _os_N_StartCOM    
#define _os_TO_STOP_COM                 _os_N_StopCOM    
#define _os_TO_COMAPP_MODE              _os_N_GetCOMApplicationMode    
#define _os_TO_TASK_GETID               _os_N_GetTaskID    
#define _os_TO_ACTIVATE_TASK            _os_N_ActivateTask    
#define _os_TO_TERMINATE_TASK           _os_N_TerminateTask      
#define _os_TO_TASK_STATE               _os_N_GetTaskState      
#define _os_TO_SCHEDULE                 _os_N_Schedule    
#define _os_TO_CHAIN_TASK               _os_N_ChainTask   
#define _os_TO_BASE_ALARM               _os_N_GetAlarmBase    
#define _os_TO_GET_ALARM                _os_N_GetAlarm    
#define _os_TO_SETREL_ALARM             _os_N_SetRelAlarm    
#define _os_TO_SETABS_ALARM             _os_N_SetAbsAlarm    
#define _os_TO_CANCEL_ALARM             _os_N_CancelAlarm    
#define _os_TO_SET_EVENT                _os_N_SetEvent    
#define _os_TO_GET_EVENT                _os_N_GetEvent    
#define _os_TO_WAIT_EVENT               _os_N_WaitEvent    
#define _os_TO_CLEAR_EVENT              _os_N_ClearEvent    
#define _os_TO_INC_COUNTER              _os_N_IncrementCounter    
#define _os_TO_INIT_MESSAGE             _os_N_InitMessage      
#define _os_TO_SEND_MSG                 _os_N_SendMessage    
#define _os_TO_REC_MSG                  _os_N_ReceiveMessage    
#define _os_TO_GET_MSGSTATUS            _os_N_GetMessageStatus    

/******************************************************************************
 * Task initialization:
 *         _os_InitIdle
 *         _os_TargetRunTimeTaskContext
 *         _os_TargetResetTaskContext
 *         _os_ORTITaskContext
 ****************************************************************************/

/******************************************************************************
 * FUNCTION:            _os_InitIdle
 * DESCRIPTION:
 * Initializes the run context of the 'idle' task.
 * This function executes during 'StartOS' to
 * ensure that at least an 'idle' task can be
 * always scheduled. Implementors of this function
 * must realize that this task only performs to
 * state transitions:
 *  - ready -> running : No user's tasks are ready
 *  - running -> ready : One (or more) user's task is ready.
 *                                                      
 * The implementor must use '_os_set_task_ready()'
 * to populate the 'ready-to-run' queue in the body of this routine.
 *                                                      
 * POST:      
 *      The idle task occupies the zero-th level of the ready-to-run queue.
 ****************************************************************************/
void    _os_InitIdle(void);


/******************************************************************************
 * FUNCTION:            _os_TargetRunTimeTaskContext
 * DESCRIPTION:
 *      This routine ends the initialization of the task
 *      control block and the task context with everything
 *      that could not be initialized by the 'init' of
 *      the compiler ( Only target parts!).
 *      Most of the information can be provided at compile
 *      time ( see _os_TASK_TARGETRTOSINIT macro in t_rtos.h).
 * PRE:
 *      'init' of the compiler has been executed.
 * POST:
 *      The target parts of the control block and context
 *      of the given task are ready for the first activation.
 ****************************************************************************/
void _os_TargetRunTimeTaskContext(TaskType);


/******************************************************************************
 * FUNCTION:            _os_TargetResetTaskContext
 * DESCRIPTION:
 *      This routine is called from the RTOS reset
 *      component in order to reset the task control
 *      and context target parts which value could have 
 *      been changed at run time.
 *      It is used in:
 *      - TerminateTask
 *      - ChainTask
 *      - ShutdownOS (if MULTISTART, only for non-suspended tasks).
 *
 *      Among typical action, the reset of:
 *      - task's start address.
 *      - some context registers.
 *      - stack pointers
 *      - ..            
 *
 * PARAMETERS: id of the task
 *
 * POST:
 *      The target parts of the control block and
 *      context of the given task are ready for the first activation.
 ****************************************************************************/
void _os_TargetResetTaskContext(TaskType t);

#if (_os_OS_0_ORTI>0)
/******************************************************************************
 * FUNCTION:            _os_ORTITaskContext
 * DESCRIPTION:
 *      Initializes the ORTI fields of the given task.
 ****************************************************************************/
_os_INLINE void _os_ORTITaskContext(TaskType id)
{
        _os_TargetORTITaskContext(id);
}
#endif



/******************************************************************************
 * FUNCTION:            _os_EnterWait
 * DESCRIPTION:
 *      Target specific actions to enter a mode of less power consumption
 *      while still handling interrupts.
 ****************************************************************************/
void _os_EnterWait(void);

/*****************************************************************************
 * Timer initialization
 *       _os_StartTimer
 ****************************************************************************/

/******************************************************************************
 * FUNCTION:            _os_StartTimer
 * DESCRIPTION:
 *  It initializes the hardware of the rtos timer.
 *  Called during the system upbringing.
 * POST:
 *  rtos timer interrupts are enabled.
 ****************************************************************************/
#if ( (_os_NO_ALARM>0) && (_os_OS_0_USERTOSTIMER>0))
void _os_StartTimer(void);
#endif

/******************************************************************************
 * Interrupt handling:
 *      _os_InitISR
 *      _os_GetISRCeilingProtocol
 *      _os_ReleaseISRCeilingProtocol
 *      _os_DisableInterrupts
 *      _os_EnableInterrupts
 *      _os_DisableRTOSTimer
 *      _os_EnableRTOSTimer
 *      _os_SuspendAllISR2
 *      _os_ResumeAllISR2
 *      _os_SuspendISR
 *      _os_ResumeISR
 *      _os_SuspendOS
 *      _os_ResumeOS
 ****************************************************************************/

#if ( ( _os_NO_ISR > 0 ) || (_os_OS_0_USERTOSTIMER>0) )
/******************************************************************************
 * FUNCTION:    _os_InitISR
 * DESCRIPTION:
 *      Target Initialization specific (if any) actions for ISRs.
 ****************************************************************************/
void    _os_InitISR(void);
#endif

#if (MAX_RESOURCE_ISR>0)
/******************************************************************************
 * FUNCTION:    _os_ReleaseISRCeilingProtocol
 * DESCRIPTION:
 *      ISR Extension Ceiling Protocol : Release
 *      Target dependent
 ****************************************************************************/
void    _os_ReleaseISRCeilingProtocol(ResourceType);

/******************************************************************************
 * FUNCTION:    _os_GetISRCeilingProtocol
 * DESCRIPTION:
 *      ISR Extension Ceiling Protocol
 *      Target dependent.
 ****************************************************************************/
void    _os_GetISRCeilingProtocol(ResourceType);
#endif


/******************************************************************************
 * FUNCTION:    _os_DisableInterrupts
 * DESCRIPTION:
 *  It disables maskable interrupts globally
 ****************************************************************************/
_os_INLINE void    _os_DisableInterrupts(void)
{
        _os_TargetDisableInterrupts();
}

/******************************************************************************
 * FUNCTION:    _os_EnableInterrupts
 * DESCRIPTION:
 *  It enables maskable interrupts globally
 * REMARK:
 *      _os_TargetEnableInterrupts in t_target.h
 ****************************************************************************/
_os_INLINE void    _os_EnableInterrupts(void)
{
        _os_TargetEnableInterrupts();
}

/******************************************************************************
 * FUNCTION:    _os_(Disable/Enable)RTOSTimer
 * DESCRIPTION:
 *  Disables/Enables the rtos timer
 ****************************************************************************/
_os_INLINE void    _os_DisableRTOSTimer(void)
{
#if ( (_os_NO_ALARM>0) && (_os_OS_0_USERTOSTIMER>0))
        _os_TargetDisableRTOSTimer();
#endif
}

_os_INLINE void    _os_EnableRTOSTimer(void)
{
#if ( (_os_NO_ALARM>0) && (_os_OS_0_USERTOSTIMER>0))
        _os_TargetEnableRTOSTimer();
#endif
}

/******************************************************************************
 * FUNCTION:    _os_SuspendAllISR2
 * DESCRIPTION:
 *  Suspends globally all ISR2s interrupts by raising
 *  priority level of the current process above the highest priority 
 *  among ISR2s.                       
 ****************************************************************************/
_os_INLINE void    _os_SuspendAllISR2(void)
{
        _os_TargetSuspendAllISR2();
}

/******************************************************************************
 * FUNCTION:    _os_ResumeAllISR2
 * DESCRIPTION:
 *  Resumes globally all ISR2s interrupts. (Reverse actions).
 ****************************************************************************/
_os_INLINE void    _os_ResumeAllISR2(void)
{
        _os_TargetResumeAllISR2();
}

/******************************************************************************
 * MACRO:       _os_(SuspendResume)ISR
 * DESCRIPTION:
 *      Suspends the ISR given as parameter to the macro. Defined in t_rtos.h.
 ****************************************************************************/


/******************************************************************************
 * REMARK:
 * The generated '_os_SuspendISR2' routine looks like:
 * 
 * void _os_SuspendISR2( void )
 * {
 *       // Interface for architectures where
 *       // all ISR2s can be disabled at once
 *       _os_SuspendAllISR2();
 *
 *       // Interfaces for architectures where
 *       // ISR2 must be disabled one by one
 *       _os_DisableRTOSTimer();
 *       _os_SuspendISR( 0 );
 * }
 *
 * '_os_SuspendISR2' is called from '_os_SuspendOS'
 * to protect critical areas in the rtos code against
 * concurrency. Critical rtos data areas are those
 * that can be accessed from ISR2 code.
 *
 * Some architectures (A) disable globally the OS interrupts:
 * - raising the processor execution level above the
 *   highest isr2 level: m32c, tricore
 * - there is no other chance (imagine one only line
 *   to all external interrupts): z80 with interrupt
 *   controller.
 * In these cases, '_os_DisableRTOSTimer' and
 * '_os_SuspendISR' MUST be dummy. 
 *
 * Other architectures (B) cannot disable globally
 * ISR2s (there is no concept of 'process execution
 * level') and need to disable ISR2 by ISR2: ex. c51
 * In these cases, '_os_SuspendAllISR2' must be dummy.
 *
 * QUESTION: Why is not '_os_SuspendISR2' equivalent
 * to '_os_SuspendOS'?                     
 * ANSWER: Because, in some architectures (B) the
 * extension to interrupt level of the ceiling
 * priority protocol needs an extra handling for
 * possible nested calls to '_os_SuspendISR2'.
 ****************************************************************************/


/******************************************************************************
 * FUNCTION:    _os_(Suspend/Resume)OS
 * DESCRIPTION:
 *      RTOS code make use of these interfaces
 *      to protect its internal critical code.
 *              ...
 *              _os_SuspendOS();
 *              critical_code() - ISR1 allowed -
 *              _os_ResumeOs();
 *
 * REMARKS:
 *      Definitions for these interfaces will typically
 *      consist of a just a call to
 *              '_os_(Suspend/Resume)ISR2'
 *      Ex. Arch (A), or Arch (B) without extension of
 *      the ceiling priority protocol, never nested.
 *      In Arch B with extension .., could be nested.
 *
 *      They can always go nested with calls to
 *      _os_EnableInterrupts/_os_DisableInterrupts
 ****************************************************************************/
_os_INLINE void    _os_SuspendOS(void)
{
        _os_TargetSuspendOS();
}

_os_INLINE void    _os_ResumeOS(void)
{
        _os_TargetResumeOS();
}

/******************************************************************************
 * Context switch:
 *       _os_GoToTask
 *       _os_SaveTaskContext
 *       _os_SaveIntTaskContext
 *       _os_EnterScheduler
 ****************************************************************************/

/******************************************************************************
 * FUNCTION:            _os_GoToTask
 * DESCRIPTION:
 *  It restores the context of the current task and
 *  restarts its execution as it has never been
 *  interrupted. The scheduler ( '_os_schedule' )
 *  should be the only user of this routine since
 *  it is the only one with task dispatching knowledge.
 *
 *  PARAMETERS:
 *      - Pointer to current task
 *
 *  RETURN: It does never return, the control is
 *      passed to the given task  who starts executing
 *      from the address indicated in its context.
 * 
 *  PRE: 
 *  - OS Interrupts are disabled when entering this
 *  routine (_os_nesting_sros =1)
 *  - '_os_current_task' points at the running task.
 *  - Running task has the highest among all ready
 *  tasks in the ready-to-run queue
 *
 *  POST:
 *  - Control is passed to the running task.
 *  - Interrupts are globally enabled.
 *  - _os_nesting_sros = 0 
 ****************************************************************************/
_os_INLINE  void    _os_GoToTask(_os_RTOSMEM _os_TASK* tcb)
{
        _os_TargetGoToTask(tcb);
}

/******************************************************************************
 * FUNCTION:            _os_SaveTaskContext 
 * DESCRIPTION:
 *  This function saves the current working environment
 *  when running system service's code:
 *  '_os_SaveTaskContext' must save the CPU environment
 *  after 'inst1' and must ensure that this environment
 *  prevails after the task resumes execution at 'inst2'
 *  
 *  System service:
 *      inst1
 *      // this context must be saved
 *      _os_Switch()
 *      // execution resumes here
 *      inst2
 *      ...
 *  PRE:
 *  - ISR2s interrupts are disabled when entering this routine.
 *  - '_os_current_task' points at the running task.
 *
 *  POST: The task control block and backup area save
 *      the environment of the task after 'inst1'
 ****************************************************************************/
_os_INLINE void  _os_SaveTaskContext(void)
{
        _os_TargetSaveTaskContext();
}

/******************************************************************************
 * FUNCTION:            _os_SaveIntTaskContext
 * DESCRIPTION:
 *  This function saves the preempted CPU environment
 *  from first nested interrupt level.
 *  The routine executes only when the preempted task
 *  was running application code (kernel is said to be
 *  non-preemptable).
 *
 *  'SaveIntTaskContext' must execute (our design)
 *  always at the same function call nested level
 *  in all our interrupts (i.e from an ISR2 or       
 *  the system timer interrupt).
 *
 *  The routine ensures that whenever thie scheduler
 *  resumes the preempted task again. 
 *  the existing environment of the task before the
 *  interrupt will be fully recovered.
 *
 *  PRE: 
 *  - ISR2s interrupts are disabled when entering this routine.
 *  - '_os_current_task' points at the running task.
 ****************************************************************************/
_os_INLINE void    _os_SaveIntTaskContext(void)
{
        _os_TargetSaveIntTaskContext();
}


/******************************************************************************
 * FUNCTION:            _os_EnterScheduler 
 * DESCRIPTION:
 *  Platform specific code upon entering the scheduler
 ****************************************************************************/
_os_INLINE void    _os_EnterScheduler(void)
{
        _os_TargetEnterScheduler();
}


/******************************************************************************
 * FUNCTION:            _os_ChangeStack
 * DESCRIPTION:
 *      Called before the running task terminates (in
 *      'TerminateTask' or 'ChainTask').
 *      In this routine some platform specific
 *      code runs once we know that scheduling
 *      shall happen nextly.     
 *      In particular, users might want to use
 *      another stack (other than the task's) to keep
 *      storing the next locals/parameters/..etc while
 *      rearranging the stack of the task for future
 *      activations.
 * REMARKS:
 *      It could be left empty.                         
 ****************************************************************************/
_os_INLINE void    _os_ChangeStack(void)
{
        _os_TargetChangeStack();
}

/******************************************************************************
 * Stack Monitoring:
 *     _os_TargetInitStackMonitor    
 *     _os_TargetStackMonitor
 *     _os_TargetIsStackInRange
 ****************************************************************************/
#if (_os_OS_0_STACKMONITOR >0)

/******************************************************************************
 * FUNCTION:    _os_InitStackMonitor
 * DESCRIPTION:
 *      This function is called upon start-up of the
 *      system to prepare what is needed to monitor
 *      possible stack overflows at run-time.
 *      It basically initializes the elements of the 
 *      '_os_stack_monitor' array (see 'c_common.h').
 * REMARK:
 *      Only if non-standard attribute STACKMONITOR is set to TRUE.
 ****************************************************************************/
void _os_TargetInitStackMonitor(void);

/******************************************************************************
 * FUNCTION:    _os_TargetStackMonitor
 * DESCRIPTION:
 *      This function is called after every task's run 
 *      to detect possible reasons of stack overflow that
 *      are target specific.
 *      returns 0 if reasons are found.
 *              1 otherwise.
 * REMARK:
 *      Only if non-standard attribute STACKMONITOR is set to TRUE.
 ****************************************************************************/
uint_fast8_t _os_TargetStackMonitor(void);

/******************************************************************************
 * FUNCTION:    _os_TargetIsStackInRange
 * DESCRIPTION:
 *      This function is called upon entering RTOS code to
 *      test target specific reasons why an stack could be
 *      currently overflowning.
 *      returns 0 if reasons are found.
 *              1 otherwise.
 * REMARK:
 *      Only if non-standard attribute STACKMONITOR is set to TRUE.
 ****************************************************************************/
uint_fast8_t _os_TargetIsStackInRange(void);

/******************************************************************************
 * FUNCTION:    _os_GetStackPointer
 * DESCRIPTION:
 *      This function returns the run-time value of the stack pointer
 *      associated with a given stack.
 ****************************************************************************/
_os_STACKMEM uint8_t * _os_GetStackPointer(TaskType);

#endif /* (_os_OS_0_STACKMONITOR >0) */

/******************************************************************************
 * Miscellaneous:
 *      _os_ReturnToStartOS
 *      _os_Clock
 *      _os_SaveStartOS
 ****************************************************************************/

/******************************************************************************
 * FUNCTION:            _os_ReturnToStartOS
 * DESCRIPTION:
 *      Called at the end of ShutdownHook.
 *      - Restore the environment of the main thread before 'StartOS'
 * PRE:
 *      - OS interrupts are disabled.
 *      - No running task (_os_current_task=0).
 * POST:
 *      Execution resumes at user's code after 'StartOS'
 ****************************************************************************/
_os_INLINE void _os_ReturnToStartOS(void)
{
        _os_TargetReturnToStartOS();
}

/******************************************************************************
 * FUNCTION:            _os_SaveStartOS
 * DESCRIPTION:
 *      Called at begin of StartOS. Saves the environment of the main thread
 *      so that it can be resumed in ShutdownOS.
 ****************************************************************************/
_os_INLINE void _os_SaveStartOS(void)
{
        _os_TargetSaveStartOS();
}

#if (_os_OS_0_ORTI>0)
/******************************************************************************
 * FUNCTION:            _os_Clock
 * DESCRIPTION:
 *      Returns the elapsed time since reset took place. Every target has 
 *      different units. In order to convert it into seconds we should 
 *      divide the return value by _os_CLOCKS_PER_SEC (also to be defined
 *      in 't_target.h').
 ****************************************************************************/
_os_INLINE clock_t _os_Clock( void )
{
        return _os_TargetClock();       
}
#endif

/******************************************************************************
 * FUNCTION:            _os_Restore
 * DESCRIPTION:         
 *      Saves the context of the preempted task, set the task's state
 *      as READY and calls the scheduler. It is used at task level 
 *      (_os_TaskSwitch) and at ISR2 level (see body of ISR2s interfaces). 
 *      ISR2s must have been previously suspended.      
 ****************************************************************************/
_os_INLINE void _os_Restore(void)
{
        #if (_os_OS_0_POSTTASKHOOK==1)
        _os_SETPOSTHOOK();
        PostTaskHook();
        _os_CLRPOSTHOOK();
        #endif
        
        #if (_os_OS_0_STACKMONITOR ==1)
        _os_StackMonitor();
        #endif
        
        _os_current_task->state = READY;
        
        #if (_os_OS_0_ORTI>0)   
        _os_UpdateCycles();
        #endif

        _os_current_task = (_os_RTOSMEM _os_TASK*)0;
        
        _os_ResumeOS();

        _os_Schedule();
}

/******************************************************************************
 * FUNCTIONS:   _os_BodyISR1(isr,label)
 *              _os_BodyISR2(isr,label)
 *              _os_BodySystemClock()
 *              
 * Common 'default' body of RTOS ISR1/2 code. These inline functions are used in
 * the toc generated 'g_isrframe.c' file and rely on the following macros 
 * that must be defined in 't_rtos.h'.
 *
 * _os_ENTERISR1/2():  Target actions upon entering the ISR1/2 RTOS
 *                     handler. Interrupts must be disabled.
 * _os_EXITISR1/2():   Target actions upon exiting the ISR1/2 RTOS
 *                     handler. Interrupts must be enabled before
 *                     returning from the ISR handler.
 * _os_SAVEINTCONTEXT(): Target actions to save the context of the 
 *                     interrupted task. It happens only if task
 *                     was not running RTOS code.
 * _os_RELOADRTOSTIMER():  Macro to reload rtos timer (optional)
 *
 * REMARKS:     These are default definitions. Should a target
 *              have specific needs (not falling in the default
 *              case), you would define them in 't_target.h'
 *              (together with the 'Def' macros).
 ****************************************************************************/
#if defined(_os_DefBodyISR1) || defined(_os_DefBodyISR2) || defined(_os_DefBodySystemClock)
#include "t_bodies.h"
#endif

#if (_os_NO_ISR>0)

#if !defined(_os_DefBodyISR1)
#define _os_DefBodyISR1
_os_INLINE void _os_BodyISR1(IsrType isr,void (*label)(void))
{
        _os_ENTERISR1()         

        #if (_os_OS_0_ORTI>0)   
        _os_UpdateCycles();
        #endif
        if (_os_isr1_counter++==0)      
        {               
                _os_SETISR1LEVEL()
        }
        _os_lock++;
        _os_isr_table[isr].prev =  _os_current_isr;
        _os_current_isr         = &_os_isr_table[isr];
        #if (_os_OS_0_ORTI>0)   
        _os_current_isr->noruns++;
        #endif
        _os_EnableInterrupts();

        label ();       

        _os_DisableInterrupts();
        _os_lock--;             
        if (--_os_isr1_counter==0)
        {                       
                _os_CLRISR1LEVEL()
        }
        _os_current_isr         = _os_current_isr->prev;
        _os_isr_table[isr].prev = (_os_RTOSMEM _os_ISR*)0;
        #if (_os_OS_0_ORTI>0)   
        _os_UpdateISRCycles(isr);
        #endif

        _os_EXITISR1()
}
#endif

#if !defined(_os_DefBodyISR2)
#define _os_DefBodyISR2
_os_INLINE void _os_BodyISR2(IsrType isr,void (*label)(void))
{
        _os_ENTERISR2()
        if ( ++_os_lock == _os_SCHEDULABLE ) 
        {                       
                _os_SAVEINTCONTEXT() 
        }                               
        #if (_os_OS_0_ORTI>0)
        _os_UpdateCycles();
        #endif                          
        if (_os_isr2_counter++==0)
        {
                _os_SETISR2LEVEL()
        }

        _os_isr_table[isr].prev =  _os_current_isr;
        _os_current_isr         = &_os_isr_table[isr];
        #if (_os_OS_0_ORTI>0)   
        _os_current_isr->noruns++;
        #endif

        _os_EnableInterrupts();

        label ();

        _os_SuspendOS();                                                
        
        _os_current_isr         = _os_current_isr->prev;
        _os_isr_table[isr].prev = (_os_RTOSMEM _os_ISR*)0;
        
        if (    _os_lock == _os_SCHEDULABLE &&                          
                _os_ready_to_run_index > _os_current_task->PRIORITY )   
        {                                                               
                _os_SaveIntTaskContext();                               
                _os_Restore();                                          
        }                                                               

        _os_DisableInterrupts();                                        
        _os_ResumeOS();
        _os_lock--;
        if (--_os_isr2_counter==0)
        {                                                               
                _os_CLRISR2LEVEL()                                      
        }
        #if (_os_OS_0_ORTI>0)   
        _os_UpdateISRCycles(isr);
        #endif

        _os_EXITISR2()
}
#endif

#endif /* _os_NO_ISR>0 */

#if (_os_OS_0_USERTOSTIMER>0)
#if !defined(_os_DefBodySystemClock)
#define _os_DefBodySystemClock
_os_INLINE void _os_BodySystemClock(void)
{
        _os_ENTERISR2()

        if ( ++_os_lock == _os_SCHEDULABLE ) 
        {                       
                _os_SAVEINTCONTEXT() 
        }                               
        #if (_os_OS_0_ORTI>0)
        _os_UpdateCycles();
        #endif                          
        #if (_os_NO_ISR>0)
        _os_isr2_counter++;
        #endif
        _os_RELOADRTOSTIMER()
        _os_EnableInterrupts();                                         

        _os_UpdateCounter(SYSTEM_TIMER);                                                        

        _os_SuspendOS();                                                
        if (    _os_lock == _os_SCHEDULABLE &&                          
                _os_ready_to_run_index > _os_current_task->PRIORITY )   
        {                                                               
                _os_SaveIntTaskContext();                               
                _os_Restore();                                          
        }                                                               

        _os_DisableInterrupts();                                        
        _os_ResumeOS();                                                 
        #if (_os_NO_ISR>0)
        _os_isr2_counter--;
        #endif
        _os_lock--;                                                     
        #if (_os_OS_0_ORTI>0)   
        _os_UpdateRTOSCycles();
        #endif

        _os_EXITISR2() 
}
#endif
#endif


#endif /* _H_OS_C_TARGET */


