/******************************************************************************
 *  FILE:      @(#)c_system.c   1.178 04/10/18
 *  DESCRIPTION:
 *      Common code/data definitions that help implementing the
 *      'system services' modules.
 *      The definitions of system services come in different files,
 *      which result in different objects, in order to optimize code
 *      size at link time.
 *  ATTENTION:
 *      The maskable interrupts are always, again ALWAYS, globally
 *      enabled when entering an OSEK system service, whether at task
 *      or at ISR level. Upon returning from a system service, the
 *      maskable interrupts must be equally enabled.
 *****************************************************************************/

/******************************************************************************
 * RACE CONDITIONS:
 *      A race condition occurs when two or more independent
 *      tasks/isr2 access the same shared data (or resource) simultaneously.
 *      Although the effect varies, they are always a bug in
 *      multitasking environment. Typically:
 *       - R1: Each non-atomic read must be protected.
 *       - R1: Each read-modify-write sequence must be protected.
 *       - R3: Each non-atomic write to the data must be protected.
 *       - R4: If checking hardware status flag to determine availability
 *       of a resource before using it, protection must be applied from the
 *       point the flag is checked until the hardware sets the flag to indicate
 *       the resource is unavailale.                            
 * 
 *      THE RTOS IS NON-PREEMPTIVE (most of it)
 *      ---------------------------------------
 *      We say that our RTOS is non-preemptive because a task running RTOS code
 *      (i.e. in a system service) cannot be preempted by another task. It can
 *      be interrupted by an ISR but scheduling will have to necessarily wait
 *      until the end of the service. Obviously all tasks can be preempted when
 *      running application code.
 *
 *      The general mechanism is presented here:
 *
 *      ----application code------application code-----
 *      
 *        ..
 *        .. (scheduler not locked for ISR2s)
 *        .. (task can be preempted by ISR2 here 
 *        ..  and go to ready state)
 *        ..
 *          ----system service -----system service ----
 *
 *              _os_lock++; (scheduler is now locked for possible ISR2s
 *                           interrupting rtos code)
 *              ..
 *              ..
 *              (actions of the service)
 *              ..
 *              ..
 *              -------test-schedule-------------------
 *              if(_os_NeedToSwitchTasks())
 *              {  
 *                  (reasons to run the scheduler)
 *                  _os_current_task->state = READY;
 *                  _os_Switch();
 *              }
 *              (no reasons to scheduler)  
 *              -------test schedule-------------------
 *              (return to user)
 *          ----rtos--------------rtos-----
 *        ..
 *        .. (scheduler not locked for ISR2s)
 *        .. (task can be preempted by ISR2 here 
 *        ..  and go to ready state)
 *
 *
 *      ----Application code------
 * 
 *
 *      WAITEVENT
 *      ---------       
 *      A task can decide himself to run the scheduler in order to
 *      enter the WAITING state if system service is 'WaitEvent'.
 *      The current task saves its context, it is removed from the ready-to-run 
 *      queue and only then the scheduler is called.
 *      The scheduler remains locked while running this service.
 * 
 *      SERVICES FOR HOOK ROUTINES
 *      --------------------------
 *      The following system services can be called from a hook-routine:
 *              GetTaskID, 
 *              GetTaskState,
 *              SuspendAllInterrupts,
 *              ResumeAllInterrupts,
 *              GetEvent,
 *              GetAlarmBase,
 *              GetActiveApplicationMode,
 *              GetCOMApplicationMode,
 *              GetMessageStatus
 *      
 *      In these services, the scheduler never runs at the end of the service.
 *      
 *      This is okay from hook routines since the scheduler cannot run there.
 *      (besides, testing the scheduler from hook routines would nest calls
 *      to _os_Suspend/ResumeOS which is forbidden by the implementation).
 *
 *      From application code it is also okay since all these fast routines
 *      never change the internal status of the ready queues. BUT then, the
 *      scheduler CANNOT be locked during these services (should the scheduler
 *      be locked when an ISR2 comes, we could easily miss a context switch).  
 *
 *      Therefore, a task running these services from application level can 
 *      be preempted by another task. Thus, these services cannot share 
 *      global variables with other tasks without protection.
 *
 *      OTHERS
 *      ------
 *      Some other services which are also very fast and have no direct impact
 *      in the internal status of the scheduler will run like 'services from
 *      hook routines' to enhance system performance. These are:
 *               ClearEvent,    
 *               StopCOM
 * 
 *      ENSURING CONSISTENCY
 *      --------------------
 *      That the Kernel is non-preemptive means we dont suffer potential risks
 *      of data corruption due to concurrency problems? No, inconsistencies
 *      can happen while handling ISR2 (they have access to rtos internals
 *      through some system services).
 *      So, we say that a region is CRITICAL CODE if the execution of 
 *      arbitrary interrupt handlers (while running the RTOS code)
 *      could result in consistency problems.
 *      These critical regions should be small and infrequent.
 *
 *      In simple designs, protection is usually implemented by disabling
 *      interrupts around the critical sections.
 *      In our OSEK system:
 *      - The pair '_os_(Disable/Enable)Interrupts' can be
 *      used to protect critical code.
 *      - The pair '_os_Suspend/ResumeOS()' can also
 *      be used to protect critical code while still allowing
 *      Category 1 interrupts in the system.
 *
 *      Our implementation follows the criteria of permanently allowing ISR1s
 *      (with exception only in specific and very fast cases).
 *
 *      1. In our implementation, the following variables must ensure atomic
 *      read/write operations:
 *              _os_isr1_counter;
 *              _os_isr2_counter;
 *              _os_lock;
 *              _os_ready_to_run_index;
 *              _os_nesting_sr;
 *              _os_nesting_sros;
 *              _os_run_level;
 *      Thus, reads and writes here require no protection.
 *
 *      2. Besides, operations like '_os_lock++' also require no protection.
 *      If at any moment during a read-modify-write cycle (in task or ISR)
 *      we are preempted (by an ISR2), when the cycle resumes execution
 *      (even after possible context switches), the previous value prevails
 *      (it has been well counterbalanced).
 *
 *      3. Although '_os_current_task' has a non-atomic read operation,
 *      no concurrency problems are expected while reading the variable since
 *      the scheduler is locked (only scheduler modifies it).
 *
 *      4. '_os_current_isr' is updated with all interrupts disabled.
 *
 *      4. An ISR2 could preempt rtos code when no task is running in the
 *      system. We must ensure a valid value for '_os_current_task'
 *      from interrupt level. This means that no accesses are allowed
 *      (with _os_isr2_counter>0) in:
 *                - ActivateTask
 *                - GetTaskID
 *                - GetTaskState
 *                - DisableAllInterrupts ....
 *                - GetResource (when _os_isr2_counter>0)
 *                - ReleaseResource (when _os_isr2_counter>0)
 *                - SetEvent
 *                - GetEvent
 *                - GetAlarmBase
 *                - GetAlarm
 *                - SetRelAlarm
 *                - SetAbsAlarm
 *                - CancelAlarm
 *                - GetActiveApplicationMode
 *                - ShutdownOS
 *                - GetCOMApplicationMode
 *                - InitMessage
 *                - GetMessageStatus
 *                - SendMessage
 *                - ReceiveMessage
 *                since these system services could be called from an ISR2
 *
 *      5. Please do understand that having '_os_current_task' no need of
 *      protection does not imply that the task block's fields are also free
 *      of concurrency problems.
 *****************************************************************************/
#include <time.h>
#include "c_common.h"
#include "c_target.h"
#include "g_conf.h"

#if (_os_OS_0_ERRORHOOK==1 || _os_COM_0_COMERRORHOOK==1)
#if ((_os_OS_0_USEGETSERVICEID == 1) || (_os_COM_0_COMERRORGETSERVICEID==1)||(_os_OS_0_ORTI>0) )
/* last failing system service */
_os_RTOSMEM uint_fast8_t                _os_errid;
#endif
#if (_os_OS_0_ORTI>0)
_os_RTOSMEM StatusType                  _os_errno = E_OK;
#endif

#if (_os_OS_0_USEPARAMETERACCESS == 1 || _os_COM_0_COMUSEPARAMETERACCESS==1)
/* Array of last system service's parameters */
int32_t                                 _os_sys_par[3];
#endif

#endif

#if (_os_OS_0_ORTI>0)
volatile uint_fast8_t _os_FASTALLOC _os_servicetrace=0;
static void _os_ServiceTraceEntry(uint_fast8_t ss);
static void _os_ServiceTraceExit(uint_fast8_t ss);
#endif

#if (_os_OS_0_ORTI>0)
_os_RTOSMEM clock_t                     _os_rtos_cycles=0;
_os_RTOSMEM clock_t                     _os_enter_cycles=0;
_os_RTOSMEM clock_t                     _os_total_cycles=0;
#endif

/******************************************************************************
 * System services are called from tasks, ISRs, hook routines and alarm
 * callbacks. Depending on the system service, there may be restrictions 
 * regarding the availability.
 * The following list divides all system services in six groups
 * (where are they allowed to be called?).
 *
 *                      ISR1   ISR2   Error   Pre Post  Startup  Shut CallBack  
 * 
 * GROUP1:
 * GetActive
 *  ApplicationMode             x       x      x   x     x       x       
 *  GetCOM
 *  ApplicationMode             x       x      x   x     x       x       
 *
 * GROUP2:
 * SuspendAllInterrupts   x     x       x      x   x                     x
 * ResumeAllInterrupts    x     x       x      x   x                     x
 *
 * GROUP3:
 * GetTaskID                    x       x      x   x    
 * GetTaskState                 x       x      x   x            
 * GetEvent                     x       x      x   x            
 * GetAlarmBase                 x       x      x   x            
 * GetAlarm                     x       x      x   x            
 * GetMessageStatus             x       x      x   x
 * 
 * GROUP4:
 * ShutdownOS                   x       x                x 
 * 
 * GROUP5:
 * DisableAllInterrupts   x     x
 * EnableAllInterrupts    x     x
 * SuspendOSInterrupts    x     x
 * ResumeOSInterrupts     x     x
 * 
 * GROUP6:
 * ActivateTask                 x
 * GetResource                  x
 * ReleaseResource              x
 * SetEvent                     x
 * SetRelAlarm                  x
 * SetAbsAlarm                  x
 * CancelAlarm                  x
 * InitMessage                  x
 * SendMessage                  x
 * ReceiveMessage               x
 * IncrementCounter             x
 * IsStackInRange               x
 *
 * TerminateTask                x
 * ChainTask                    x
 * Schedule                     x
 * WaitEvent                    x
 * ClearEvent                   x
 * 
 * GROUP7:
 * StartCOM
 * StopCOM
 *
 *****************************************************************************/
#define _os_G1_LV       (_os_ISR2LV|_os_HOOKERRLV|_os_HOOKPRELV|_os_HOOKPOSTLV|_os_HOOKSTARTLV|_os_HOOKSHUTLV)
#define _os_G2_LV       (_os_ISR1LV|_os_ISR2LV|_os_HOOKERRLV|_os_HOOKPRELV|_os_HOOKPOSTLV|_os_CALLBACKLV)
#define _os_G3_LV       (_os_ISR2LV|_os_HOOKERRLV|_os_HOOKPRELV|_os_HOOKPOSTLV )
#define _os_G4_LV       (_os_ISR2LV|_os_HOOKERRLV|_os_HOOKSTARTLV)
#define _os_G5_LV       (_os_ISR1LV|_os_ISR2LV)
#define _os_G6_LV       (_os_ISR2LV)
#define _os_G7_LV       (_os_TASKLV)

const uint_fast8_t _os_run_levels[_os_N_SYSSER+1] =
{
        0        ,_os_G6_LV,_os_G6_LV,0        ,_os_G1_LV,_os_G4_LV,_os_G7_LV,_os_G7_LV,
        _os_G1_LV,_os_G3_LV,_os_G6_LV,_os_G6_LV,_os_G3_LV,_os_G6_LV,_os_G6_LV,_os_G3_LV,
        _os_G3_LV,_os_G6_LV,_os_G6_LV,_os_G6_LV,_os_G6_LV,_os_G3_LV,_os_G6_LV,_os_G6_LV,
        _os_G6_LV,_os_G6_LV,_os_G6_LV,_os_G6_LV,_os_G3_LV,_os_G5_LV,_os_G5_LV,_os_G2_LV,
        _os_G2_LV,_os_G5_LV,_os_G5_LV,_os_G6_LV
};

/******************************************************************************
 * FUNCTION:            _os_LogError
 * DESCRIPTION:
 *      Logs system errors and dispatches execution to
 *      the Hook routines.
 *****************************************************************************/
#if (_os_OS_0_ERRORHOOK==1 || _os_COM_0_COMERRORHOOK==1)
void _os_LogError(StatusType status,  uint_fast8_t errorId, int32_t par1,
                 int32_t par2, int32_t par3)
{

        /* avoid nesting calls to ErrorHook */
        if (_os_INERRORHOOK())
        {
                return;
        }

        /* Hook Routines run with disabled ISR2s */
        _os_SuspendOS();
       
        /* enter ErrorHook */
        _os_SETERRORHOOK();
        
        /* Error login information: This info may be accessed from the
         * next hook routine. Data cannot be corrupted before their use
         * since no other system call will occur ( no 're-scheduling' 
         * and no category 2 interrupts).
         */

#if ((_os_OS_0_USEGETSERVICEID == 1) || (_os_COM_0_COMERRORGETSERVICEID==1)||(_os_OS_0_ORTI>0) )
        _os_errid                       = errorId;
#endif
#if (_os_OS_0_ORTI>0)
        _os_errno                       = status;
        _os_current_task->errid         = errorId;
#endif

#if (_os_OS_0_USEPARAMETERACCESS==1|| _os_COM_0_COMUSEPARAMETERACCESS==1)
        _os_sys_par[0]      = par1;
        _os_sys_par[1]      = par2;
        _os_sys_par[2]      = par3;
#endif        

        switch(errorId)
        {
                case _os_TO_SEM_TAKE: 
                case _os_TO_SEM_GIVE:
                case _os_TO_START_OS:
                case _os_TO_APP_MODE:
                case _os_TO_SHUTDOWN:
                case _os_TO_TASK_GETID:
                case _os_TO_ACTIVATE_TASK:
                case _os_TO_TERMINATE_TASK:
                case _os_TO_TASK_STATE:
                case _os_TO_SCHEDULE:
                case _os_TO_CHAIN_TASK:
                case _os_TO_BASE_ALARM:
                case _os_TO_GET_ALARM:
                case _os_TO_SETREL_ALARM: 
                case _os_TO_SETABS_ALARM:
                case _os_TO_CANCEL_ALARM:
                case _os_TO_SET_EVENT:
                case _os_TO_GET_EVENT:
                case _os_TO_WAIT_EVENT:
                case _os_TO_CLEAR_EVENT:
                case _os_TO_INC_COUNTER:
                #if (_os_OS_0_ERRORHOOK==1) 
                        ErrorHook(status);
                #endif
                break;
                
                case _os_TO_START_COM:
                case _os_TO_STOP_COM:
                case _os_TO_COMAPP_MODE:
                case _os_TO_INIT_MESSAGE:
                case _os_TO_SEND_MSG:
                case _os_TO_REC_MSG:
                case _os_TO_GET_MSGSTATUS:
                #if (_os_COM_0_COMERRORHOOK==1)
                        COMErrorHook(status);
                #endif
                break;

                default:break;
        }
        
        /* exit Error Hook */
        _os_CLRERRORHOOK();
        
        /* resume ISR2s  */
        _os_ResumeOS();
        return;
}
#endif

/******************************************************************************
 * FUNCTION:            _os_NeedToSwitchTasks
 * DESCRIPTION:
 *      Evaluates whether scheduling shall occur next upon termination of       
 *      system services (due to its own code or an ISR2 preemption).  
 *****************************************************************************/         
uint_fast8_t _os_NeedToSwitchTasks(void)
{
        _os_SuspendOS();
        
        if (    /* can we schedule */
                _os_lock == _os_SCHEDULABLE  &&
                /* a new higher ready task exists */
                _os_ready_to_run_index > _os_current_task->PRIORITY
           )
        {
                /* OS interrupts remain suspended */
                return 1;
        }
        else
        {
                /* last action in a system service */ 
                _os_lock--;
                _os_ResumeOS();
                return 0;
        }
}


/******************************************************************************
 *  FUNCTION:       _os_TaskSwitch
 *  DESCRIPTION:
 *      Reasons to preempt the running task have been found.
 *      Saves the context of the preempted task, set the task's state
 *      as READY and calls the scheduler. It executes at task level 
 *      and at ISR2 level (this is the reason for a common interface
 *      '_os_Restore'). ISR2s must have been previously suspended.
 *
 *  REMARKS:
 *      NEVER (again NEVER) do change the interface neither the body
 *      of this call.
 *      Saving context is not trivial and it assumes no parameters
 *      in the call and no automatics in the body.
 *****************************************************************************/         
void _os_TaskSwitch(void)
{
        _os_SaveTaskContext();
    
        _os_Restore();

        /* never here */
}

/******************************************************************************
 *  FUNCTION:       _os_WaitSwitch
 *  DESCRIPTION:
 *      Saves the context of the running task, set the state of the task
 *      as WAITING and calls the scheduler.
 *      It must executes at task level and ISR2s must have been 
 *      previously suspended.
 *
 *  REMARKS:
 *      NEVER (again NEVER) do change the interface neither the body
 *      of this call.
 *      Saving context is not trivial and it assumes no parameters
 *      in the call and no automatics in the body.
 *****************************************************************************/         
void _os_WaitSwitch(void)
{
        _os_SaveTaskContext();
       
        #if (_os_OS_0_POSTTASKHOOK==1)
        _os_SETPOSTHOOK();
        PostTaskHook();
        _os_CLRPOSTHOOK();
        #endif
        
        #if (_os_OS_0_STACKMONITOR ==1)
        _os_StackMonitor();
        #endif
        
        _os_current_task->state = WAITING;
        
        #if (_os_OS_0_ORTI>0)
        /* no cycles spent in last run */
        _os_UpdateTaskCycles();
        #endif
        
        _os_current_task = (_os_RTOSMEM _os_TASK*)0;

        _os_ResumeOS();

        _os_Schedule();

        /* never here */
}

/******************************************************************************
 *  FUNCTION:       _os_ServiceEntry
 *  DESCRIPTION:
 *      System actions upon entrying all system services.
 *      - ORTI trace of services.
 *      - Check stacks are in range.
 *      - Check running level is correct.
 *  REMARKS:
 *****************************************************************************/         
void _os_ServiceEntry(uint_fast8_t ss)
{       
#if (_os_OS_0_ORTI>0)
        _os_ServiceTraceEntry(ss);
#endif
#if (_os_OS_0_STACKMONITOR >0)
        _os_CheckStackRange();
#endif
        _os_CheckRunLevel(ss);
        return;
}

/******************************************************************************
 *  FUNCTION:       _os_CheckRunLevel
 *  DESCRIPTION:
 *      System actions upon exiting all system services.
 *      - ORTI trace of services.
 *  REMARKS:
 *****************************************************************************/         
void _os_CheckRunLevel(uint_fast8_t ss)
{
        if ( _os_run_level & (~_os_run_levels[ss])  )
        {                                       
                ShutdownOS(E_OS_SYS_RUNLEVEL);          
        }
        return;
}

/******************************************************************************
 *  FUNCTION:       _os_ServiceExit
 *  DESCRIPTION:
 *      System actions upon exiting all system services.
 *      - ORTI trace of services.
 *  REMARKS:
 *****************************************************************************/         
void _os_ServiceExit(uint_fast8_t ss)
{
#if (_os_OS_0_ORTI>0)
        _os_ServiceTraceExit(ss);
#endif

}

#if (_os_OS_0_ORTI>0)
static void _os_ServiceTraceEntry(uint_fast8_t ss)
{
        #if (_os_NO_ISR>0)
        if (_os_current_isr != (_os_RTOSMEM _os_ISR*)0)
        {
                _os_current_isr->servicetrace = 2*ss+1;
        }
        else
        {
        #endif
                _os_current_task->servicetrace = 2*ss+1;
        #if (_os_NO_ISR>0)
        }
        #endif
        _os_servicetrace = 2*ss+1;
        return;
}

static void _os_ServiceTraceExit(uint_fast8_t ss)
{
        #if (_os_NO_ISR>0)
        if (_os_current_isr != (_os_RTOSMEM _os_ISR*)0)
        {
                _os_current_isr->servicetrace = 2*ss;
        }
        else
        {
        #endif
                _os_current_task->servicetrace = 2*ss;
        #if (_os_NO_ISR>0)
        }
        #endif
        _os_servicetrace = 2*ss;
        return;
}
#endif

#if (_os_OS_0_ORTI>0)

/******************************************************************************
 *  FUNCTION:       _os_UpdateTaskCycles
 *  DESCRIPTION:
 *      It updates the number of cycles for the 
 *      terminating/waiting task.
 *      
 *  REMARKS:
 *      OS interrupts are disabled
 *      (_os_current_task!=0 && (_os_isr2_counter+_os_isr1_counter)==0
 *      
 *****************************************************************************/         
void _os_UpdateTaskCycles(void)
{
        clock_t delta = _os_Clock() - _os_enter_cycles;
        _os_current_task->cycles += delta;
        _os_total_cycles         += delta;
        _os_enter_cycles         = _os_Clock();
}       


/******************************************************************************
 *  FUNCTION:       _os_UpdateRTOSCycles
 *  DESCRIPTION:
 *      It updates the number of cycles spent in the scheduler.
 *      
 *  REMARKS:
 *      OS interrupts are disabled
 *      "end of scheduler code" or grey area "_os_current_task==0"
 *      
 *****************************************************************************/         
void _os_UpdateRTOSCycles(void)
{
        clock_t delta          = _os_Clock() - _os_enter_cycles;
        _os_rtos_cycles        += delta;
        _os_total_cycles       += delta;
        _os_enter_cycles       = _os_Clock();
}

#if (_os_NO_ISR>0)
/******************************************************************************
 *  FUNCTION:       _os_UpdateISRCycles
 *  DESCRIPTION:
 *      It updates the number of cycles spent in the falling isr.
 *  REMARKS:
 *      OS interrupts are disabled
 *****************************************************************************/         
void _os_UpdateISRCycles(IsrType isr)
{
        clock_t delta   = _os_Clock() - _os_enter_cycles;
        _os_isr_table[isr].cycles   += delta;
        _os_total_cycles          += delta;
        _os_enter_cycles          = _os_Clock();
}
#endif

/******************************************************************************
 *  FUNCTION:       _os_UpdateCycles
 *  DESCRIPTION:
 *      It updates the number of cycles spent in the last run.
 *  REMARKS:
 *      OS interrupts are disabled
 *****************************************************************************/         
void _os_UpdateCycles(void)
{
        clock_t delta   = _os_Clock() - _os_enter_cycles;

#if (_os_NO_ISR>0)
        if      (_os_current_isr)
        {
                /* an is was preempted */
                _os_current_isr->cycles   += delta;
        }
        else
#endif
        if (_os_current_task)
        {
                /* a task has been preempted */
                _os_current_task->cycles   += delta;
        }
        else
        {
                /* the scheduler was preempted */
                _os_rtos_cycles           += delta;
        }

        _os_total_cycles     += delta;
        _os_enter_cycles     = _os_Clock();
}


#endif


