/************************************************************************
**  FILE:      @(#)c_startos.c  1.25 04/11/10
**  DESCRIPTION:                                                        *
**      Source code for 'StartOS'                                       *
************************************************************************/
#include "c_common.h"
#include "c_target.h"
#include "g_conf.h"

/* Storage for application modes */
_os_RTOSMEM AppModeType         _os_system_mode;
#if (_os_OS_0_MULTISTART>0)
/* it counts the number of application resets */
_os_RTOSMEM uint_fast8_t                _os_no_reset = 0; 
#endif

static void _os_InitRTOS(void);
static void _os_InitTask(void);
#if (_os_NO_ALARM >0)
static void _os_InitAlarm(void);
#endif
#if (_os_NO_USER_TASK>0)
static void _os_RunTimeTaskContext(TaskType);
#endif

void
#if (_os_SIA>0)
_os_StartOS
#else
StartOS
#endif
(AppModeType mode)
{
        _os_system_mode = mode;
        
        /********************************** 
         * Save the current preOS thread.
         * We might need to restore it in
         * ShutdownOS.
         **********************************/
        _os_SaveStartOS();

        _os_lock++;

#if (_os_OS_0_STATUS == _os_OS_EXTENDED)
       
        /* RTOS must be initialise from task level */
        if ( _os_run_level )
        {
            ShutdownOS(E_OS_SYS_ERROR);            
        }
        if (_os_lock!=_os_SCHEDULABLE)
        {
            ShutdownOS(E_OS_SYS_ERROR);            
        }
#endif

        /* init the OS kernel: Upon returning from this function all interrupts
         * will be enabled.*/
        _os_InitRTOS();

        /*********************************************
         * dummy to always extract the isr framework 
         * in the 'g_isrframe' object from library
         ********************************************/
#if (_os_NO_ISR>0)
        _os_ExtractISR();
#endif
        
        /***********************************************************
         * start scheduling, it does never return.
         * the scheduler starts the autostarting tasks and alarms
         * as declared in the oil file. Activation order of tasks
         * with equal priority is not defined. 
         **********************************************************/
         _os_Schedule();

         return; /* satisfy compiler */
}

/************************************************************************
 *  FUNCTION:           _os_InitRTOS                                    *
 *  DESCRIPTION:                                                        *
 *      This routine is called from the system service 'StartOS'        *
 *      and it is responsible of initialising the RTOS in               *
 *      both 'power-up' reset and software reset.                       *
 *      Only ISR1s could be running at this moment since they are       *
 *      allowed to be set in the user's start-up code.                  *
 ***********************************************************************/
static void _os_InitRTOS(void)
{
#if ( (_os_NO_ISR>0) || (_os_OS_0_USERTOSTIMER>0) )
        /* ISR initialization */
        _os_InitISR();
#endif
        
        /* tasks initialisation */      
        _os_InitTask();

#if ( _os_NO_ALARM >0 )   
        /* alarms initialisation */
        _os_InitAlarm();
#endif

#if (_os_OS_0_STACKMONITOR >0)
        _os_InitStackMonitor();
#endif  

#if (_os_OS_0_STARTUPHOOK == 1)
        _os_DisableInterrupts();
        _os_SETSTARTHOOK();
        /* User initialisation code */
        StartupHook();
        _os_CLRSTARTHOOK();
#endif
        
#if ((_os_NO_ALARM>0) &&(_os_OS_0_USERTOSTIMER>0))
        /* Start RTOS clock */
        _os_StartTimer();
#endif

        /* enable maskable interrupts */
        _os_EnableInterrupts();
        return;
}




/************************************************************************
 *  FUNCTION:           _os_InitTask                                    *
 *  DESCRIPTION:                                                        *
 *      This routine interpretates the configurational data array       *
 *      '_os_task_table' to initialize the tasks of the system.          *
 *                                                                      *               
 *  - _os_no_reset = 0 : First call to StartOS(). Task contexts are not * 
 *      fully yet created (need for these extra run time                *
 *      initialization).                                                *       
 *  - _os_no_reset>1 :   Tasks are ready. They were re-set again in     *
 *      in ShutdownOS().                                                *
 ***********************************************************************/
static void _os_InitTask(void)
{
#if (_os_NO_USER_TASK >0)
        uint_fast8_t i,j=0;

        _os_RTOSMEM _os_TASK* task = &_os_task_table[1];
#endif
    
    #if (_os_OS_0_MULTISTART)
    if (_os_no_reset==0)
    {
    #endif
        /* configures the 'idle' task the first time */
        _os_InitIdle();
    #if (_os_OS_0_MULTISTART)
    }
    #endif
    
#if (_os_NO_USER_TASK >0)
        /* this code is optimized away if '_os_NO_USER_TASK==0' */
        for ( i=1;i<_os_NO_USER_TASK+1;i++,task++ )
        {
                #if (_os_OS_0_MULTISTART)
                if (_os_no_reset==0)
                {
                #endif
                    _os_RunTimeTaskContext((TaskType)i);
                #if (_os_OS_0_MULTISTART)
                }
                #endif

                if ( task->AUTOSTART )
                {
#if (MAX_APPMODE_TASK>0)
                  /* check all the application modes of the task*/
                   for(j=0;(task->APPMODE[j]!=_os_END_SEARCH)&&j<MAX_APPMODE_TASK;j++)
                   {
                           /* compare them with current mode */
                        if ( task->APPMODE[j] == _os_system_mode ||
                             task->APPMODE[j] == OSDEFAULTAPPMODE )
                        {
                                /* set task to ready: The task autostarts
                                 * in this application mode. */
                                _os_SetTaskReady(task);
                                task->repeat = 1;
                                break;
                        }
                   }
#endif
                } 
                #if (_os_OS_0_MULTISTART)
                if (_os_no_reset==0)
                {
                #endif
                        /* Only the first time. Remains forever */
                        
                        #if (_os_NO_EVENT>0)
                                /* '_os_EVENTMASKTYPE' optimizes the size
                                 * of the EVENTS field to the current
                                 * target.
                                 */
                                task->events = (_os_EVENTMASKTYPE)task->EVENTS;
                        #endif

                        #if (_os_NO_USER_RESOURCE>0)
                        /* Done the first time, remains forever */
                        /* Init 'internal' with the ID of the internal resource (if present) */
                        for ( j=0;(task->RESOURCE[j]!=_os_END_SEARCH)&&j<MAX_RESOURCE_TASK;j++)
                        {
                                if (_os_resource_table[task->RESOURCE[j]].RESOURCEPROPERTY==_os_RE_INTERNAL)
                                {
                                        task->internal = task->RESOURCE[j];
                                        break;  
                                }
                        }
                        #endif

                #if (_os_OS_0_MULTISTART)
                }
                #endif
        } /* for */

#endif
        _os_current_task =(_os_RTOSMEM _os_TASK*)0;
        
        return;
}

/************************************************************************
 *  FUNCTION:           _os_InitAlarm                                   *
 *  DESCRIPTION:                                                        *
 *      This routine interpretates the configurational data array       *
 *      '_os_alarm_table' to initialize the alarms of the system.       *
 *                                                                      *
 *      In case of a MULTISTART system, the RTOS adds two extra fields  *
 *      to _os_ALARM (statictime,staticcycle) where it tracks the       *
 *      initial values of all autostarting alarms.                      *
 ***********************************************************************/
#if (_os_NO_ALARM>0)
static void _os_InitAlarm(void)
{
        uint_fast8_t i;
        _os_RTOSMEM _os_ALARM *     alarm = &_os_alarm_table[0];
#if (MAX_APPMODE_ALARM>0)
        uint_fast8_t notfound,j;
#endif
        for (i=0 ;i<_os_NO_ALARM;i++,alarm++)
        {
        #if     (_os_OS_0_POSTMORTEM != _os_OS_NONE)
                alarm->id = (AlarmType)i;
        #endif

              if ( alarm->AUTOSTART)
                {
                    #if (_os_OS_0_MULTISTART)
                    if (_os_no_reset==0)
                    {
                        /* save these data across software resets for
                         * AUTOSTART alarms */
                        alarm->statictime   =  alarm->ALARMTIME;
                        alarm->staticcycle  =  alarm->CYCLETIME;
                    }
                    #endif

#if (MAX_APPMODE_ALARM>0)
                    notfound = 1;
                   /* check all initialisation modes of the 'i'alarm */
                   for(j=0;(alarm->APPMODE[j]!=_os_END_SEARCH)&&j<MAX_APPMODE_ALARM;j++)
                   {
                           /* compare them with current mode */ 
                        if ( alarm->APPMODE[j] == _os_system_mode ||
                             alarm->APPMODE[j] == OSDEFAULTAPPMODE )
                        {
                             notfound = 0;
                             #if (_os_OS_0_MULTISTART>0)
                                alarm->ALARMTIME = alarm->statictime;
                                alarm->CYCLETIME = alarm->staticcycle;
                             #endif
                             
                             if ( alarm->ALARMTIME <= _os_counter_table[alarm->COUNTER].MAXALLOWEDVALUE )
                             {
                                alarm->status    = _os_AL_RUNNING;

                                /* ISR2s are disabled at this moment */
                                _os_ActivateAlarm(alarm); 
                             }
                             break;
                        }
                   } /* for */
                    
                   if(notfound)
                   {
                           alarm->ALARMTIME = 0;
                           alarm->CYCLETIME = 0;
                   }
#endif  
                } /* autostart */
        } 
        return;
}
#endif


/****************************************************************
 * FUNCTION:    _os_RunTimeTaskContext                          *
 * DESCRIPTION:                                                 *
 *      This routine ends the initialization of the task        *
 *      control block and the task context with everything      *
 *      that could not be initialized by the 'init' of          *
 *      the compiler.                                           *
 *      Most of the information can be provided at compile      *
 *      time ( see _os_TASK_RTOSINIT macro in c_rtos.h).        *
 * PRE:                                                         *
 *      'init' of the compiler has been executed.               *
 * POST:                                                        *
 *      The control block and context of the given task         *
 *      is ready for the first activation.                      *
 **************************************************************/        
#if (_os_NO_USER_TASK>0)
static void
_os_RunTimeTaskContext(TaskType task_id)
{
        /* common run-time task context initialization */
        _os_task_table[task_id].staticprio  = _os_task_table[task_id].PRIORITY;
        _os_task_table[task_id].user_id     = task_id;

        /* target run-time task context initialization */
        _os_TargetRunTimeTaskContext(task_id);
        
        /* ADDRRESS is initialized in previous routine */
        #if (_os_OS_0_ORTI>0)
        _os_task_table[task_id].pc           = (uintptr_t) _os_task_table[task_id].ADDRESS;
        _os_ORTITaskContext(task_id);
        #endif

}
#endif


