/******************************************************************************
 * FILE:        @(#)c_stackmonitor.c    1.6 04/09/15
 * DESCRIPTION:
 *      Source code for the stack monitoring module.
 *****************************************************************************/
#include <osek/osek.h>
#include "c_common.h"
#include "c_target.h"

#if (_os_OS_0_STACKMONITOR >0)

/* see description in c_common.h */
_os_RTOSMEM struct _os_stack_info _os_stack_monitor[_os_NO_TASK + _os_SYSTEM_STACK_NO];

/* monitor a particular stack, returning 1 is okay and 0 otherwise */
static uint_fast8_t _os_TestStackMonitor(TaskType id);
/* test the range of a particular stack, returning 1 is okay and 0 otherwise */
static uint_fast8_t _os_TestStackRange(TaskType id);
/* test whether task's and system stacks are in range */
static StatusType _os_StackRange(void);
#endif
        
StatusType
#if (_os_SIA>0)
_os_IsStackInRange
#else
IsStackInRange
#endif
(void)
{
#if (_os_OS_0_STACKMONITOR >0)
        
        /* check good run-level */
        _os_CheckRunLevel(_os_N_IsStackInRange);        
        
        return _os_StackRange();
#else
        return E_OK;
#endif
}


#if (_os_OS_0_STACKMONITOR >0)

void _os_CheckStackRange(void)
{
        StatusType ret;

        if ( (_os_run_level & ~(_os_ISR2LV | _os_TASKLV ) ) ==0 )
        {
                if ( (ret=_os_StackRange()) != E_OK)
                {
                        ShutdownOS(ret);
                }
        }

        return;
}


void _os_StackMonitor(void)
{
        uint_fast8_t index;
        
        /* test last running task did not overwrite its patterns     */
        /* _os_StackMonitor always runs with '_os_current_task != 0' */
        if (_os_TestStackMonitor(_os_current_task->user_id)==0)
        {
                ShutdownOS(E_OS_SYS_UOFLW);
        }
        /* neither those of global stacks */
        for (index = _os_NO_TASK;index<_os_NO_TASK+_os_SYSTEM_STACK_NO;index++)
        {
                if (_os_TestStackMonitor(index)==0)
                {
                        ShutdownOS(E_OS_SYS_SOFLW);
                }
        }
        /* check extra (if any) target tests */
        if (_os_TargetStackMonitor()==0)
        {
                ShutdownOS(E_OS_SYS_TOFLW);
        }

        /* no overflow found .. hope there wasn't any */
        return;
}

/* writes the stack patterns */
void _os_InitStackMonitor(void)
{
        _os_RTOSMEM struct _os_stack_info* ptr = &_os_stack_monitor[0];         
        uint_fast8_t id;

        /* target initialization code */
        _os_TargetInitStackMonitor();

        /* do the pattern filling */
        for (id=0; id<_os_NO_TASK + _os_SYSTEM_STACK_NO ;id++,ptr++)
        {
                *(ptr->address)                 = _os_STCK_PATTERN_1;
                *(ptr->address + ptr->growth)   = _os_STCK_PATTERN_2;

        }
        return;
}


static uint_fast8_t _os_TestStackMonitor(TaskType id)
{
        _os_RTOSMEM struct _os_stack_info* ptr = &_os_stack_monitor[id];
        
        if (    (*(ptr->address) != _os_STCK_PATTERN_1) ||
                (*(ptr->address + ptr->growth) != _os_STCK_PATTERN_2 )
           )
        {
                        return 0;
        }

        return 1;
}

static StatusType _os_StackRange(void)
{
        uint_fast8_t index;
        
        /* Test 'running task' has its stack in range.
         * '_os_current_task' could be zero in the grey area
         * (an interrupt coming after a task has been already
         * saved/removed but the scheduler has not been called
         * yet to resume a new task).
         */
        if ( (_os_current_task != (_os_RTOSMEM _os_TASK*)0)  &&
             (_os_TestStackRange(_os_current_task->user_id)==0) 
           )
        {
                return E_OS_SYS_UOFLW;
        }
        /* and the global stacks */
        for (index = _os_NO_TASK;index<_os_NO_TASK+_os_SYSTEM_STACK_NO;index++)
        {
                if (_os_TestStackRange(index)==0)
                {
                        return E_OS_SYS_SOFLW;
                }
        }
        /* check extra (if any) target tests */
        if (_os_TargetIsStackInRange()==0)
        {
                return E_OS_SYS_TOFLW;
        }

        return E_OK;
}

static uint_fast8_t _os_TestStackRange(TaskType id)
{
        _os_STACKMEM uint8_t *sptr = _os_GetStackPointer(id);
        _os_STACKMEM struct _os_stack_info *ptr = &_os_stack_monitor[id];

        if ( ptr->growth == _os_STACK_DOWNWARDS) 
        {
                if(sptr < ptr->address )
                {
                        return 0;
                }
        }
        else
        {
                if (sptr > ptr->address)
                {
                        return 0;
                }
        }

        return 1;
}
#endif /* STACKMONITOR */


