/************************************************************************
 *  FILE:       @(#)c_shutdownos.c      1.27 04/10/18    
 *  DESCRIPTION:
 *      Source code for 'ShutdownOS' 
 ***********************************************************************/
#include "c_common.h"
#include "c_target.h"
#include "g_conf.h"

/* indicates the system is "shutting down" */
uint_fast8_t                    _os_shutting;

/********************************************************
 * Routines for multireset RTOSs.                       *
 * If the OS attribute MULTISTART is set in the OIL     *
 * file the application could undergo multiple starts   *
 * (ie application resets) everytime the user calls     *
 * 'ShutdownOS()' or if a fatal error is encountered.   *
 *      .....                                           *
 *      while(1)                                        *
 *      {                                               *
 *              StartOS(mode);                          *
 *      }                                               *
 *      .....                                           *
 * ShutdownOS()-> app reset -> StartOS()                *
 *                                                      *
 * The first power-up RTOS initialization differs       * 
 * from the subsequent initializations in many ways.    *
 * The 'Resetxxxx' routines takes care of preparing     *
 * the 'xxxx' component across these software resets.   *
 *                                                      *
 * To maximize execution speed and image size the       *
 * MULTISTART attribute should be defined as FALSE if   *
 * no application reset is allowed in the system.       *
 *******************************************************/
#if (_os_OS_0_MULTISTART>0)

/* static reset routines */
static void _os_ResetTask(void);
static void _os_ResetTaskContext(_os_RTOSMEM _os_TASK* task);
static void _os_ResetAlarm(void);
#if (_os_NO_USER_RESOURCE >0)
static void _os_ResetResource(void);
#endif
#if (_os_NO_MESSAGE >0)
static void _os_ResetMessage(void);
#endif
static void _os_ResetGlobals(void);

static void _os_ResetRTOS(void);
#endif


void  
#if (_os_SIA>0)
_os_ShutdownOS
#else
ShutdownOS
#endif
(StatusType error)
{

        /* dont need to check good run-level, otherwise
         * we could end up calling shutdownos recursively
         *
         * _os_CHECKRUNLEVEL(4); */
        
        /* only ISR1s from now on */
        _os_SuspendOS();
        
        #if (_os_OS_0_SHUTDOWNHOOK==1)
        {
                /********************************************* 
                 * IT DOES NOT NEED TO RETURN: user specific *
                 * BUT if it does return, the implementation *
                 * cleans-up the system and returns from     *
                 * StartOS() (like)                          *
                 ********************************************/
                _os_SETSHUTHOOK();
                ShutdownHook(error);
                _os_CLRSHUTHOOK();
        }
        #else
                (void)error;
        #endif

        /* see (TerminateTask/ChainTask) */ 
        _os_ChangeStack();

        /* system shuts down */
        _os_shutting = 1;

#if (_os_OS_0_MULTISTART>0)
         /* reset all RTOS variables/status/... */
         _os_ResetRTOS();
#endif
        /* ISR2s are disabled now */

        /* return from StartOS() */
         _os_ReturnToStartOS();

        /* never here */
        return;       
}


#if (_os_OS_0_MULTISTART>0)
/************************************************************************
 *  FUNCTION:           _os_ResetRTOS                                   *
 *  DESCRIPTION:                                                        *
 *      This routine is called from the system service 'ShutdownOS()'   *
 *      and it is responsible for the clean-up of the RTOS.             *
 ***********************************************************************/
static void
_os_ResetRTOS(void)
{
        _os_SuspendOS();

        /* count up application resets */
        _os_no_reset++;
        
        /* tasks re-initialization */      
        _os_ResetTask();
        
#if (_os_NO_USER_RESOURCE>0)
        /* resources re-initialization */
        _os_ResetResource();
#endif
        /* alarms re-initialization */
        _os_ResetAlarm();

#if (_os_NO_USER_MESSAGE>0)
        /* messages re-initialization */
        _os_ResetMessage();
#endif

        /* ISRs initialisation */
#if (_os_NO_ISR > 0)
        _os_ResetISR();
#endif
            
        /* Reset RTOS globals */
        _os_ResetGlobals();
        
        /* Reset current task */
        _os_ResetTaskContext(_os_current_task);
        
        /* no current task any more */
         _os_current_task=(_os_RTOSMEM _os_TASK*)0;
        
         return;
}

/************************************************************************
 *  FUNCTION:           ResetGlobals                                    *
 *  DESCRIPTION:                                                        *
 *      This routine is called from the system service 'ShutdownOS()'   *
 *      and it is responsible for the clean-up of the RTOS globals.     *
 ***********************************************************************/
static void
_os_ResetGlobals(void)
{
    uint_fast8_t i;

    /* we could shutdown from ISR2 (never from ISR1) */
#if (_os_NO_ISR>0)
    _os_isr2_counter            = 0;
#endif
    _os_lock                    = -1;
    _os_ready_to_run_index      = 0;
    _os_nesting_sros            = 0;
    _os_nesting_sr              = 0;
    _os_run_level               = 0;
#if (_os_OS_0_ORTI>0)
   _os_servicetrace             = 0;
   _os_rtos_cycles              = 0;
   _os_enter_cycles             = 0;
   _os_total_cycles             = 0;
#endif
    
#if (_os_OS_0_ERRORHOOK==1 || _os_COM_0_COMERRORHOOK==1)
#if ((_os_OS_0_USEGETSERVICEID == 1) || (_os_COM_0_COMERRORGETSERVICEID==1)||(_os_OS_0_ORTI>0) )
/* last failing system service */
        _os_errid               = 0;
#endif
#if (_os_OS_0_ORTI>0)
        _os_errno               = E_OK;
#endif
#endif

    for (i=1;i<_os_OS_0_NO_PRIORITIES;i++)
    {
#if (_os_OS_0_ONE_TASK_PER_PRIO !=1)
        _os_ready_to_run[i]     = (_os_RTOSMEM _os_TASK**)0;
#else
        _os_ready_to_run[i]     = (_os_RTOSMEM _os_TASK*)0;
#endif
    }
}

/*************************************************************
 *  FUNCTION:           ResetTask                            *
 *  DESCRIPTION:                                             *
 *      It does re-set again all the non-suspended tasks.    *
 *      Nothing needs being done for the idle task.          *
 *      The suspended tasks are set already.                 *
 ************************************************************/
static void
_os_ResetTask(void)
{
#if (_os_NO_USER_TASK >0)
        uint_fast8_t i;
        _os_RTOSMEM _os_TASK* task = &_os_task_table[1];

        /* let us init again with t_rtos.h values */
        for ( i=1;i<_os_NO_USER_TASK+1;i++,task++ )
        {
            if (
                task->state != SUSPENDED &&
                task != _os_current_task
                )
            {
        
                 _os_ResetTaskContext(task);
            }
            
        }
#endif
}


#if (_os_NO_USER_RESOURCE >0)
/***************************************************************
 *  FUNCTION:           ResetResource                          *
 *  DESCRIPTION:                                               *
 *      It does reset all locked resources.                    *
 **************************************************************/
static void
_os_ResetResource(void)
{
        uint_fast8_t i;
        _os_RTOSMEM _os_RESOURCE* res = &_os_resource_table[1];

        /* let us init again with t_rtos.h values */
        for ( i=1;i<_os_NO_USER_RESOURCE;i++,res++ )
        {
            if (res->preprio)
            {
                /* unlock it */
                res->preprio      = 0;
#if (_os_OS_0_STATUS == _os_OS_EXTENDED)
                /* dequeue it */
                res->nestedto     = _os_INVALID_RES;
#endif
            }
        }
}
#endif


/****************************************************************
 *  FUNCTION:           ResetAlarm                              *              
 *  DESCRIPTION:                                                *
 *      It does reset the non-stopped alarms and the running    *
 *      counters.                                               *
 ***************************************************************/
static void
_os_ResetAlarm(void)
{
        uint_fast8_t i;
        _os_RTOSMEM _os_COUNTER* counter = &_os_counter_table[SYSTEM_TIMER];

#if (_os_NO_USER_ALARM >0)
        _os_RTOSMEM _os_ALARM* alarm = &_os_alarm_table[0];

        /* let us init again with t_rtos.h values */
        for ( i=0;i<_os_NO_USER_ALARM;i++,alarm++ )
        {
            if (alarm->status != _os_AL_STOPPED)
            {
                /* Alarm is stopped */
                alarm->status       = _os_AL_STOPPED;
                /* dequeue it */
                alarm->next         = (_os_RTOSMEM _os_ALARM*)0;
                /* we assume all are non-autostart here */
                alarm->ALARMTIME    = 0;
                alarm->CYCLETIME    = 0;
            }
        }
#endif

        /* let us init again with t_rtos.h values */
        for ( i=0;i<_os_NO_USER_COUNTER+1;i++,counter++ )
        {
            /* reset counter */
            counter->value = (TickType)0;
            counter->ticks = 1;
            counter->times = (_os_RTOSMEM _os_ALARM*)0;
        }
        return;
}

#if (_os_NO_USER_MESSAGE>0)
/****************************************************
 *  FUNCTION:           ResetMessage                *
 *  DESCRIPTION:                                    *
 *      Resets internals of the receive queued msgs * 
 ***************************************************/
 static void _os_ResetMessage(void)
 {
        uint_fast8_t i;
        _os_RTOSMEM _os_MESSAGE* msg = &_os_message_table[0];

        /* let us init again with t_rtos.h values */
        for ( i=0;i<_os_NO_USER_MESSAGE;i++,msg++ )
        {
            if (msg->MESSAGEPROPERTY == _os_ME_RECEIVE_QUEUED_INTERNAL)
            {
                msg->queuestart     = 0;
                msg->queuelength    = 0;
                #if (_os_OS_0_LONGMSG>0)
                msg->queuestatus    = 0;
                #endif
            }
        }
 }
#endif   

static void _os_ResetTaskContext(_os_RTOSMEM _os_TASK* task)
{
        /* reset task context */
       
        /* common parts */      
        #if ((_os_OS_0_STATUS == _os_OS_EXTENDED) ||(_os_OS_0_ORTI>0))
        task->res              = _os_INVALID_RES;
        #endif
#if (_os_NO_EVENT>0)
        task->set              = 0;
        task->wait             = 0;
#endif
        task->PRIORITY         = task->staticprio;

        /* task is now suspended */
        task->state     =  SUSPENDED;
        #if (_os_OS_0_ONE_TASK_PER_PRIO !=1)
        /* dequeue task from ready-queue */
        task->ready     =  (_os_RTOSMEM _os_TASK*)0;
        #endif

        /* just in case */
        task->rc        =  0;
        task->repeat    =  0;

        #if (_os_OS_0_ORTI>0)
        task->servicetrace = 0;
        task->errid        = 0;
        task->cycles       = 0; 
        #endif       
        
        /* target parts */
        _os_TargetResetTaskContext(task->user_id);
        
        return;
}

#endif // MULTISTART





