/******************************************************************************
**  FILE:       @(#)c_rtos.h    1.26 04/11/03 
**  DESCRIPTION:
**      Common definitions for this osek implementation
******************************************************************************/
#ifndef _C_RTOS_H
#define _C_RTOS_H

#include <osek/osek.h>
#include <osek/types.h>
#include <time.h>
#include "t_rtos.h"

#define         _os_END_SEARCH          -1
#define         _os_idle                 0

/******************************************************************************
 * TYPEDEF:     _os_EVENTMASKTYPE
 * DESCRIPTION: Internal typedefs to optimize handling of 
 *      event masks: Max 32 bits.
******************************************************************************/
#if   (_os_NO_EVENT<9)
typedef uint8_t  _os_EVENTMASKTYPE;
#elif (_os_NO_EVENT<17)
typedef uint16_t _os_EVENTMASKTYPE;
#elif (_os_NO_EVENT<33)
typedef uint32_t _os_EVENTMASKTYPE;
#else
#error "OIL Error: Max number of events is 32"
#endif

/******************************************************************************
 * TYPEDEF:     _os_BOOLEAN
 * DESCRIPTION: Internal rtos boolean type
******************************************************************************/
typedef uint_fast8_t    _os_BOOLEAN;

/******************************************************************************
 * Default OIL attributes for the IDLE task:
 *  (OIL part of _os_TASKtable[0])
 *                                      
 * _os_TASK0_TARGETRTOS_INIT initializes the
 *  non-standard attributes of every target (t_rtos.h)
 *****************************************************************************/
#define   _os_TASK0_RTOS_INIT   0,\
                                _os_TA_FULL,\
                                1,\
                                1,\
                                {OSDEFAULTAPPMODE},\
                                {-1},\
                                {-1},\
                                &_os_Idle,\
                                0, \
                                _os_TASK0_TARGETRTOS_INIT 

/******************************************************************************
 * Default OIL attributes for the SCHEDULER resource:
 * (OIL part of _os_RESOURCEtable[0])
 *****************************************************************************/
#define         _os_RESOURCE0_RTOS_INIT         _os_RE_STANDARD,\
                                                _os_INVALID_RES,\
                                                _os_OS_0_NO_PRIORITIES-1

/******************************************************************************
 * Default OIL attributes for the system timer counter
 * (OIL part of _os_COUNTERtable[0])
 *****************************************************************************/
#define         _os_COUNTER0_RTOS_INIT          OSMAXALLOWEDVALUE,\
                                                OSTICKSPERBASE,\
                                                OSMINCYCLE,

/******************************************************************************
 * MACRO:       _os_TASK_RTOS 
 * DESCRIPTION
 *      Non-OIL part of _os_TASK.
 *
 *      - ready: Points to next 'ready' task in the same priority level.
 *      - state: Current 'state' of the task.
 *      - rc: ready counter.
 *      - staticprio: The statically assigned priority.
 *      - event_info: Stores the current information
 *               regarding events for the task.
 *      - user_id: Links the TASK with its index in the TASKtable.
 *      - repeat: for multiple activations.
 *      - internal: the internal resource of the task (if any).
 *      - res: Indicates the latest 'resource' occupied by the task
 *             (for linked resources).
 *      - noruns: Indicates the number of runs of every task
 *      - pc: Indicates task next execution address
 *      - servicetrace: last service (entry/exit)
 *      - errid: last failing service
 *      - cycles: (no of cycles spent in the task - as counted by
 *                 library service 'clock' -).
 *      _os_TASK_TARGETRTOS defines the non-OIL target specific part
 * 
 * REMARKS:
 *      _os_TASK_RTOS always defined non-empty.
 *****************************************************************************/
#ifndef _os_TASK_TARGETRTOS
#define _os_TASK_TARGETRTOS
#endif

#if (_os_OS_0_ONE_TASK_PER_PRIO !=1)
#define _os_TASKREADY           _os_RTOSMEM _os_TASK   *ready;
#else
#define _os_TASKREADY   
#endif

#if ((_os_OS_0_STATUS == _os_OS_EXTENDED)||(_os_OS_0_ORTI>0))
#define _os_TASKRESOURCE        ResourceType            res;
#else
#define _os_TASKRESOURCE        
#endif
                                                
#if (_os_OS_0_ORTI>0)
#define _os_TASKNORUNS          uint16_t         noruns;
#define _os_TASKPC              uintptr_t        pc;
#define _os_TASKSERVICETRACE    uint_fast8_t     servicetrace;
#define _os_TASKERRID           uint_fast8_t     errid;
#define _os_TASKCYCLES          clock_t          cycles;
#else
#define _os_TASKNORUNS          
#define _os_TASKPC      
#define _os_TASKSERVICETRACE            
#define _os_TASKERRID           
#define _os_TASKCYCLES  
#endif

#if (_os_NO_EVENT>0)
#define _os_TASKEVENTS          _os_EVENTMASKTYPE       events;         
#define _os_TASKSET             _os_EVENTMASKTYPE       set;            
#define _os_TASKWAIT            _os_EVENTMASKTYPE       wait;           
#else
#define _os_TASKEVENTS                          
#define _os_TASKSET                     
#define _os_TASKWAIT                            
#endif

#define  _os_TASK_RTOS  _os_TASK_TARGETRTOS \
                        TaskStateType        state;\
                        _os_TASKREADY \
                        uint_fast8_t         rc;\
                        uint_fast8_t         staticprio;\
                        _os_TASKEVENTS \
                        _os_TASKSET \
                        _os_TASKWAIT \
                        TaskType             user_id;\
                        uint_fast8_t         repeat;\
                        _os_TASKNORUNS \
                        _os_TASKPC \
                        _os_TASKSERVICETRACE \
                        _os_TASKERRID \
                        _os_TASKCYCLES \
                        _os_TASKRESOURCE \
                        ResourceType         internal;
                                                

/******************************************************************************
 * MACRO:       _os_TASK_RTOS_INIT
 * DESCRIPTION:
 *      Default initialisation for the NON_OIL part of _os_TASK.
 *
 *      - ready '0': indicates no chain of ready tasks
 *      - state 'SUSPENDED': A task is initially in suspended state. The system
 *           startup will set the state to 'READY' for the
 *           autostarting tasks.
 *      - rc '0':
 *      - staticprio 'irrelevant': value assigned during startup.
 *      - events/set/wait '0': All fields being empty mean no events are set/wait.
 *      - user_id: Value assigned during startup.
 *      - repeat '0': '0' means that after the first activation no other
 *           activation request is queued.
 *      - internal '_os_INVALID_RES': Initially we do assume the task has no
 *           internal resource assigned. Startup    
 *           will overwrite this field if necessary.
 *      - res '_os_INVALID_RES': Initially we do assume that the task is not
 *           occupying any resource.
 *      - 'noruns': 0
 *      - 'pc': 0
 *      - 'servicetrace': 0
 *      - 'errid': 0
 *      - 'cycles': 0
 *      _os_TASK_TARGETRTOSINIT initializes the target part.
 * 
 * REMARKS:
 *      _os_TASK_RTOS_INIT always defined non-empty.
 *****************************************************************************/
#ifndef _os_TASK_TARGETRTOSINIT
#define _os_TASK_TARGETRTOSINIT
#endif

#if (_os_OS_0_ONE_TASK_PER_PRIO !=1)
#define _os_TASKREADYINIT               (_os_RTOSMEM _os_TASK*)0,
#else
#define _os_TASKREADYINIT       
#endif

#if ((_os_OS_0_STATUS == _os_OS_EXTENDED)||(_os_OS_0_ORTI>0))
#define _os_TASKRESOURCEINIT            _os_INVALID_RES,
#else
#define _os_TASKRESOURCEINIT
#endif

#if (_os_OS_0_ORTI>0)
#define _os_TASKNORUNSINIT              0,
#define _os_TASKPCINIT                  0,
#define _os_TASKSERVICETRACEINIT        0,
#define _os_TASKERRIDINIT               0,
#define _os_TASKCYCLESINIT              0,
#else
#define _os_TASKNORUNSINIT              
#define _os_TASKPCINIT          
#define _os_TASKSERVICETRACEINIT
#define _os_TASKERRIDINIT               
#define _os_TASKCYCLESINIT
#endif

#if (_os_NO_EVENT>0)
#define _os_TASKEVENTSINIT              0,              
#define _os_TASKSETINIT                 0,              
#define _os_TASKWAITINIT                0,              
#else
#define _os_TASKEVENTSINIT                              
#define _os_TASKSETINIT                 
#define _os_TASKWAITINIT                                
#endif

#define         _os_TASK_RTOS_INIT      _os_TASK_TARGETRTOSINIT \
                                        SUSPENDED,\
                                        _os_TASKREADYINIT \
                                        0,\
                                        0,\
                                        _os_TASKEVENTSINIT \
                                        _os_TASKSETINIT \
                                        _os_TASKWAITINIT \
                                        0,\
                                        0,\
                                        _os_TASKNORUNSINIT \
                                        _os_TASKPCINIT \
                                        _os_TASKSERVICETRACEINIT \
                                        _os_TASKERRIDINIT \
                                        _os_TASKCYCLESINIT \
                                        _os_TASKRESOURCEINIT \
                                        _os_INVALID_RES,
                                        
/******************************************************************************
 * MACRO:       _os_RESOURCE_RTOS
 * DESCRIPTION:
 *      NON_OIL part of _os_RESOURCE.
 *      - isrceiling: extension of ceiling priority to isr
 *      - nestedto:  For handling nested accesses.
 *      - locker:    who locks the resource
 *      - preprio: Priority before the resource was accessed.
 *                 0 indicates resource is not locked.                  
 *      _os_RESOURCE_TARGETRTOS defines the non-OIL target specific part
 *
 * REMARKS:
 *      _os_RESOURCE_RTOS always defined non-empty.
 *****************************************************************************/
#ifndef _os_RESOURCE_TARGETRTOS
#define _os_RESOURCE_TARGETRTOS
#endif

#if ((_os_OS_0_STATUS == _os_OS_EXTENDED)||(_os_OS_0_ORTI>0))
#define _os_RESOURCENESTED              ResourceType nestedto;
#else
#define _os_RESOURCENESTED              
#endif
                                        
#if (_os_OS_0_ORTI>0)
#define _os_RESOURCELOCKER              int_fast8_t locker;
#else
#define _os_RESOURCELOCKER
#endif
                                        
/* compulsory presence, no chance to
 * include 'g_conf.h' in here.
 * 
#if (MAX_RESOURCE_ISR>0)
#define _os_RESOURCECEILING     uint_fast8_t    isrceiling;                                     
#else
#define _os_RESOURCECEILING                                             
#endif
 */
                                        
#define  _os_RESOURCE_RTOS      _os_RESOURCE_TARGETRTOS \
                                _os_RESOURCENESTED \
                                _os_RESOURCELOCKER \
                                uint_fast8_t        isrceiling; \
                                uint_fast8_t        preprio;
                                        
/******************************************************************************
 * MACRO:       _os_RESOURCE_RTOS_INIT
 * DESCRIPTION:
 *      Default initialisation for the NON_OIL part of _os_RESOURCE.
 *      - nestedto: _os_INVALID_RES
 *      - locker:   0 (no locker)
 *      - preprio 'run time' 
 *      - isrceiling 'run time' 
 *      
 *      _os_RESOURCE_TARGETRTOSINIT initializes the target part.
 * 
 * REMARKS:
 *      _os_RESOURCE_RTOS_INIT always defined non-empty.
 *****************************************************************************/
#ifndef _os_RESOURCE_TARGETRTOSINIT
#define _os_RESOURCE_TARGETRTOSINIT
#endif

#if ((_os_OS_0_STATUS == _os_OS_EXTENDED)||(_os_OS_0_ORTI>0))
#define _os_RESOURCENESTEDINIT          _os_INVALID_RES,
#else
#define _os_RESOURCENESTEDINIT          
#endif

#if (_os_OS_0_ORTI>0)
#define _os_RESOURCELOCKERINIT              0, 
#else
#define _os_RESOURCELOCKERINIT          
#endif

#define        _os_RESOURCE_RTOS_INIT    _os_RESOURCE_TARGETRTOSINIT \
                                         _os_RESOURCENESTEDINIT \
                                         _os_RESOURCELOCKERINIT \
                                          0, \
                                          0,
                                        
/******************************************************************************
 * MACRO:       _os_ALARM_RTOS
 * DESCRIPTION:
 *      NON_OIL part of _os_ALARM.
 *                                                              
 *      - next: chain to next alarm in same counter timer queue.
 *      - status: running,stopped,busy
 *      - statictime/cycle: stores ALARMTIME/CYCLETIME for autostart
 *      alarms across application resets.
 *      - id: identifies the alarm for postmortem analysis.
 *
 *      _os_ALARM_TARGETRTOS defines the non-OIL target specific part
 *
 * REMARKS:
 *      _os_ALARM_RTOS always defined non-empty.
 *****************************************************************************/
#if (_os_OS_0_MULTISTART>0)
        #define _os_ALARM_MULTISTART    TickType   statictime;\
                                        TickType   staticcycle;
#else
        #define _os_ALARM_MULTISTART      
#endif

#ifndef _os_ALARM_TARGETRTOS
#define _os_ALARM_TARGETRTOS
#endif

#if     (_os_OS_0_POSTMORTEM != _os_OS_NONE)
        #define _os_ALARM_ID            AlarmType    id;
#else
        #define _os_ALARM_ID    
#endif

#define         _os_ALARM_RTOS      _os_ALARM_TARGETRTOS \
                                    _os_RTOSMEM _os_ALARM   *next;\
                                    _os_ALARM_ID \
                                    _os_ALARM_MULTISTART \
                                    uint_fast8_t      status;

/******************************************************************************
 * MACRO:       _os_ALARM_RTOS_INIT
 * DESCRIPTION:
 *      Default initialisation for the NON_OIL part of _os_ALARM.
 *               - not chained to any counter queue ('next' =0).
 *               - stopped
 *               - statictime/cycle at run time
 *               - id at run time 
 *      _os_ALARM_TARGETRTOSINIT initializes the target part.
 * 
 * REMARKS:
 *      _os_ALARM_RTOS_INIT always defined non-empty.
 *****************************************************************************/
#if (_os_OS_0_MULTISTART>0)
#       define  _os_ALARM_MULTISTART_INIT       0,0,
#else
#       define  _os_ALARM_MULTISTART_INIT      
#endif

#ifndef _os_ALARM_TARGETRTOSINIT
#define _os_ALARM_TARGETRTOSINIT
#endif

#if     (_os_OS_0_POSTMORTEM != _os_OS_NONE)
        #define _os_ALARM_ID_INIT               0,
#else
        #define _os_ALARM_ID_INIT
#endif

#define         _os_ALARM_RTOS_INIT     _os_ALARM_TARGETRTOSINIT \
                                        (_os_RTOSMEM _os_ALARM*)0,\
                                        _os_ALARM_ID_INIT \
                                        _os_ALARM_MULTISTART_INIT \
                                        _os_AL_STOPPED,     

/******************************************************************************
 * MACRO:       _os_COUNTER_RTOS
 * DESCRIPTION:
 *      NON_OIL part of _os_COUNTER.
 *
 *      - value: This field increments every time a counter recurring event
 *          happens in the system.
 *      - ticks: Counts counter 'ticks'. The counter value must be incremented
 *          every TICKSPERBASE ticks.
 *      - times: Beginning of the queue of all time slots associated for
 *          this counter. 
 *
 *      _os_COUNTER_TARGETRTOS defines the non-OIL target specific part
 *
 * REMARKS:
 *      _os_COUNTER_RTOS always defined non-empty.
 *****************************************************************************/
#ifndef _os_COUNTER_TARGETRTOS
#define _os_COUNTER_TARGETRTOS
#endif

#define         _os_COUNTER_RTOS        _os_COUNTER_TARGETRTOS \
                                        TickType   value;\
                                        TickType     ticks;\
                                        _os_ALARM *times;
                                
/******************************************************************************
 * MACRO:       _os_COUNTER_RTOS_INIT
 * DESCRIPTION:
 *      Default initialisation for the NON_OIL part of _os_COUNTER.
 *
 *      - value '0': The initial value is zero.
 *      - ticks '1': The initial value is one.
 *      - times '0': The counter queue is initially empty.
 *              'times' indicates the start of the queue. The startup
 *              code may set 'times' to point at an specific time slot
 *              corresponding to an autostart alarm. 
 *
 *      _os_COUNTER_TARGETRTOSINIT initializes the target part.
 *
 * REMARKS:
 *      _os_COUNTER_RTOS_INIT always defined non-empty.
 *****************************************************************************/
#ifndef _os_COUNTER_TARGETRTOSINIT
#define _os_COUNTER_TARGETRTOSINIT
#endif

#define         _os_COUNTER_RTOS_INIT  _os_COUNTER_TARGETRTOSINIT \
                                        0,1,0,

/******************************************************************************
 * MACRO:       _os_MESSAGE_RTOS 
 * DESCRIPTION:
 *      NON_OIL part of _os_MESSAGE. 
 *
 * - 'queuestart': Start of the receive queue (for queued receiver message objects).
 * - 'queuelength': Length of the receive queue (for queued receiver message objects).                           
 * - 'queuestatus': every n-bit of this mask indicates the disponibility
 *                  of the nth message of the queue to be read
 *                  (only set while the message is copied to the queue).
 *                  It limits the maximum size of the queue to 32!!!!
 *
 *      _os_MESSAGE_TARGETRTOS defines the non-OIL target specific part
 *
 * REMARKS:
 *      _os_MESSAGE_RTOS always defined non-empty.
 *****************************************************************************/

#if     (_os_NO_USER_MESSAGE < 9)
typedef uint_fast8_t    _os_MSGSTATUSTYPE;
#elif   (_os_NO_USER_MESSAGE < 17)
typedef uint_fast16_t   _os_MSGSTATUSTYPE;
#elif    (_os_NO_USER_MESSAGE < 33)
typedef uint_fast32_t  _os_MSGSTATUSTYPE;
#else
#error Error: "Number MESSAGEs > 32"
#endif
                                        
#if (_os_OS_0_LONGMSG>0)
#       define  _os_MESSAGE_LONGMSG     _os_MSGSTATUSTYPE        queuestatus;
#else
#       define  _os_MESSAGE_LONGMSG
#endif

#ifndef _os_MESSAGE_TARGETRTOS
#define _os_MESSAGE_TARGETRTOS
#endif

#define         _os_MESSAGE_RTOS        _os_MESSAGE_TARGETRTOS \
                                        uint_fast8_t       queuestart;\
                                        _os_MESSAGE_LONGMSG \
                                        uint_fast8_t    queuelength;

/******************************************************************************
 * MACRO:       _os_MESSAGE_RTOS_INIT
 * DESCRIPTION:
 *      Default initialisation for the NON_OIL part of _os_MESSAGE
 *
 *       - queuestart, queuelength, queuestatus '0': Initially queues are empty
 *
 *       _os_MESSAGE_TARGETRTOSINIT initializes the target part.        
 *
 * REMARKS:
 *      _os_MESSAGE_RTOS_INIT always defined non-empty.
 *****************************************************************************/
#if (_os_OS_0_LONGMSG>0)
#       define  _os_MESSAGE_LONGMSG_INIT       0,
#else
#       define  _os_MESSAGE_LONGMSG_INIT
#endif

#ifndef _os_MESSAGE_TARGETRTOSINIT
#define _os_MESSAGE_TARGETRTOSINIT
#endif

#define _os_MESSAGE_RTOS_INIT       _os_MESSAGE_TARGETRTOSINIT \
                                    0,\
                                    _os_MESSAGE_LONGMSG_INIT \
                                    0,

/******************************************************************************
 * MACRO:       _os_ISR_RTOS
 * DESCRIPTION:
 *      NON_OIL part of _os_ISR.
 *      - 'prev' : previous nested interrupt
 *      - 'noruns': number of times the ISR has executed.
 *      - 'servicetrace': last service in ISR (entry/exit)
 *      - 'res': Current resource owned by ISR 
 *      - cycles: (no of cycles spent in the task - as counted by
 *                 library service 'clock' -).
 *      
 *      _os_ISR_TARGETRTOS defines the non-OIL target specific part (if present)
 *
 * REMARKS:
 *      The macro _os_ISR_RTOS must not be defined if the 
 *      NON_OIL part of _os_ISR is empty.
 *****************************************************************************/

#if ( defined(_os_ISR_TARGETRTOSINIT) || (_os_OS_0_ORTI>0) || (_os_OS_0_STATUS == _os_OS_EXTENDED) )

#ifndef _os_ISR_TARGETRTOS
#define _os_ISR_TARGETRTOS
#endif

#if (_os_OS_0_ORTI>0)
#define _os_ISRNORUNS           uint16_t        noruns;
#define _os_ISRSERVICETRACE     uint_fast8_t    servicetrace;
#define _os_ISRCYCLES           clock_t         cycles;
#else
#define _os_ISRNORUNS
#define _os_ISRSERVICETRACE
#define _os_ISRCYCLES
#endif
            
#if ((_os_OS_0_STATUS == _os_OS_EXTENDED)||(_os_OS_0_ORTI>0))
#define _os_ISRRESOURCE         ResourceType            res;
#else
#define _os_ISRRESOURCE 
#endif

#define _os_ISR_RTOS            _os_ISR_TARGETRTOS \
                                _os_RTOSMEM _os_ISR * prev; \
                                _os_ISRNORUNS \
                                _os_ISRSERVICETRACE \
                                _os_ISRCYCLES \
                                _os_ISRRESOURCE
#else
/* _os_ISR_RTOS is left undefined */
#endif

                                
/******************************************************************************
 * MACRO:       _os_ISR_RTOS_INIT
 * DESCRIPTION:
 *      Default initialisation for the NON_OIL part of _os_ISR.
 *       - 0: no nested interrupt
 *       - 0: noruns
 *       - 0: servicetrace
 *       - _os_INVALID_RESOURCE: resource initially owned by the ISR 
 *       - cycles: 0    
 *      _os_ISR_TARGETRTOSINIT initializes the target part (if present).
 *
 * REMARKS:
 *      The macro _os_ISR_RTOS_INIT must not be defined if the 
 *      NON_OIL part of _os_ISR is empty.
 *****************************************************************************/

#if ( defined(_os_ISR_TARGETRTOSINIT) || (_os_OS_0_ORTI>0) || (_os_OS_0_STATUS == _os_OS_EXTENDED) )

#ifndef _os_ISR_TARGETRTOSINIT
#define _os_ISR_TARGETRTOSINIT
#endif

#if (_os_OS_0_ORTI>0)
#define _os_ISRNORUNSINIT               0,
#define _os_ISRSERVICETRACEINIT         0,
#define _os_ISRCYCLESINIT               0,
#else
#define _os_ISRNORUNSINIT               
#define _os_ISRSERVICETRACEINIT 
#define _os_ISRCYCLESINIT
#endif

#if ((_os_OS_0_STATUS == _os_OS_EXTENDED)||(_os_OS_0_ORTI>0))
#define _os_ISRRESOURCEINIT     _os_INVALID_RES,
#else
#define _os_ISRRESOURCEINIT     
#endif

#define _os_ISR_RTOS_INIT       _os_ISR_TARGETRTOSINIT \
                                (_os_RTOSMEM _os_ISR *)0,\
                                _os_ISRNORUNSINIT \
                                _os_ISRSERVICETRACEINIT \
                                _os_ISRCYCLESINIT \
                                _os_ISRRESOURCEINIT 
#else
/* _os_ISR_RTOS_INIT is left undefined */
#endif

                                
#endif /* _C_RTOS_H */ 











