/*******************************************************************************
 *  FILE:      @(#)c_receivemessage.c   1.13 04/09/15
 *  DESCRIPTION:
 *      Source code for 'ReceiveMessage'
 *******************************************************************************/
#include "c_common.h"
#include "c_target.h"
#include "flag.h"

/*******************************************************************************
 * FUNCTION:    GetDataReceiver
 * DESCRIPTION: It retrieves the data stored in the receiver message object
 *              identified by 'msg' and copy them into the application data.
 *******************************************************************************/
static StatusType _os_GetDataReceiver(SymbolicName msg,uint8_t* dst);

StatusType     
#if (_os_SIA>0)
_os_ReceiveMessage   
#else
ReceiveMessage   
#endif
(SymbolicName msg, ApplicationDataRef data)
{
        StatusType ret = E_COM_ID;
#if (_os_NO_MESSAGE > 0)

        _os_lock++;
        
        /* system entry actions */
        _os_ServiceEntry(_os_N_ReceiveMessage);
        
        #if (_os_COM_0_COMSTATUS == _os_CO_COMEXTENDED)

         if (   msg < _os_NO_MESSAGE 
                && _os_message_table[msg].MESSAGEPROPERTY != _os_ME_SEND_STATIC_INTERNAL
                && (ret=_os_IsMessageOwned(msg))==E_OK 
            )
         {
       #endif

                /* copy data from receive object buffer 'msg' to application data */ 
                ret = _os_GetDataReceiver(msg,(uint8_t*)data);
        
        #if (_os_COM_0_COMSTATUS == _os_CO_COMEXTENDED)
        }
        #endif

        #if (_os_COM_0_COMERRORHOOK==1)
        if(ret!=E_OK)
        {
                _os_LogError(ret,_os_TO_REC_MSG,(int32_t)msg,(int32_t)data,-1);
        }
        #endif

        if (_os_NeedToSwitchTasks())
        {
                /* OS are suspended now */
                _os_TaskSwitch();
                /* never here */
        }
        
        /* system exit actions */
        _os_ServiceExit(_os_N_ReceiveMessage);
#else
        (void)msg;(void)data;
#endif

        return ret;
}


#if (_os_NO_MESSAGE>0)

static StatusType _os_GetDataReceiver(SymbolicName msg,uint8_t* dst)
{
        StatusType ret = E_OK;
        
        /* to copy the msg */
        _os_RTOSMEM uint8_t *src      = 0;
        _os_RTOSMEM _os_MESSAGE*    msgptr  = &_os_message_table[msg];
        DataLength mlength           = msgptr->LENGTH;
        
        /* queueud messages only: next reading -possible-
         * position in the queue */
        uint_fast8_t qid;
        
        _os_SuspendOS();
        
        if ( msgptr->MESSAGEPROPERTY ==_os_ME_RECEIVE_UNQUEUED_INTERNAL)
        {
                /* message is unqueued */
                src = (_os_RTOSMEM uint8_t*)_os_message_data[msg];

                /* While reading from an unqueued buffer we need to
                 * protect it against reentrancy. Imagine that
                 * 'ReceiveMessage()'is preempted -while in the
                 * copy process- an SendMessage() is issued */
#if (_os_OS_0_LONGMSG>0)
                while(mlength--)
                {
                     *dst++ = *src++;
                }
                _os_flag_table[msg] = COM_FALSE;
                _os_ResumeOS();
#endif
        }
        else if ( msgptr->MESSAGEPROPERTY==_os_ME_RECEIVE_QUEUED_INTERNAL )
        {
                /* message is queued */
                
                qid = msgptr->queuestart;
        
                        /* 1. Message is empty, nothing to read */      
                if ( msgptr->queuelength==0
#if (_os_OS_0_LONGMSG>0)
                        /* 2. This block still write-busy.
                         *    Imagine a SendMessage() on a empty queue.
                         *    The writing process is preempted and 
                         *    ReceiveMessage() is issued. The queue must
                         *    be considered empty until the message has
                         *    been completely written first.*/
                        || (msgptr->queuestatus & (1 <<qid ) )
#endif
                )
                {
                        /* queue is empty or block is write-busy */
                        _os_ResumeOS();
                        /* the queued message is empty */
                        ret = E_COM_NOMSG;
                        
                }
                else
                {
                        if (msgptr->queuelength == msgptr->QUEUESIZE+1)
                        {
                            /* Indicates queue was full and at least one 
                             * message has been discarded. 
                             * Nevertheless the service is performed and a message
                             * is returned. */
                             ret = E_COM_LIMIT;
                            msgptr->queuelength -=2;
                        }
                        else
                        {
                            msgptr->queuelength -=1;
                        }       
                        
                        /* Get the object*/
                        src = (_os_RTOSMEM uint8_t*)((_os_RTOSMEM uint8_t*)_os_message_data[msg]+
                                qid*mlength);
                        
                        /* next read in next block */
                        INCREASE(msgptr->queuestart,msg);
                        
#if (_os_OS_0_LONGMSG>0)
                        /* set this buffer to read-busy */
                        msgptr->queuestatus |= (1 << qid);
                        
                        _os_ResumeOS();
                        
                        /* all locals now */
                        
                        /* Copying needs no protection */
                        while(mlength--)
                        {
                             *dst++ = *src++;
                        }
                        
                        /* set this buffer to free
                         * Protect 'status' */
                        _os_DisableInterrupts();
                        _os_flag_table[msg] = COM_FALSE;
                        msgptr->queuestatus &= ~(1 << qid);
                        _os_EnableInterrupts();
#endif
                }
        }
#if (_os_OS_0_LONGMSG==0)
        if(ret!=E_COM_NOMSG)
        {
                while(mlength--)
                {
                     *dst++ = *src++;
                }
                _os_flag_table[msg] = COM_FALSE;
                _os_ResumeOS();
        }
#endif
        return ret;
}



#endif


