/***********************************************
 * FILE: @(#)c_postmortem.c     1.9 04/09/16
 * DESCRIPTION:
 *      Post mortem analysis software
 **********************************************/
#include "c_postmortem.h"
#include "c_common.h"
#include "c_target.h"

#if     (_os_OS_0_POSTMORTEM == _os_OS_CIRCULAR)
#       define RINGBUFSIZE      _os_OS_0_CIRCULARBUFSIZE        
#elif   (_os_OS_0_POSTMORTEM == _os_OS_REALTIME)
#       if (_os_OS_0_HARDLOGGING==1)
#       define RINGBUFSIZE      0       
#       else
#       define RINGBUFSIZE      _os_OS_0_REALTIMEBUFSIZE
#       endif
#else
#       define RINGBUFSIZE      0
#endif

#if     (_os_OS_0_POSTMORTEM == _os_OS_REALTIME)
static void _os_SendRT(_os_PEVENT *);
#endif



/**********************************************************
 * ring buffer ring buffer ring buffer ring buffer
 *********************************************************/

#if (RINGBUFSIZE>0)

/* static data */
/* circular history buffer */
_os_RTOSMEM _os_PEVENT _os_h_buffer[RINGBUFSIZE];
/* head and tail of ring buffer */
_os_RTOSMEM uint_fast8_t _os_head, _os_tail;

#if ( _os_OS_0_POSTMORTEM == _os_OS_REALTIME) && (_os_OS_0_HARDLOGGING==0) 

static uint_fast8_t  _os_EmptyRing(void);
_os_PEVENT _os_PopRing(void);

static uint_fast8_t _os_EmptyRing(void)
{
        return _os_head==_os_tail;
}

static void _os_PushRing(const _os_PEVENT* e);

_os_PEVENT _os_PopRing(void)
{
        _os_PEVENT tmp = _os_h_buffer[_os_head];
        if (++_os_head >= RINGBUFSIZE)
        {
                _os_head = 0;
        }
        return tmp;
}
#endif 


static void _os_PushRing(const _os_PEVENT* e)
{
     _os_PEVENT *tailptr = &_os_h_buffer[_os_tail];
     uint_fast8_t i;
     
     for (i=0;i<sizeof(uint32_t);i++)
             tailptr->time[i]  = e->time[i];
     tailptr->runprio  = e->runprio;
     tailptr->type  = e->type;
     tailptr->actor = e->actor;
     for (i=0;i<sizeof(uint32_t);i++)
             tailptr->info[i]  = e->info[i];
     _os_tail++;     
     if (_os_tail>=RINGBUFSIZE)
             _os_tail=0;
    
#if ( _os_OS_0_POSTMORTEM == _os_OS_REALTIME) && (_os_OS_0_HARDLOGGING==0) 
     if (_os_tail==_os_head)
     {
             ShutdownOS(E_OS_SYS_RTOFLW);
     }
#endif     
     return;
}
#endif 


/**********************************************************
 * real time send real time send real time send
 *********************************************************/
#if     (_os_OS_0_POSTMORTEM == _os_OS_REALTIME)

#define DLL                     (uint8_t)0x10
#define SPE                     (uint8_t)0x5A
#define _os_DLLSPE              2

static void _os_SendRT(_os_PEVENT*);

/* This buffer packs what needs to be sent.
 * No risk of concurrency problems.
 * Big enough for worst case scenario: all 0x10s.
 */
static uint8_t _os_rtbuf[ _os_DLLSPE + sizeof(_os_PEVENT)*2 ] =
{
        /* first two bytes are fixed */
        DLL,
        SPE,
};

static uint8_t _os_i_rtbuf=_os_DLLSPE;
static void _os_WriteRTBuf(uint8_t);

static void _os_SendRT(_os_PEVENT* pevent)
{
        uint_fast8_t i;
     
        /* build outgoing event package */
        for (i=0;i<sizeof(uint32_t);i++)
        {
             _os_WriteRTBuf((pevent->time[i]));
        }
        _os_WriteRTBuf(pevent->runprio);
        _os_WriteRTBuf(pevent->type);
        _os_WriteRTBuf(pevent->actor);
        for (i=0;i<sizeof(uint32_t);i++)
        {
             _os_WriteRTBuf((pevent->info[i]));
        }

        /* send it */
        SendPlaybackEvent(&_os_rtbuf[0],_os_i_rtbuf);
        
        /* reset variables */
        _os_i_rtbuf = _os_DLLSPE;
        return;
}

static void _os_WriteRTBuf(uint8_t c)
{
        if (c==DLL)
        {
                _os_rtbuf[_os_i_rtbuf++] = DLL;
        }
        _os_rtbuf[_os_i_rtbuf++] = c;
}
#endif



/*************************************************
 * External Interface External Interface
 ************************************************/
#if     (_os_OS_0_POSTMORTEM != _os_OS_NONE)
void _os_LogPlaybackEvent(uint8_t type,
                      uint8_t actor,
                      uint32_t info)
{
#if (_os_OS_0_HARDLOGGING==1)||(RINGBUFSIZE>0)
        _os_PEVENT event;
        _os_COUNTER* pc = &_os_counter_table[SYSTEM_TIMER];

        /* build playback event */
        
        /* system information: time and running priority */
        uint_fast8_t i;
        for (i=0;i<sizeof(uint32_t);i++)
        {
             event.time[i]  = (uint8_t)((pc->value)>>(24-8*i));
        
        }
        event.runprio   = (uint8_t) _os_current_task->PRIORITY;
        /* event information */
        event.type      = type;
        event.actor     = actor; 
        for (i=0;i<sizeof(uint32_t);i++)
        {
             event.info[i]  = (uint8_t)(info>>(24-8*i));
        
        }
        
        #       if      (RINGBUFSIZE>0)
                /* log event in circular buffer */
                _os_PushRing(&event);
        #       else
                /* hard real time logging */
                _os_SendRT(&event);
        #       endif
#else
        (void)type;(void)actor;(void)info;
#endif
}
#endif



#if ( _os_OS_0_POSTMORTEM == _os_OS_REALTIME) && (_os_OS_0_HARDLOGGING==0) 

uint_fast8_t _os_SoftRTEventLog(void)
{
        uint_fast8_t ret = 0;
#if (RINGBUFSIZE>0)
        _os_PEVENT e;
        
        _os_SuspendOS();
        
        if (!_os_EmptyRing())
        {
                e = _os_PopRing();
                ret = 1;
                _os_ResumeOS();
                _os_SendRT(&e);
        }
        else
        {
                _os_ResumeOS();
        }
#endif
        return ret;
}

#endif

