/**************************************************************** 
 *  FILE:    @(#)c_isr.c        1.21 04/10/18
 *  DESCRIPTION:   
 *      Common definitions for ISR handling.
 ****************************************************************/
#include "c_common.h"
#include "c_target.h"


/*******************************************************************
 * Indicates the depth nested ISR1/2 level.                        *
 ******************************************************************/
#if (_os_NO_ISR>0)
volatile uint_fast8_t            _os_FASTALLOC     _os_isr1_counter = 0;
volatile uint_fast8_t            _os_FASTALLOC     _os_isr2_counter = 0;
_os_RTOSMEM _os_ISR*  volatile   _os_FASTALLOC     _os_current_isr
            = (_os_RTOSMEM _os_ISR*)0;
#endif

/********************************************************************************
 * The pair 'Suspend/ResumeAllInterrupts' can be nested.                        *
 * '_os_nesting_sr' stores the current nesting level.                           *
 *******************************************************************************/
volatile int_fast8_t _os_FASTALLOC  _os_nesting_sr = 0;

/********************************************************************************
 * The pair 'Suspend/ResumeOSInterrupts' can be nested.                         *
 * '_os_nesting_sros' stores the current nesting level.                         *
 *******************************************************************************/
volatile int_fast8_t _os_FASTALLOC _os_nesting_sros =0;

/********************************************************************************
 * No need to save/restore the status of the interrupts since they must be      *
 * always enabled before entering RTOS code.                                    *
 *******************************************************************************/
void
#if (_os_SIA>0)
_os_DisableAllInterrupts
#else
DisableAllInterrupts
#endif
(void)
{
        /* check good run-level */
        _os_CheckRunLevel(_os_N_DisableAllInterrupts);

        _os_DisableInterrupts();
}

void
#if (_os_SIA>0)
_os_EnableAllInterrupts
#else
EnableAllInterrupts
#endif
(void)
{
        /* check good run-level */
        _os_CheckRunLevel(_os_N_EnableAllInterrupts);
        
        _os_EnableInterrupts();
}

void
#if (_os_SIA>0)
_os_SuspendAllInterrupts
#else
SuspendAllInterrupts
#endif
(void)
{
        _os_CheckRunLevel(_os_N_SuspendAllInterrupts);
        
        /* only at first nesting level */
        if (_os_nesting_sr==0)
        {
                _os_DisableInterrupts();
        }
        _os_nesting_sr++;
}

void
#if (_os_SIA>0)
_os_ResumeAllInterrupts
#else
ResumeAllInterrupts
#endif
(void)
{
        _os_CheckRunLevel(_os_N_ResumeAllInterrupts);
        
        /* coming back to first nesting level */
        if (--_os_nesting_sr == 0)
        {
                _os_EnableInterrupts();
        }
}

void
#if (_os_SIA>0)
_os_SuspendOSInterrupts
#else
SuspendOSInterrupts
#endif
( void )
{
        /* check good run-level */
        _os_CheckRunLevel(_os_N_SuspendOSInterrupts);

        /* only at first nesting level 'sros' */
        if ( _os_nesting_sros == 0 )
        {
                _os_DisableInterrupts();
                
                /* disable OS Interrupts: TOC generated */
                _os_SuspendOS();
                
                /* only at first nested level 'sr' */
                if (_os_nesting_sr == 0)
                {
                        _os_EnableInterrupts();
                }
        }
        _os_nesting_sros++;
}

void
#if (_os_SIA>0)
_os_ResumeOSInterrupts
#else
ResumeOSInterrupts
#endif
( void )
{
        /* check good run-level */
        _os_CheckRunLevel(_os_N_ResumeOSInterrupts);

        /* only coming back to first nested level */
        if ( --_os_nesting_sros == 0 )
        {
                _os_DisableInterrupts();

                /* enables OS Interrupts: TOC generated */
                _os_ResumeOS();
                
                /* only at first nested level 'sr' */
                if (_os_nesting_sr == 0)
                {
                        _os_EnableInterrupts();
                }
        }
}



#if (_os_NO_ISR >0)
/********************************************************
 * FUNCTION:            _os_ResetISR                    *
 * DESCRIPTION:                                         *
 *      It resets the ISR module.                       *
 *******************************************************/
#if (_os_OS_0_MULTISTART>0)
void _os_ResetISR(void)
{
        uint_fast8_t i;
        _os_RTOSMEM _os_ISR* isr = (_os_RTOSMEM _os_ISR*)&_os_isr_table[0];

        _os_nesting_sros = 0;
        _os_nesting_sr = 0;

        for (i=0;i<_os_NO_ISR;i++,isr++)
        {
               isr->prev = (_os_RTOSMEM _os_ISR*)0;
               #if (_os_OS_0_ORTI>0)
               isr->cycles = 0;
               isr->servicetrace = 0;
               #endif

               #if (MAX_RESOURCE_ISR>0)
               #if ((_os_OS_0_STATUS == _os_OS_EXTENDED)||(_os_OS_0_ORTI>0))
                        /* no owner of any resource */
                         isr->res = _os_INVALID_RES;
               #endif
               #endif
        }
        return;
}
#endif

#endif




